package ru.yandex.wmconsole.servantlet.hostlist;

import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import javax.xml.bind.JAXBException;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;

import ru.yandex.common.framework.core.ServRequest;
import ru.yandex.common.framework.core.ServResponse;
import ru.yandex.common.util.collections.CollectionFactory;
import ru.yandex.common.util.concurrent.CommonThreadFactory;
import ru.yandex.common.util.xml.XmlConvertable;
import ru.yandex.wmconsole.data.info.XmlSearchFastHostInfo;
import ru.yandex.wmconsole.servantlet.WMCAuthenticationServantlet;
import ru.yandex.wmconsole.service.LinksCacheService;
import ru.yandex.wmconsole.service.UrlTreeService;
import ru.yandex.wmconsole.util.XmlUtil;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.InternalProblem;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.error.UserProblem;
import ru.yandex.wmtools.common.util.XmlConvertableCollectionWrapper;
import ru.yandex.wmtools.common.util.concurrent.ParallelMap;

/**
 * Получение информации о сайтах на странице "мои сайты".
 * Предназначен для асинхронной подгрузки информации о сайтах (загружено, в поиске, ТИЦ) на странице "мои сайты".
 *
 * User: azakharov
 * Date: 04.04.12
 * Time: 12:34
 */
public class FastHostInfoServantlet extends WMCAuthenticationServantlet {
    private static final Logger log = LoggerFactory.getLogger(FastHostInfoServantlet.class);

    private static final int MAX_THREAD_NUM = 50;
    private static final String PARAM_DATA = "data";

    private LinksCacheService linksCacheService;
    private UrlTreeService urlTreeService;

    private ExecutorService pool;

    public void init() {
        CommonThreadFactory threadFactory = new CommonThreadFactory(true, FastHostInfoServantlet.class.getSimpleName() + "-");
        pool = Executors.newFixedThreadPool(MAX_THREAD_NUM, threadFactory);
    }

    @Override
    protected void doProcess(final ServRequest req, final ServResponse res, final long userId) throws UserException, InternalException {
        final String data = getRequiredStringParam(req, PARAM_DATA);
        final List<XmlSearchFastHostInfo> hosts = unmarshall(userId, data);
        final List<XmlSearchFastHostInfo> results;

        // параллельно запускается извлечение информации из xml-поиска
        final ParallelMap pm = new ParallelMap(pool);
        try {
            final PullXmlSearchHostData f = new PullXmlSearchHostData(
                    linksCacheService,
                    getHostInfoService(),
                    urlTreeService,
                    getUsersHostsService());
            results = CollectionFactory.list(pm.parMap(hosts, f));
        } catch (ExecutionException e) {
            throw new InternalException(InternalProblem.INTERNAL_PROBLEM,
                    "ExecutionException while pulling xml search data", e);
        } catch (InterruptedException e) {
            throw new InternalException(InternalProblem.INTERNAL_PROBLEM,
                    "InterruptedException while pulling xml search data", e);
        }

        final List<XmlConvertable> wrappers = new LinkedList<XmlConvertable>();
        for (final XmlSearchFastHostInfo hostInfo : results) {
            try {
                if (hostInfo.getException() != null) {
                    log.error("exception in parMap: " + hostInfo.getHostId(), hostInfo.getException());
                }
                wrappers.add(XmlUtil.xmlConvertable(hostInfo));
            } catch (JAXBException e) {
                throw new InternalException(InternalProblem.INTERNAL_PROBLEM, "jaxb marshalling exception", e);
            }
        }
        res.addData(new XmlConvertableCollectionWrapper(wrappers, "hosts"));
    }

    /**
     * Делает проверки, что получилось, то, что ожидали, в противном случае выдает диагностику
     * @param data      строка с json-представлением списка запрашиваемых хостов
     * @return          список объектов с информацией о хостах
     * @throws UserException        выдается при ошибках маршаллинга.
     */
    protected List<XmlSearchFastHostInfo> unmarshall(final long userId, final String data) throws UserException {
        List<XmlSearchFastHostInfo> hosts = new LinkedList<XmlSearchFastHostInfo>();
        try {
            JSONArray array = new JSONArray(data);
            for (int i = 0; i < array.length(); i++) {
                JSONObject hostObject = array.getJSONObject(i);
                long hostId = hostObject.getLong("hostId");
                Long indexCount = getOptionalLongValue(hostObject, "indexCount");
                Long urls = getOptionalLongValue(hostObject, "urls");
                Long tcy = getOptionalLongValue(hostObject, "tcy");
                hosts.add(new XmlSearchFastHostInfo(userId, hostId, indexCount, urls, tcy, null));
            }
        } catch (JSONException e) {
            log.error(e.getMessage(), e);
            throw new UserException(UserProblem.ILLEGAL_PARAM_VALUE, "Parameter parse error: " + data, PARAM_DATA);
        }
        return hosts;
    }

    private Long getOptionalLongValue(JSONObject object, String key) throws JSONException {
        return object.has(key) && !object.isNull(key) ? object.getLong(key) : null;
    }

    @Required
    public void setLinksCacheService(LinksCacheService linksCacheService) {
        this.linksCacheService = linksCacheService;
    }

    @Required
    public void setUrlTreeService(UrlTreeService urlTreeService) {
        this.urlTreeService = urlTreeService;
    }
}
