package ru.yandex.wmconsole.servantlet.hostlist;

import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.common.util.functional.Function;
import ru.yandex.webmaster.common.urltree.YandexSearchShard;
import ru.yandex.wmconsole.data.HostInfoStatusEnum;
import ru.yandex.wmconsole.data.info.BriefHostInfo;
import ru.yandex.wmconsole.data.info.HostStatusInfo;
import ru.yandex.wmconsole.data.info.UsersHostsInfo;
import ru.yandex.wmconsole.data.info.XmlSearchFastHostInfo;
import ru.yandex.wmconsole.data.wrappers.HostInfoWrapper;
import ru.yandex.wmconsole.service.HostInfoService;
import ru.yandex.wmconsole.service.LinksCacheService;
import ru.yandex.wmconsole.service.UrlTreeService;
import ru.yandex.wmconsole.service.UsersHostsService;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.servantlet.AbstractServantlet;

/**
 * Функция получения числа страниц в поиске и ТИЦ из xml-поиска
 *
 * На входе текущие данные (index_count, tcy, urls) с верстки.
 * На выходе обновленные данные из xml-поиска.
 * Side-effect - кэширование полученных значений index_count и tcy
 *
 * User: azakharov
 * Date: 12.04.12
 * Time: 13:28
 */
public class PullXmlSearchHostData  extends Function<XmlSearchFastHostInfo,XmlSearchFastHostInfo> {
    private static final Logger log = LoggerFactory.getLogger(PullXmlSearchHostData.class);

    private final LinksCacheService linksCacheService;
    private final HostInfoService hostInfoService;
    private final UrlTreeService urlTreeService;
    private final UsersHostsService usersHostsService;

    public PullXmlSearchHostData(
            LinksCacheService linksCacheService,
            HostInfoService hostInfoService,
            UrlTreeService urlTreeService,
            UsersHostsService usersHostsService) {
        this.linksCacheService = linksCacheService;
        this.hostInfoService = hostInfoService;
        this.urlTreeService = urlTreeService;
        this.usersHostsService = usersHostsService;
    }

    @Override
    public XmlSearchFastHostInfo apply(@NotNull final XmlSearchFastHostInfo arg) {
        try {
            return doApply(arg);
        } catch (InternalException e) {
            return new XmlSearchFastHostInfo(arg.getHostId(), e);
        } catch (UserException e) {
            return new XmlSearchFastHostInfo(arg.getHostId(), e);
        }
    }

    public XmlSearchFastHostInfo doApply(@NotNull final XmlSearchFastHostInfo hostInfo) throws InternalException, UserException {
        BriefHostInfo briefHostInfo = hostInfoService.getBriefHostInfoByIdOrName(Long.toString(hostInfo.getHostId()));
        if (briefHostInfo == null) {
            // Если в пользовательской базе не нашли хост, то вернем то, что получили
            log.error("host " + hostInfo.getHostId() + " not found in userdb");
            return hostInfo;
        }
        Long indexCount;
        if (hostInfo.getIndexCount() == null) {
            indexCount = urlTreeService.getPreciseIndexCount(briefHostInfo);
            if (indexCount == null) {
                indexCount = linksCacheService.checkCacheAndGetIndexCount(briefHostInfo, YandexSearchShard.RU);
            }
        } else {
            indexCount = hostInfo.getIndexCount();
        }

        final Long longTcy;
        if (hostInfo.getTcy() == null) {
            Integer tcy = linksCacheService.checkCacheAndGetTcy(
                    AbstractServantlet.prepareUrl(briefHostInfo.getName(), true));
            longTcy = (tcy != null) ? tcy.longValue() : null;
        } else {
            longTcy = hostInfo.getTcy();
        }

        // Если загружено меньше, чем в поиске, то загружено = в поиске
        final long safeUrls = hostInfo.getUrls() != null ? hostInfo.getUrls() : 0;
        final Long urls = safeUrls < indexCount ? indexCount : hostInfo.getUrls();

        final HostStatusInfo hsi = hostInfoService.getHostInfoStatus(briefHostInfo);
        final HostInfoStatusEnum status = hsi.getCalculatedHostInfoStatus();
        final UsersHostsInfo verificationInfo = usersHostsService.getUsersHostsInfo(hostInfo.getUserId(), hostInfo.getHostId());
        if (verificationInfo == null || !verificationInfo.getVerificationState().isVerified()) {
            log.warn("attempt to get fasthostinfo for not verified host " + briefHostInfo.getName());
        }
        final Boolean needShowIndexInfo = HostInfoWrapper.needShowIndexInfo(
                (verificationInfo != null) ? verificationInfo.getVerificationState() : null,
                null,
                indexCount,
                hostInfo.getUrls(),
                status);
        return new XmlSearchFastHostInfo(
                hostInfo.getUserId(), hostInfo.getHostId(), indexCount, urls, longTcy, needShowIndexInfo);
    }
}
