package ru.yandex.wmconsole.servantlet.mainmirror;

import org.springframework.beans.factory.annotation.Required;
import ru.yandex.common.framework.core.ServRequest;
import ru.yandex.common.framework.core.ServResponse;
import ru.yandex.common.util.collections.Cu;
import ru.yandex.common.util.functional.Filter;
import ru.yandex.webmaster.common.host.dao.TblHostsMainDao;
import ru.yandex.wmconsole.data.info.BriefHostInfo;
import ru.yandex.wmconsole.data.info.UsersHostsInfo;
import ru.yandex.wmconsole.data.mirror.MirrorGroupChangeRequest;
import ru.yandex.wmconsole.data.wrappers.BriefHostInfoWrapper;
import ru.yandex.wmconsole.servantlet.WMCAuthorizedHostOperationServantlet;
import ru.yandex.wmconsole.service.MirrorGroupsChangeService;
import ru.yandex.wmconsole.service.UsersHostsService;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.util.XmlConvertableCollectionWrapper;
import ru.yandex.wmtools.common.util.XmlDataWrapper;

import java.util.List;
import java.util.Set;

/**
 * Provides information for Indexing options -> Main mirror
 *
 * User: azakharov
 * Date: 19.03.14
 * Time: 14:51
 */
public class MainMirrorInfoServanlet extends WMCAuthorizedHostOperationServantlet {

    private TblHostsMainDao tblHostsMainDao;
    private MirrorGroupsChangeService mirrorGroupsChangeService;
    private UsersHostsService usersHostsService;

    @Override
    protected void doProcess(final ServRequest req, final ServResponse res, final long userId) throws UserException, InternalException {
        BriefHostInfo hostInfo = getHostInfoAndVerify(req, userId);

        // get main mirror
        final BriefHostInfo mainMirrorInfo;
        if (hostInfo.getMainMirrorId() != null) {
            mainMirrorInfo = tblHostsMainDao.getBriefHostInfoByHostId(hostInfo.getMainMirrorId());
        } else {
            mainMirrorInfo = hostInfo;
        }

        // check main mirror is verified
        final boolean mainMirrorVerified;
        if (mainMirrorInfo != hostInfo) {
            mainMirrorVerified = isHostOwnedByUser(mainMirrorInfo, userId);
        } else {
            mainMirrorVerified = true;
        }

        // get eligible not main mirrors
        final List<BriefHostInfo> verifiedNotMainMirrors = usersHostsService.getVerifiedMirrorsForHostByUser(
                mainMirrorInfo, userId);
        final String simplifiedHostName = mirrorGroupsChangeService.getSimplifiedHostName(mainMirrorInfo);
        final Set<String> similarHosts = mirrorGroupsChangeService.getSimilarHostNames(simplifiedHostName);
        final Filter<BriefHostInfo> verifiedNotWwwCloneHttpsCloneFilter = new Filter<BriefHostInfo>() {
            @Override
            public boolean fits(BriefHostInfo hostInfo) {
                return !similarHosts.contains(hostInfo.getName().toLowerCase());
            }
        };
        List<BriefHostInfo> eligableNotMainMirrors = Cu.filterList(verifiedNotMainMirrors, verifiedNotWwwCloneHttpsCloneFilter);

        // get existing main mirror requests
        List<MirrorGroupChangeRequest> requests = mirrorGroupsChangeService.getMainMirrorChangeRequests(hostInfo);

        boolean mayUnstick = hostInfo.getMainMirrorId() != null;

        res.addData(
                new MirrorInfoWrapper(
                        hostInfo, mainMirrorInfo, mayUnstick, mainMirrorVerified, simplifiedHostName, eligableNotMainMirrors, requests));
    }

    private boolean isHostOwnedByUser(final BriefHostInfo hostInfo, final long userId) throws InternalException {
        final UsersHostsInfo usersHostsInfo = usersHostsService.getUsersHostsInfo(userId, hostInfo.getId());
        return usersHostsInfo != null &&
                    usersHostsInfo.getVerificationState() != null &&
                    usersHostsInfo.getVerificationState().isVerified();
    }

    private static class MirrorInfoWrapper extends XmlDataWrapper<BriefHostInfo> {

        private final BriefHostInfo mainMirrorInfo;
        private final boolean mayUnstick;
        private final boolean mayChooseMainMirror;
        private final String simlifiedHostName;
        private final List<BriefHostInfo> notMainMirrorHosts;
        private final List<MirrorGroupChangeRequest> requests;

        private MirrorInfoWrapper(BriefHostInfo data, BriefHostInfo mainMirrorInfo, boolean mayUnstick, boolean mayChooseMainMirror, String simlifiedHostName, List<BriefHostInfo> notMainMirrorHosts, List<MirrorGroupChangeRequest> requests) {
            super(data, "host", "id", Long.toString(data.getId()));
            this.mainMirrorInfo = mainMirrorInfo;
            this.mayUnstick = mayUnstick;
            this.mayChooseMainMirror = mayChooseMainMirror;
            this.simlifiedHostName = simlifiedHostName;
            this.notMainMirrorHosts = notMainMirrorHosts;
            this.requests = requests;
        }

        @Override
        protected void doToXml(StringBuilder result) {
            if (mainMirrorInfo != null) {
                putTag(result, "punycode-main-name", mainMirrorInfo.getName());
            }
            putTag(result, "punycode-name", data.getName());
            putTag(result, "punycode-main-simple-name", simlifiedHostName);
            putBooleanTag(result, "may-unstick", mayUnstick);
            putBooleanTag(result, "may-select-main", mayChooseMainMirror);
            XmlConvertableCollectionWrapper.wrap(notMainMirrorHosts, BriefHostInfoWrapper.class, "not-main-mirrors").toXml(result);
            if (!requests.isEmpty()) {
                new MirrorGroupChangeRequestWrapper(requests.get(0)).toXml(result);
            }
        }
    }

    public static class MirrorGroupChangeRequestWrapper extends XmlDataWrapper<MirrorGroupChangeRequest> {
        private static final String TAG_MAIN_MIRROR_REQUEST = "main-mirror-request";
        private static final String TAG_STATE = "state";
        private static final String TAG_ACTION = "action";
        private static final String TAG_DESIRED_MAIN_MIRROR = "desired-main-mirror";
        private static final String TAG_CREATED_AT = "created-at";
        private static final String TAG_MODIFIED_AT = "modified-at";
        private static final String TAG_NOTIFIED = "notified";
        private static final String TAG_USER_ID = "user-id";
        private static final String TAG_MIRRORING_RESPONSE = "mirroring-response";
        private static final String TAG_ACTION_STATUS = "status";
        private static final String TAG_HOST_RESULT = "host-result";
        private static final String TAG_HOST_NAME = "host-name";
        private static final String TAG_HOST_STATUS = "host-status";
        private static final String TAG_OLD_MAIN = "old-main-mirror";
        private static final String TAG_LAST_VALID_CHOICE = "last-valid-choice";

        public MirrorGroupChangeRequestWrapper(MirrorGroupChangeRequest data) {
            super(data, TAG_MAIN_MIRROR_REQUEST);
        }

        @Override
        protected void doToXml(StringBuilder result) {
            putTag(result, TAG_STATE, data.getState().name());
            putTag(result, TAG_ACTION, data.getAction().name());
            putTag(result, TAG_DESIRED_MAIN_MIRROR, data.getDesiredMain());
            putTag(result, TAG_OLD_MAIN, data.getOldMainMirrorName());
            putExtendedDateTimeTag(result, TAG_CREATED_AT, data.getCreateDate());
            putExtendedDateTimeTag(result, TAG_MODIFIED_AT, data.getModificationDate());
            putBooleanTag(result, TAG_NOTIFIED, data.isNotified());
            putLongTag(result, TAG_USER_ID, data.getUserId());
            putTag(result, TAG_LAST_VALID_CHOICE, data.getLastSuccessfulMainMirrorName());
            if (data.getActionInfo().getActionStatus() != null &&
                    data.getActionInfo().getHostName1() != null &&
                    data.getActionInfo().getHostStatus1() != null &&
                    data.getActionInfo().getHostName2() != null &&
                    data.getActionInfo().getHostStatus2() != null) {

                final String tagMirroringResponse = TAG_MIRRORING_RESPONSE;
                putOpenTag(result, tagMirroringResponse);
                try {
                    putTag(result, TAG_ACTION_STATUS, data.getActionInfo().getActionStatus().toString());
                    final String tagHostResult = TAG_HOST_RESULT;
                    putOpenTag(result, tagHostResult);
                    try {
                        putTag(result, TAG_HOST_NAME, data.getActionInfo().getHostName1());
                        putTag(result, TAG_HOST_STATUS, data.getActionInfo().getHostStatus1().toString());
                    } finally {
                        putCloseTag(result, tagHostResult);
                    }
                    putOpenTag(result, tagHostResult);
                    try {
                        putTag(result, TAG_HOST_NAME, data.getActionInfo().getHostName2());
                        putTag(result, TAG_HOST_STATUS, data.getActionInfo().getHostStatus2().toString());
                    } finally {
                        putCloseTag(result, tagHostResult);
                    }
                } finally {
                    putCloseTag(result, tagMirroringResponse);
                }
            }
        }
    }

    @Required
    public void setTblHostsMainDao(TblHostsMainDao tblHostsMainDao) {
        this.tblHostsMainDao = tblHostsMainDao;
    }

    @Required
    public void setMirrorGroupsChangeService(MirrorGroupsChangeService mirrorGroupsChangeService) {
        this.mirrorGroupsChangeService = mirrorGroupsChangeService;
    }

    @Required
    public void setUsersHostsService(UsersHostsService usersHostsService) {
        this.usersHostsService = usersHostsService;
    }
}
