package ru.yandex.wmconsole.servantlet.poll;

import java.util.List;

import javax.xml.bind.JAXBException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;

import ru.yandex.common.framework.core.ServRequest;
import ru.yandex.common.framework.core.ServResponse;
import ru.yandex.wmconsole.data.poll.PollInfo;
import ru.yandex.wmconsole.data.poll.PollStatusEnum;
import ru.yandex.wmconsole.data.poll.QuestionInfo;
import ru.yandex.wmconsole.servantlet.WMCAuthenticationServantlet;
import ru.yandex.wmconsole.service.PollService;
import ru.yandex.wmconsole.service.error.WMCUserProblem;
import ru.yandex.wmconsole.util.XmlUtil;
import ru.yandex.wmtools.common.data.wrappers.TagWrapper;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.InternalProblem;
import ru.yandex.wmtools.common.error.UserException;

/**
 * Генерация формы опроса
 *
 * User: azakharov
 * Date: 14.03.12
 * Time: 14:23
 */
public class PollServantlet extends WMCAuthenticationServantlet {
    private static final Logger log = LoggerFactory.getLogger(PollServantlet.class);

    private static final String PARAM_POLL = "poll_id";
    private static final String TAG_ALREADY_ANSWERED = "POLL_ALREADY_ANSWERED";

    private PollService pollService;

    @Override
    protected void doProcess(ServRequest req, ServResponse res, long userId) throws UserException, InternalException {
        // проверяем, что пользователь существует
        assertUserExists(userId);
        // проверяем, что у него есть подтвержденные хосты
        if (getUsersHostsService().getAllHostsVerifiedByUser(userId).isEmpty()) {
            throw new UserException(WMCUserProblem.NO_VERIFIED_HOSTS, "User doesn't have verified hosts");
        }

        final long pollId = getRequiredLongParam(req, PARAM_POLL);
        final PollInfo pollInfo = pollService.getPoll(pollId);
        if (pollInfo == null) {
            throw new UserException(WMCUserProblem.POLL_NOT_FOUND, "No such poll");
        }
        // Проверяем, что опрос выполняется
        if (!PollStatusEnum.RUNNING.equals(pollInfo.getStatus())) {
            throw new UserException(WMCUserProblem.POLL_STATUS_ERROR, "Poll can not be used in status " + pollInfo.getStatus());
        }
        // проверяем, что пользователь еще не отвечал на данный опрос
        if (pollService.isAnswered(userId, pollInfo.getId())) {
            res.addData(new TagWrapper(TAG_ALREADY_ANSWERED, ""));
        }

        final List<QuestionInfo> questions = pollService.getQuestions(pollInfo);
        pollInfo.setQuestions(questions);

        try {
            res.setContentType("text/xml; charset=UTF-8");
            res.addData(XmlUtil.xmlConvertable(pollInfo));
        } catch (JAXBException e) {
            throw new InternalException(InternalProblem.INTERNAL_PROBLEM, "jaxb xml marshalling error");
        }
    }

    @Required
    public void setPollService(PollService pollService) {
        this.pollService = pollService;
    }
}
