package ru.yandex.wmconsole.servantlet.regions;

import java.net.MalformedURLException;
import java.net.URL;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;

import ru.yandex.common.framework.core.ServRequest;
import ru.yandex.common.framework.core.ServResponse;
import ru.yandex.webmaster.viewer.service.NewWebmasterProxyService;
import ru.yandex.wmconsole.data.info.BriefHostInfo;
import ru.yandex.wmconsole.data.info.HostDbHostInfo;
import ru.yandex.wmconsole.servantlet.WMCAuthorizedHostOperationServantlet;
import ru.yandex.wmconsole.service.HostDbHostInfoService;
import ru.yandex.wmconsole.service.HostRegionService;
import ru.yandex.wmconsole.service.error.WMCUserProblem;
import ru.yandex.wmtools.common.error.ExtraTagInfo;
import ru.yandex.wmtools.common.error.ExtraTagNameEnum;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.InternalProblem;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.error.UserProblem;
import ru.yandex.wmtools.common.sita.DocumentFormatEnum;
import ru.yandex.wmtools.common.sita.SitaException;
import ru.yandex.wmtools.common.sita.SitaRedirectService;
import ru.yandex.wmtools.common.sita.SitaRequestTimeout;
import ru.yandex.wmtools.common.sita.SitaUrlFetchRequest;
import ru.yandex.wmtools.common.sita.SitaUrlFetchRequestBuilder;
import ru.yandex.wmtools.common.sita.SitaUrlFetchResponse;
import ru.yandex.wmtools.common.sita.UserAgentEnum;
import ru.yandex.wmtools.common.util.http.YandexHttpStatus;

/**
 * @author baton
 */
public class HostRegionModifyServantlet extends WMCAuthorizedHostOperationServantlet {
    private static final Logger log = LoggerFactory.getLogger(HostRegionModifyServantlet.class);

    private static final String PARAM_REGION = "region";
    private static final String PARAM_URL = "url";

    private HostDbHostInfoService hostDbHostInfoService;
    private HostRegionService hostRegionService;
    private SitaRedirectService sitaNoCheckRedirectService;
    private NewWebmasterProxyService newWebmasterProxyService;

    @Override
    protected void doProcess(ServRequest req, ServResponse res, long userId)
            throws InternalException, UserException {

        BriefHostInfo hostInfo = getHostInfoAndVerify(req, userId);
        if (hostInfo == null) {
            return;
        }
        checkIndexed(hostInfo);

        HostDbHostInfo hostDbHostInfo = hostDbHostInfoService.getHostDbHostInfo(hostInfo.getName());

        int region = getRequiredIntParam(req, PARAM_REGION);

        URL url;
        if (region != HostRegionService.UNDEFINED_REGION) {
            String urlParam = getRequiredStringParam(req, PARAM_URL);
            url = prepareUrl(urlParam, true);
        } else {
            url = null;
        }
        newWebmasterProxyService.changeHostRegion(userId, hostInfo, region, url);
    }

    public URL assertUrlCorrespondsHost(BriefHostInfo hostInfo, String urlParam, boolean omitWwwPrefix)
            throws UserException, InternalException {
        URL url = prepareUrl(urlParam, true);
        URL hostUrl = prepareHostname(hostInfo.getName(), true);

        if (omitWwwPrefix) {
            try {
                url = new URL(url.getProtocol(), bringToStandardWwwPrefix(url.getHost(), hostUrl.getHost()), url.getPort(), url.getFile());
            } catch (MalformedURLException e) {
                throw new UserException(UserProblem.INVALID_URL, "Invalid url: " + urlParam, e);
            }
        }

        if (!url.getAuthority().equalsIgnoreCase(hostUrl.getAuthority())) {
            throw new UserException(UserProblem.WRONG_HOSTNAME_USED, "URL must be from host: " + hostUrl.getAuthority(),
                    new ExtraTagInfo(ExtraTagNameEnum.EXPECTED, hostUrl.getAuthority()), new ExtraTagInfo(ExtraTagNameEnum.GIVEN, url.getAuthority()));
        }

        SitaUrlFetchRequest sitaUrlFetchRequest = new SitaUrlFetchRequestBuilder(url)
                .setDocumentFormat(DocumentFormatEnum.DF_HTTP_RESPONSE)
                .setUserAgent(UserAgentEnum.WEBMASTER)
                .setCheckAllowedInRobotsTxt(false)
                .setRequestTimeout(SitaRequestTimeout._15_SECONDS)
                .createSitaUrlFetchRequest();

        SitaUrlFetchResponse sitaUrlFetchResponse;
        try {
            sitaUrlFetchResponse = sitaNoCheckRedirectService.followRedirects(sitaUrlFetchRequest);
        } catch (InternalException e) {
            log.error("Unable to check host", e);
            throw new InternalException(InternalProblem.PROCESSING_ERROR, "Unable to check host",
                    new ExtraTagInfo(ExtraTagNameEnum.URL, urlParam));
        } catch (SitaException e) {
            log.error("Unable to check host", e);
            throw new InternalException(InternalProblem.PROCESSING_ERROR, "Unable to check host",
                    new ExtraTagInfo(ExtraTagNameEnum.URL, urlParam));
        }

        if (sitaUrlFetchResponse.isAllowedInRobotsTxt() != null && !sitaUrlFetchResponse.isAllowedInRobotsTxt()) {
            throw new UserException(WMCUserProblem.URL_IS_DISALLOWED_IN_ROBOTS_TXT, "");
        } else if (!YandexHttpStatus.isStandardHttpCode(sitaUrlFetchResponse.getSitaHttpStatus())) {
            log.error("Wrong status: " + sitaUrlFetchResponse.getSitaHttpStatus());
            throw new InternalException(InternalProblem.PROCESSING_ERROR, "Unable to check host",
                    new ExtraTagInfo(ExtraTagNameEnum.URL, urlParam));
        }

        if (sitaUrlFetchResponse.getSitaHttpStatus() != YandexHttpStatus.HTTP_200_OK) {
            log.error("Wrong status: " + sitaUrlFetchResponse.getSitaHttpStatus());
            throw new UserException(WMCUserProblem.HOST_NOT_ACCESSIBLE, "Host " + urlParam + " is not accessible.",
                    new ExtraTagInfo(ExtraTagNameEnum.URL, urlParam));
        }

        return url;
    }

    @Required
    public void setHostRegionService(HostRegionService hostRegionService) {
        this.hostRegionService = hostRegionService;
    }

    @Required
    public void setHostDbHostInfoService(HostDbHostInfoService hostDbHostInfoService) {
        this.hostDbHostInfoService = hostDbHostInfoService;
    }

    @Required
    public void setSitaNoCheckRedirectService(SitaRedirectService sitaNoCheckRedirectService) {
        this.sitaNoCheckRedirectService = sitaNoCheckRedirectService;
    }

    @Required
    public void setNewWebmasterProxyService(NewWebmasterProxyService newWebmasterProxyService) {
        this.newWebmasterProxyService = newWebmasterProxyService;
    }
}
