package ru.yandex.wmconsole.servantlet.robotstxt;

import java.net.URL;
import java.util.Arrays;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;

import ru.yandex.common.framework.core.ServRequest;
import ru.yandex.common.framework.core.ServResponse;
import ru.yandex.wmconsole.data.info.AnalysisResult;
import ru.yandex.wmconsole.data.info.ErrorInfo;
import ru.yandex.wmconsole.data.info.FormatErrorType;
import ru.yandex.wmconsole.data.info.UrlAllowInfo;
import ru.yandex.wmconsole.data.wrappers.StringWrapper;
import ru.yandex.wmconsole.service.DispatcherHttpService;
import ru.yandex.wmconsole.service.RobotsTxtService;
import ru.yandex.wmconsole.service.error.WMCUserProblem;
import ru.yandex.wmconsole.util.XmlUtil;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.servantlet.AbstractServantlet;

/**
 * Loads robots.txt file from a given host or analyzes it.
 * No authentication and host verification are required.
 *
 * @author ailyin
 */
public class PublicRobotsTxtServantlet extends AbstractServantlet {
    private static final Logger log = LoggerFactory.getLogger(PublicRobotsTxtServantlet.class);

    private static final String PARAM_HOSTNAME = "hostname";

    private RobotsTxtService robotsTxtService;
    private DispatcherHttpService dispatcherHttpService;

    @Override
    protected void doProcess(ServRequest req, ServResponse res) throws UserException, InternalException {
        log.debug("PublicRobotsTxt: started");

        String hostname = req.getParam(PARAM_HOSTNAME, true);
        Boolean loadOnlyParam = getBooleanParam(req, RobotsTxtHelper.PARAM_ONLY_LOAD);
        @SuppressWarnings({"PointlessBooleanExpression"})
        boolean loadOnly = (loadOnlyParam != null) && (loadOnlyParam == true);

        String robotsTxtContent = getAndOutputRobotsTxtContent(req, res, hostname, loadOnly);
        if (!loadOnly && (robotsTxtContent != null)) {
            URL hostnameUrl = StringUtils.isEmpty(hostname) ? null : prepareHostname(hostname, true);
            outputAnalysisResult(req, hostnameUrl, robotsTxtContent, res);
        }

        log.debug("PublicRobotsTxt: Finished");
    }

    private String getAndOutputRobotsTxtContent(ServRequest req, ServResponse res, String hostname, boolean loadOnly)
            throws UserException, InternalException
    {
        String robotsTxtContent = req.getParam(RobotsTxtHelper.PARAM_ROBOTSTXT, true);

        try {
            // robotsTxtContent==null only when the page is opened for the first time
            if (StringUtils.isEmpty(robotsTxtContent) || loadOnly) { // need to download
                if (StringUtils.isEmpty(hostname)) {
                    throw new UserException(WMCUserProblem.NO_HOST_OR_ROBOTSTXT,
                            "neither hostname nor robots.txt have been specified");
                }
                robotsTxtContent = robotsTxtService.getRobotsTxtContent(prepareHostname(hostname, true));
            }
        } finally {
            if (robotsTxtContent != null) {
                res.addData(new StringWrapper(XmlUtil.stripInvalidXMLCharacters(robotsTxtContent), "robots-txt"));
            }
        }
        return robotsTxtContent;
    }

    private void outputAnalysisResult(ServRequest req, URL hostname, String robotsTxtContent, ServResponse res)
            throws InternalException, UserException {
        String[] urls = splitInput(req.getParam(RobotsTxtHelper.PARAM_URLS));
        if (urls.length > RobotsTxtHelper.MAX_URLS_COUNT) {
            throw new UserException(WMCUserProblem.TOO_MANY_URLS, "too many urls");
        }

        //noinspection RedundantArrayCreation
        List<UrlAllowInfo> allowInfos = Arrays.asList(new UrlAllowInfo[urls.length]);
        List<String> validUrls = RobotsTxtHelper.getValidUrls(hostname, urls, allowInfos);
        AnalysisResult result = dispatcherHttpService.analyzeRobotsTxt(robotsTxtContent,
                validUrls.toArray(new String[validUrls.size()]));

        String[] robotsTxtLines = RobotsTxtHelper.splitRobotsTxt(robotsTxtContent);
        if (robotsTxtContent.length() > RobotsTxtHelper.MAX_ROBOTS_TXT_SIZE) {
            int failLine = RobotsTxtHelper.countOccurences(robotsTxtContent.substring(0, RobotsTxtHelper.MAX_ROBOTS_TXT_SIZE), '\n') + 1;
            res.addData(RobotsTxtHelper.getErrorsData(
                    RobotsTxtHelper.addToRightPosition(result.getErrors(), new ErrorInfo(FormatErrorType.ERR_ROBOTS_HUGE, failLine)),
                    robotsTxtLines));
        } else {
            res.addData(RobotsTxtHelper.getErrorsData(result.getErrors(), robotsTxtLines));
        }
        res.addData(RobotsTxtHelper.getAcceptedLinesData(result.getAcceptedLines(), robotsTxtLines));
        res.addData(RobotsTxtHelper.getAllowData(result.getAreAllowed(), allowInfos, urls));
    }

    @Required
    public void setDispatcherHttpService(DispatcherHttpService dispatcherHttpService) {
        this.dispatcherHttpService = dispatcherHttpService;
    }

    @Required
    public void setRobotsTxtService(RobotsTxtService robotsTxtService) {
        this.robotsTxtService = robotsTxtService;
    }
}
