package ru.yandex.wmconsole.servantlet.serplinks;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;
import ru.yandex.common.framework.core.ServRequest;
import ru.yandex.common.framework.core.ServResponse;
import ru.yandex.wmconsole.data.info.BriefHostInfo;
import ru.yandex.wmconsole.data.info.SerpLinkInfo;
import ru.yandex.wmconsole.data.info.XMLSerpLinkInfo;
import ru.yandex.wmconsole.servantlet.WMCAuthenticationServantlet;
import ru.yandex.wmconsole.service.SerpLinksService;
import ru.yandex.wmtools.common.SupportedProtocols;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.error.UserProblem;

import java.io.UnsupportedEncodingException;
import java.net.*;
import java.util.Map;

/**
 * @author Andrey Mima (amima@yandex-team.ru)
 */
public class SerpLinksChangeServantlet extends WMCAuthenticationServantlet {
    private static final Logger log = LoggerFactory.getLogger(SerpLinksChangeServantlet.class);

    private static final String PARAM_REMOVE = "remove";
    private static final String PARAM_PAGE = "page";
    private static final String PARAM_SORT = "sort";
    private static final String VALUE_WEIGHT = "weight";
    private static final String VALUE_ALPHA = "alpha";

    private static final String PREFIX_URL = "url_";
    private static final String PREFIX_SHOW = "show_";
    private static final String PREFIX_PAGEURL = "pageurl_";

    private static final String PARAM_SECTION = "section";

    private SerpLinksService serpLinksService;

    @Override
    protected void doProcess(ServRequest req, ServResponse res, long userId) throws InternalException, UserException {

        BriefHostInfo hostInfo = getHostInfoAndVerify(req, userId);

        String path = getStringParam(req, PARAM_SECTION);
        String sectionUrl = XMLSerpLinkInfo.getSectionName(hostInfo.getName(), path);

        parseUrlParameters(req, sectionUrl);

        if ((req.getParam(PARAM_REMOVE, true) != null)) {
            boolean remove = req.getBoolean(PARAM_REMOVE);
            String blockUrlString = req.getParam(PARAM_PAGE);
            log.debug(blockUrlString);
            try {
                SupportedProtocols.getURL(blockUrlString);
            } catch (Exception e) {
                throw new UserException(UserProblem.ILLEGAL_PARAM_VALUE, "Sort type not supported", PARAM_SORT, blockUrlString);
            }

            if (remove) {
                serpLinksService.remove(sectionUrl, blockUrlString);
            } else {
                serpLinksService.unremove(sectionUrl, blockUrlString);
            }
        } else if (req.getParam(PARAM_SORT, true) != null) {
            String sort = req.getParam(PARAM_SORT, true);
            if (VALUE_WEIGHT.equals(sort)) {
                serpLinksService.sort(sectionUrl, SerpLinksService.Sort.WEIGHT);
            } else if (VALUE_ALPHA.equals(sort)) {
                serpLinksService.sort(sectionUrl, SerpLinksService.Sort.ALPHA);
            } else {
                throw new UserException(UserProblem.ILLEGAL_PARAM_VALUE, "Sort type not supported", PARAM_SORT, sort);
            }
        } else {
            throw new UserException(UserProblem.INVALID_PARAM, "No action request param");
        }
    }

    private void parseUrlParameters(ServRequest req, String hostName) throws InternalException {
        //noinspection unchecked
        Map<String, String> params = req.getParams();
        SerpLinkInfo serpLinkInfo = serpLinksService.getSerpLinkInfo(hostName);

        for (Map.Entry<String, String> param : params.entrySet()) {
            String paramName = param.getKey();

            if (paramName.startsWith(PREFIX_URL)) {
                final String position = paramName.substring(PREFIX_URL.length());

                final String url = req.getParam(PREFIX_PAGEURL + position);
                final String name = req.getParam(PREFIX_URL + position);
                final Boolean show = req.getBoolean(PREFIX_SHOW + position);

                boolean found = false;

                for (SerpLinkInfo.SerpLinksPage page : serpLinkInfo.getPages()) {
                    for (SerpLinkInfo.SerpLinksPageName pageName : page.getPageNames()) {
                        if (pageName.getName().equals(name)) {
                            found = true;
                            break;
                        }
                    }
                }

                if (found) {
                    serpLinksService.rename(hostName, toPunycode(url), name);
                    if (show) {
                        serpLinksService.remove(hostName, toPunycode(url));
                    } else {
                        serpLinksService.unremove(hostName, toPunycode(url));
                    }
                }
            }
        }
    }

    /**
     * Преобразование кириллических доменов в url в punycode
     *
     * @param urlString url-encoded строковое представление url
     * @return url-encoded строковое представление url с хостом, закодированным в punycode
     */
    private String toPunycode(String urlString) {
        // Раскодируем
        String decoded;
        try {
            decoded = URLDecoder.decode(urlString, "UTF-8");
            URL originalUrl = SupportedProtocols.getURL(decoded);
            final String originalHost = originalUrl.getHost();
            // Строим URL
            final URL url = prepareUrl(decoded, true);
            // Делаем так, потому что encode(decode(url)) даёт результат, отличный от url в части path
            final String replaced = urlString.replace(originalHost, url.getHost());
            return URLEncoder.encode(replaced, "UTF-8");
        } catch (UnsupportedEncodingException e) {
            log.error("Unknown encoding ", e);
            // Если не получилось - возвращаем входную строку
            return urlString;
        } catch (UserException e) {
            // Если не получилось - возвращаем входную строку
            log.error("User exception ", e);
            return urlString;
        } catch (MalformedURLException e) {
            // Если не получилось - возвращаем входную строку
            log.error("MalformedURLException ", e);
            return urlString;
        } catch (SupportedProtocols.UnsupportedProtocolException e) {
            // Если не получилось - возвращаем входную строку
            log.error("UnsupportedProtocolException ", e);
            return urlString;
        } catch (URISyntaxException e) {
            // Если не получилось - возвращаем входную строку
            log.error("URISyntaxException ", e);
            return urlString;
        }
    }

    @Required
    public void setSerpLinksService(SerpLinksService serpLinksService) {
        this.serpLinksService = serpLinksService;
    }
}
