package ru.yandex.wmconsole.servantlet.sitemaps;

import java.io.IOException;

import javax.xml.bind.JAXBException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;

import ru.yandex.common.framework.core.ServRequest;
import ru.yandex.common.framework.core.ServResponse;
import ru.yandex.wmconsole.data.SitemapAnalysisResult;
import ru.yandex.wmconsole.service.DispatcherHttpService;
import ru.yandex.wmconsole.service.error.WMCUserProblem;
import ru.yandex.wmconsole.util.XmlUtil;
import ru.yandex.wmconsole.viewer.sitemap.SitemapService2;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.InternalProblem;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.servantlet.AbstractServantlet;

/**
 * Сервантлет для анализа файлов sitemap
 *
 * User: azakharov
 * Date: 22.02.12
 * Time: 13:43
 */
public class AnalyzeSitemapServantlet extends AbstractServantlet {
    private static final Logger log = LoggerFactory.getLogger(AnalyzeSitemapServantlet.class);

    private static final String PARAM_SITEMAP = "sitemap";
    private static final String PARAM_URL = "url";

    private DispatcherHttpService dispatcherHttpService;
    private SitemapService2 sitemapService2;

    @Override
    protected void doProcess(ServRequest req, ServResponse res) throws UserException, InternalException {
        log.debug("AnalyzeSitemap: Started");
        final SitemapRequestBuilder builder = new SitemapRequestBuilder();
        final SitemapDataProvider dataProvider = createDataProvider(req);
        dataProvider.provideData(builder);

        final SitemapAnalysisResult result = dispatcherHttpService.analyzeSitemapBytes(builder.getBytesToSend());
        if (builder.isRedirected()) {
            // Добавляем предупреждение о редиректе
            result.addWrnRedirected();
        }
        try {
            res.addData(XmlUtil.xmlConvertable(result));
        } catch (JAXBException e) {
            log.error("Error in marshaling: ", e);
            throw new InternalException(InternalProblem.INTERNAL_PROBLEM, "error in marshaling", e);
        }
        log.debug("AnalyzeSitemap: finished");
    }

    /**
     * Поставщик данных по тексту файла sitemap, переданному через параметр
     */
    public static class SitemapParameterDataProvider implements SitemapDataProvider {
        private String sitemapText;

        public SitemapParameterDataProvider(String paramName) {
            this.sitemapText = paramName;
        }

        @Override
        public void provideData(SitemapRequestBuilder builder) throws UserException{
            try {
                builder.getStream().write(sitemapText.getBytes());
            } catch (IOException e) {
                log.error("Unable to load sitemap", e);
                throw new UserException(WMCUserProblem.SITEMAP_ACCESS_ERROR, "Failed to load sitemap from parameter", e);
            }
        }
    }

    private SitemapDataProvider createDataProvider(ServRequest request) {
        final String sitemapParam = getPreciseParam(request, PARAM_SITEMAP);
        final String urlParam = getStringParam(request, PARAM_URL);

        log.debug("params: sitemap=" + sitemapParam + " url=" + urlParam);
        if (sitemapParam != null) {
            return new SitemapParameterDataProvider(sitemapParam);
        } else if (urlParam != null) {
            return new SitemapUrlDataProvider(urlParam, sitemapService2);
        } else {
            return new SitemapRemoteFileDataProvider(request);
        }
    }

    @Required
    public void setDispatcherHttpService(DispatcherHttpService dispatcherHttpService) {
        this.dispatcherHttpService = dispatcherHttpService;
    }

    @Required
    public void setSitemapService2(SitemapService2 sitemapService2) {
        this.sitemapService2 = sitemapService2;
    }
}
