package ru.yandex.wmconsole.servantlet.sitemaps;

import java.util.List;

import javax.xml.bind.JAXBException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;

import ru.yandex.common.framework.core.ServRequest;
import ru.yandex.common.framework.core.ServResponse;
import ru.yandex.common.framework.pager.Pager;
import ru.yandex.common.util.db.OrderByClause;
import ru.yandex.wmconsole.data.info.BriefHostInfo;
import ru.yandex.wmconsole.data.info.HostDbHostInfo;
import ru.yandex.wmconsole.data.info.MergedSitemapInfo;
import ru.yandex.wmconsole.data.info.SitemapErrorInfo;
import ru.yandex.wmconsole.data.info.SitemapNavigationInfo;
import ru.yandex.wmconsole.data.sitemaps.SitemapListCorrespondence;
import ru.yandex.wmconsole.data.sitemaps.SitemapSourceEnum;
import ru.yandex.wmconsole.data.wrappers.MergedSitemapInfoWrapper;
import ru.yandex.wmconsole.data.wrappers.SitemapErrorInfoWrapper;
import ru.yandex.webmaster.viewer.sitemap.CurrentSitemapService;
import ru.yandex.wmconsole.servantlet.WMCAuthorizedHostOperationServantlet;
import ru.yandex.wmconsole.service.HostDbHostInfoService;
import ru.yandex.webmaster.common.sitemap.SitemapService;
import ru.yandex.wmconsole.util.XmlUtil;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.InternalProblem;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.error.UserProblem;
import ru.yandex.wmtools.common.util.XmlConvertableCollectionWrapper;

/**
 * Сервантлет для выдачи ошибок разбора файлов sitemap и sitemapindex для последней загруженной версии sitemap
 */
public class SitemapParseErrorsServantlet extends WMCAuthorizedHostOperationServantlet {
    private static final Logger log = LoggerFactory.getLogger(SitemapParseErrorsServantlet.class);

    private static final String PARAM_SITEMAP = "sitemap";
    private static final String PARAM_SOURCE = "source";
    private static final String PARAM_INDEX = "index";

    private CurrentSitemapService currentSitemapService;
    private HostDbHostInfoService hostDbHostInfoService;

    @Override
    protected void doProcess(ServRequest req, ServResponse res, long userId)
            throws UserException, InternalException {

        final long sitemapId = getRequiredLongParam(req, PARAM_SITEMAP);
        final Integer src = getIntParam(req, PARAM_SOURCE);
        final SitemapSourceEnum source = src != null ? SitemapSourceEnum.R.fromValueOrNull(src) : null;
        if (source != SitemapSourceEnum.LATEST) {
            throw new UserException(UserProblem.ILLEGAL_PARAM_VALUE,
                    "Invalid value of parameter source: " + source + ". " +
                    "Only " + SitemapSourceEnum.LATEST + " or null are eligible.");
        }
        final Long indexId = getLongParam(req, PARAM_INDEX);

        BriefHostInfo briefHostInfo = getHostInfoAndVerify(req, userId);
        final HostDbHostInfo hostDbHostInfo = hostDbHostInfoService.getHostDbHostInfo(briefHostInfo.getName());

        final MergedSitemapInfo sitemapInfo = currentSitemapService.getSitemapInfo(hostDbHostInfo, sitemapId, indexId);
        if (sitemapInfo == null) {
            // нет такого sitemap
            return;
        }
        res.addData(new MergedSitemapInfoWrapper(sitemapInfo));

        final SitemapNavigationInfo navigationInfo =
                currentSitemapService.getNavigationInfo(hostDbHostInfo, sitemapId, indexId);
        try {
            if (navigationInfo != null) {
                res.addData(XmlUtil.xmlConvertable(navigationInfo));
            }
        } catch (JAXBException e) {
            log.error("Jaxb marshalling error ", e);
            throw new InternalException(InternalProblem.INTERNAL_PROBLEM, "Jaxb marshalling error", e);
        }

        Pager pager = createOutputStrategy(req).createPager();
        List<SitemapErrorInfo> sitemapErrors = currentSitemapService.getSitemapErrors(
                pager,
                new OrderByClause(
                        req,
                        SitemapService.FIELD_SEVERITY,
                        true,
                        new SitemapListCorrespondence()
                ),
                hostDbHostInfo,
                sitemapId);

        res.addData(XmlConvertableCollectionWrapper.wrap(sitemapErrors, SitemapErrorInfoWrapper.class, "sitemap-errors"));
        res.addData(pager);
    }

    @Required
    public void setCurrentSitemapService(CurrentSitemapService currentSitemapService) {
        this.currentSitemapService = currentSitemapService;
    }

    @Required
    public void setHostDbHostInfoService(HostDbHostInfoService hostDbHostInfoService) {
        this.hostDbHostInfoService = hostDbHostInfoService;
    }
}
