package ru.yandex.wmconsole.servantlet.top;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;

import ru.yandex.common.framework.core.ServRequest;
import ru.yandex.common.framework.core.ServResponse;
import ru.yandex.common.framework.pager.Pager;
import ru.yandex.common.util.collections.Pair;
import ru.yandex.common.util.db.OrderByClause;
import ru.yandex.wmconsole.data.TopQueryHistoryCorrespondence;
import ru.yandex.wmconsole.data.info.BriefHostInfo;
import ru.yandex.wmconsole.data.info.HostDbHostInfo;
import ru.yandex.wmconsole.data.wrappers.TopQueryHistoryInfoWrapper;
import ru.yandex.wmconsole.servantlet.WMCAuthorizedHostOperationServantlet;
import ru.yandex.wmconsole.service.HostDbHostInfoService;
import ru.yandex.wmconsole.service.WMCTopInfoService;
import ru.yandex.wmtools.common.data.TopQueryHistoryType;
import ru.yandex.wmtools.common.data.TopQueryHistoryTypeEnum;
import ru.yandex.wmtools.common.data.info.RegionInfo;
import ru.yandex.wmtools.common.data.info.RegionInfoTreeNode;
import ru.yandex.wmtools.common.data.info.TopQueryHistoryInfo;
import ru.yandex.wmtools.common.data.wrappers.DateIntervalWrapper;
import ru.yandex.wmtools.common.data.wrappers.IntegerWrapper;
import ru.yandex.wmtools.common.data.wrappers.RegionInfoTreeNodeWrapper;
import ru.yandex.wmtools.common.error.ExtraTagInfo;
import ru.yandex.wmtools.common.error.ExtraTagNameEnum;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.error.UserProblem;
import ru.yandex.wmtools.common.servantlet.top.WMToolsQueriesInfoServantlet;
import ru.yandex.wmtools.common.service.RegionsTreeCacheService;
import ru.yandex.wmtools.common.util.XmlConvertableCollectionWrapper;

public class TopQueryHistoryServantlet extends WMCAuthorizedHostOperationServantlet {
    private static final Logger log = LoggerFactory.getLogger(TopQueryHistoryServantlet.class);

    private static final String PARAM_QUERY_ID = "query_id";
    private static final String PARAM_REGION = "region";
    private static final String PARAM_TYPE = "type";
    private static final String PARAM_INCLUDE_SUBREGIONS = "include_subregions";
    private static final String PARAM_DATE = "date";

    private static final String TAG_SELECTED_REGION = "selected-region";

    private WMCTopInfoService topInfoService;
    private HostDbHostInfoService hostDbHostInfoService;
    private RegionsTreeCacheService regionsTreeCacheService;

    private Date getShiftedDate(ServRequest req) throws UserException {
        Date date = getDateParam(req, PARAM_DATE);
        if (date == null) {
            return null;
        }

        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.add(Calendar.DATE, 6);
        return calendar.getTime();
    }

    @Override
    public void doProcess(ServRequest req, ServResponse res, long userId) throws UserException, InternalException {

        BriefHostInfo briefHostInfo = getHostInfo(req, true, userId, true, true, res, true, true);
        if (briefHostInfo == null) {
            return;
        }

        HostDbHostInfo hostDbHostInfo = hostDbHostInfoService.getHostDbHostInfo(briefHostInfo.getName());

        Boolean includeSubregions = getBooleanParam(req, PARAM_INCLUDE_SUBREGIONS, WMToolsQueriesInfoServantlet.INCLUDE_SUBREGIONS_BY_DEFAULT);

        Integer region = getIntParam(req, PARAM_REGION);
        if (region == null || region < 0) {
            region = 0;
        }

        List<Long> selectedQueryIds = Arrays.asList(getMultiParamLong(req, PARAM_QUERY_ID));

        Date latestDate = topInfoService.getLatestInfoDateIfDateIsNull(region, hostDbHostInfo, null, includeSubregions);
        if (latestDate == null) {
            // отсутствуют данные по топам
            return;
        }
        Date date = getShiftedDate(req);
        if (date == null) {
            date = latestDate;
        }

        //TODO: Include subregions feature
        Map<Date, List<RegionInfo>> keyRegionsForDate = topInfoService.getKeyRegionsForAllTime(hostDbHostInfo);
        List<RegionInfo> keyRegions = keyRegionsForDate.get(date);
        if (keyRegions == null) {
            // отсутствуют даты для данного хоста
            throw new UserException(UserProblem.INVALID_TOP_DATE, "top date is invalid");
        }

        //TODO: КОСТЫЛЬ!!!
        List<RegionInfo> list = new ArrayList<RegionInfo>(keyRegions);
        for (Iterator<RegionInfo> i = list.iterator(); i.hasNext(); ) {
            int reg = i.next().getId();
            if (reg == 0 || reg == 10000) {
                i.remove();
            }
        }
        if (!topInfoService.existsInKeyRegions(region, keyRegions)) {
            // Если заданный в параметрах регион отсутствует в списке регионов за указанный период, то
            // по-умолчанию используется регион "Общая статистика"
            region = 0;
        }

        List<RegionInfoTreeNode> tree = regionsTreeCacheService.getTreeWithCountries(list);
        res.addData(XmlConvertableCollectionWrapper.wrap(tree, RegionInfoTreeNodeWrapper.class, "key-regions"));
        // Выводим выбранный регион
        log.debug("selected region is " + region);
        res.addData(new IntegerWrapper(TAG_SELECTED_REGION, region));

        TopQueryHistoryType type = getTypeParam(req);

        List<TopQueryHistoryInfo> selectedQueriesInfoList = topInfoService.getQueriesHistoryInfo(region,
                type.getTopQueryHistoryTypeEnum(),
                keyRegions,
                hostDbHostInfo,
                WMToolsQueriesInfoServantlet.HISTORY_LENGTH,
                includeSubregions,
                selectedQueryIds,
                keyRegionsForDate);
        res.addData(XmlConvertableCollectionWrapper.wrap(selectedQueriesInfoList, TopQueryHistoryInfoWrapper.class, "selected-top-queries-list"));

        Pager pager = createOutputStrategy(req).createPager();

        List<Date> dates = topInfoService.getKeyRegionsChangeDates(keyRegionsForDate);
        dates.add(new Date());
        List<Pair<Date, Date>> intervals = new ArrayList<Pair<Date, Date>>();
        for (int i = 0; i < dates.size() - 1; i++) {
            Calendar left = Calendar.getInstance();
            if (i == 0) {
                left.setTime(dates.get(i));
                left.add(Calendar.DATE, -6);
            } else {
                left.setTime(intervals.get(i - 1).getSecond());
                left.add(Calendar.DATE, 1);
            }

            Calendar right = Calendar.getInstance();
            if (i == dates.size() - 2) {
                right.setTime(dates.get(i + 1));
            } else {
                right.setTime(dates.get(i + 1));
                right.add(Calendar.DATE, -7);
            }
            intervals.add(new Pair<Date, Date>(left.getTime(), right.getTime()));
        }
        res.addData(XmlConvertableCollectionWrapper.wrap(intervals, DateIntervalWrapper.class, "dates"));

        OrderByClause order = new OrderByClause(req, type.getOrderByDbField(), type.isAscendingOrder(), new TopQueryHistoryCorrespondence());
        List<TopQueryHistoryInfo> historyInfoList = topInfoService.getQueriesHistoryInfo(region, type.getTopQueryHistoryTypeEnum(),
                keyRegions, hostDbHostInfo, WMToolsQueriesInfoServantlet.HISTORY_LENGTH, includeSubregions, order, pager, latestDate);

        for (Iterator<TopQueryHistoryInfo> iter = historyInfoList.iterator(); iter.hasNext(); ) {
            TopQueryHistoryInfo info = iter.next();
            if (info.getInTopCount() < WMToolsQueriesInfoServantlet.HISTORY_LENGTH) {
                iter.remove();
            }
        }
        if (!historyInfoList.isEmpty()) {
            res.addData(XmlConvertableCollectionWrapper.wrap(historyInfoList, TopQueryHistoryInfoWrapper.class, "top-queries-list"));
            res.addData(pager);
        }
    }

    private TopQueryHistoryType getTypeParam(ServRequest req) throws UserException {
        String paramType = req.getParam(PARAM_TYPE, true);
        TopQueryHistoryTypeEnum typeEnum = TopQueryHistoryTypeEnum.getByName(paramType);

        TopQueryHistoryType result;
        if (TopQueryHistoryTypeEnum.CLICKS.equals(typeEnum)) {
            result = new TopQueryHistoryType(typeEnum, WMCTopInfoService.FIELD_CLICKS, false);
        } else if (TopQueryHistoryTypeEnum.POSITION.equals(typeEnum)) {
            result = new TopQueryHistoryType(typeEnum, WMCTopInfoService.FIELD_POSITION, true);
        } else if (TopQueryHistoryTypeEnum.PART.equals(typeEnum)) {
            result = new TopQueryHistoryType(typeEnum, WMCTopInfoService.FIELD_PART, false);
        } else if (TopQueryHistoryTypeEnum.CTR.equals(typeEnum)) {
            result = new TopQueryHistoryType(typeEnum, WMCTopInfoService.FIELD_VALUE, false);
        } else if (TopQueryHistoryTypeEnum.SHOWS.equals(typeEnum)) {
            result = new TopQueryHistoryType(typeEnum, WMCTopInfoService.FIELD_SHOWS, false);
        } else {
            throw new UserException(UserProblem.ILLEGAL_PARAM_VALUE, "Parameter " + PARAM_TYPE + " has illegal value: " + paramType,
                    new ExtraTagInfo(ExtraTagNameEnum.PARAM, PARAM_TYPE), new ExtraTagInfo(ExtraTagNameEnum.VALUE, paramType));
        }
        return result;
    }

    @Required
    public void setTopInfoService(WMCTopInfoService topInfoService) {
        this.topInfoService = topInfoService;
    }

    @Required
    public void setHostDbHostInfoService(HostDbHostInfoService hostDbHostInfoService) {
        this.hostDbHostInfoService = hostDbHostInfoService;
    }

    @Required
    public void setRegionsTreeCacheService(RegionsTreeCacheService regionsTreeCacheService) {
        this.regionsTreeCacheService = regionsTreeCacheService;
    }
}
