package ru.yandex.wmconsole.servantlet.virus;

import java.util.List;

import org.jetbrains.annotations.Nullable;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Required;

import ru.yandex.common.framework.core.ServRequest;
import ru.yandex.common.framework.core.ServResponse;
import ru.yandex.common.util.db.OrderByClause;
import ru.yandex.webmaster.viewer.service.NewWebmasterProxyService;
import ru.yandex.wmconsole.data.VirusRecheckEnum;
import ru.yandex.wmconsole.data.info.BriefHostInfo;
import ru.yandex.wmconsole.data.info.RecheckInfo;
import ru.yandex.wmconsole.data.info.VirusInfo;
import ru.yandex.wmconsole.data.wrappers.VirusInfoWrapper;
import ru.yandex.wmconsole.data.wrappers.VirusRecheckInfoWrapper;
import ru.yandex.wmconsole.servantlet.WMCAuthorizedHostOperationServantlet;
import ru.yandex.wmconsole.service.VirusedHostService;
import ru.yandex.wmtools.common.data.wrappers.DateWrapper;
import ru.yandex.wmtools.common.data.wrappers.SimpleWrapper;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.util.XmlConvertableCollectionWrapper;

/**
 * @author Andrey Mima (amima@yandex-team.ru)
 */
public class VirusInfoServantlet extends WMCAuthorizedHostOperationServantlet {
    private static final String TAG_HOST = "host";
    private static final String TAG_INFECTED = "infected";
    private static final String TAG_RECHECK = "recheck";
    private static final String TAG_RECHECK_DATE = "recheck-date";
    private static final String TAG_VIRUS_LIST = "virus-list";

    private VirusedHostService virusedHostService;
    private NewWebmasterProxyService newWebmasterProxyService;

    @Override
    protected void doProcess(ServRequest req, ServResponse res, long userId) throws UserException, InternalException {

        BriefHostInfo briefHostInfo = getHostInfoAndVerify(req, userId);

        List<VirusInfo> virusInfoList = virusedHostService.getVirusInfo(briefHostInfo, getOrderBy(req));
        XmlConvertableCollectionWrapper wrapper = XmlConvertableCollectionWrapper.wrap(
                virusInfoList,
                VirusInfoWrapper.class,
                TAG_VIRUS_LIST
        );
        res.addData(wrapper);
        res.addData(new SimpleWrapper<Long>(briefHostInfo.getId(), TAG_HOST));

        res.addData(new SimpleWrapper<Boolean>(!virusInfoList.isEmpty(), TAG_INFECTED));

        DateTime lastCheck = getMaxLastCheck(virusInfoList);
        RecheckInfo recheckInfo = newWebmasterProxyService.getVirusRecheckInfo(userId, briefHostInfo);

        if (virusInfoList.isEmpty()) {
            recheckInfo = null;
        } else if (recheckInfo == null) {
            recheckInfo = new RecheckInfo(VirusRecheckEnum.NONE, null);
        } else if (lastCheck != null && recheckInfo.getDate().before(lastCheck.toDate())) {
            recheckInfo = new RecheckInfo(VirusRecheckEnum.NONE, recheckInfo.getDate());
        }

        if (recheckInfo != null) {
            // new version of recheck info representation
            res.addData(new VirusRecheckInfoWrapper(recheckInfo));

            // do not remove because recheck tag is used by frontend to enable or disable recheck button
            res.addData(new SimpleWrapper<Boolean>(!recheckInfo.isRecheck(), TAG_RECHECK));
            res.addData(new DateWrapper(TAG_RECHECK_DATE, recheckInfo.getDate()));
        }
    }

    private @Nullable
    DateTime getMaxLastCheck(List<VirusInfo> virusInfoList) {
        if (virusInfoList.isEmpty()) {
            return null;
        }
        DateTime result = null;
        for (VirusInfo info : virusInfoList) {
            if (info.getLastCheck() != null) {
                if (result == null) {
                    result = new DateTime(info.getLastCheck());
                } else if (new DateTime(info.getLastCheck()).isAfter(result)) {
                    result = new DateTime(info.getLastCheck());
                }
            }
        }
        return result;
    }

    private OrderByClause getOrderBy(ServRequest req) {
        return new OrderByClause(
                req,
                virusedHostService.getDefaultOrderByColumn(),
                virusedHostService.getDefaultIsAscending(),
                virusedHostService.getColumnNameCorrespondence());
    }

    @Required
    public void setVirusedHostService(VirusedHostService virusedHostService) {
        this.virusedHostService = virusedHostService;
    }

    @Required
    public void setNewWebmasterProxyService(NewWebmasterProxyService newWebmasterProxyService) {
        this.newWebmasterProxyService = newWebmasterProxyService;
    }
}
