package ru.yandex.wmconsole.viewer.api.servantlet.sitemap;

import java.util.Date;

import org.springframework.beans.factory.annotation.Required;

import ru.yandex.common.framework.core.ServRequest;
import ru.yandex.common.framework.core.ServResponse;
import ru.yandex.webmaster.common.util.xml.SimpleXmlBuilder;
import ru.yandex.wmconsole.authorization.ApiOAuthPermission;
import ru.yandex.wmconsole.data.info.BriefHostInfo;
import ru.yandex.wmconsole.data.info.HostDbHostInfo;
import ru.yandex.wmconsole.data.info.MergedSitemapInfo;
import ru.yandex.wmconsole.data.sitemaps.SitemapSourceEnum;
import ru.yandex.wmconsole.data.sitemaps.SitemapTypeEnum;
import ru.yandex.wmconsole.error.ClientException;
import ru.yandex.wmconsole.error.ClientProblem;
import ru.yandex.wmconsole.servantlet.api.AbstractApiServantlet;
import ru.yandex.wmconsole.service.HostDbHostInfoService;
import ru.yandex.wmconsole.viewer.api.ApiLink;
import ru.yandex.wmconsole.viewer.api.ApiLinkType;
import ru.yandex.wmconsole.viewer.api.ApiUrl;
import ru.yandex.wmconsole.viewer.api.ApiXmlUtils;
import ru.yandex.wmconsole.viewer.api.SitemapIdApiFormatter;
import ru.yandex.wmconsole.viewer.api.servantlet.ApiParameterNames;
import ru.yandex.wmconsole.viewer.sitemap.SitemapId;
import ru.yandex.wmconsole.viewer.sitemap.SitemapService2;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.util.XmlDataWrapper;

/**
 * @author aherman
 */
public class ApiGetSitemapServantlet extends AbstractApiServantlet {
    private SitemapService2 sitemapService2;
    private HostDbHostInfoService hostDbHostInfoService;
    private ApiUrl apiUrl;

    @Override
    protected void apiDoProcess(ServRequest req, ServResponse res, long userId) throws ClientException, InternalException {
        String externalId = getRequiredStringParam(req, ApiParameterNames.SITEMAP_ID);
        SitemapId sitemapId = SitemapIdApiFormatter.parseFromExternalId(externalId);
        Long indexId = null;

        BriefHostInfo briefHostInfo;
        MergedSitemapInfo latestSitemap = null;
        MergedSitemapInfo inSearchSitemap = null;

        try {
            briefHostInfo = getHostInfo(req, false, userId, true, true, res, false, false);
            HostDbHostInfo hostDbHostInfo = hostDbHostInfoService.getHostDbHostInfo(briefHostInfo.getName());

            if (sitemapId.getSitemapSource() == SitemapSourceEnum.LATEST) {
                latestSitemap = sitemapService2.getCurrentSitemap(hostDbHostInfo, sitemapId, indexId);
                if (latestSitemap != null) {
                    inSearchSitemap = sitemapService2.getOldSitemap(hostDbHostInfo, latestSitemap);
                }
            } else {
                inSearchSitemap = sitemapService2.getOldSitemap(hostDbHostInfo, sitemapId);
            }

        } catch (UserException e) {
            throw new ClientException(ClientProblem.INVALID_URL, e.getMessage(), e);
        }

        if (inSearchSitemap == null) {
            throw new ClientException(ClientProblem.SITEMAP_NOT_FOUND, "Sitemap not found");
        }

        ApiUrl.V2.Host host = apiUrl.v2().host(briefHostInfo.getId());
        res.addData(new ApiGetSitemapResponse(sitemapId, latestSitemap, inSearchSitemap, host));
    }

    private static class ApiGetSitemapResponse extends XmlDataWrapper<MergedSitemapInfo> {
        private final SitemapId sitemapId;
        private final MergedSitemapInfo latestSitemap;
        private final MergedSitemapInfo inSearchSitemap;
        private ApiUrl.V2.Host apiHostUrl;

        public ApiGetSitemapResponse(SitemapId sitemapId, MergedSitemapInfo latestSitemap, MergedSitemapInfo inSearchSitemap,
                ApiUrl.V2.Host apiHostUrl)
        {
            super(null);
            this.sitemapId = sitemapId;
            this.latestSitemap = latestSitemap;
            this.inSearchSitemap = inSearchSitemap;
            this.apiHostUrl = apiHostUrl;
        }

        @Override
        protected void doToXml(StringBuilder result) {
            SimpleXmlBuilder xml = new SimpleXmlBuilder(result);
            xml.open("sitemap");

            String externalSitemapUrl;
            if (latestSitemap != null) {
                externalSitemapUrl = latestSitemap.getUrl();
            } else {
                externalSitemapUrl = inSearchSitemap.getUrl();
            }
            ApiXmlUtils.link(xml, externalSitemapUrl, ApiLinkType.SITEMAP_CONTENT);

            ApiLink sitemapApiLink = apiHostUrl.sitemap(sitemapId);
            ApiXmlUtils.link(xml, sitemapApiLink.getUri(), ApiLinkType.SELF);

            if (latestSitemap != null && latestSitemap.getSitemapType() == SitemapTypeEnum.SITEMAPINDX) {
                ApiLink sitemapChildrenLink = apiHostUrl.sitemapChildrens(sitemapId);
                ApiXmlUtils.link(xml, sitemapChildrenLink);
            }

            if (latestSitemap != null && latestSitemap.getProccessedDate() != null) {
                xml.open("info").attribute("type", "latest");
                writeSitemapInfo(xml, latestSitemap);
                xml.close(); // info
            }

            if (inSearchSitemap != null) {
                Date processedDate = inSearchSitemap.getRealProcessedDate();
                if (processedDate == null) {
                    processedDate = inSearchSitemap.getProccessedDate();
                }
                if (processedDate != null) {
                    xml.open("info").attribute("type", "in-search");
                    writeSitemapInfo(xml, inSearchSitemap);
                    xml.close(); // info
                }
            }

            xml.close(); // sitemap
        }

        private void writeSitemapInfo(SimpleXmlBuilder xml, MergedSitemapInfo sitemap) {
            xml.element("url-count", sitemap.getUrlsCount());
            xml.element("url-error-count", sitemap.getUrlErrorsCount());
            xml.element("error-count", sitemap.getXmlErrorsCount());
            xml.element("warning-count", sitemap.getWarningsCount());

            Date submittedDate = sitemap.getSubmittedDate();
            if (submittedDate != null) {
                ApiXmlUtils.dateElement(xml, "submitted-on", submittedDate);
            }
            if (sitemap.isFromUser()) {
                xml.element("added-by", "user");
            }
            if (sitemap.isFromRobotsTxt()) {
                xml.element("added-by", "robots-txt");
            }

            ApiXmlUtils.dateTimeElement(xml, "processed-on", sitemap.getProccessedDate());
            ApiXmlUtils.dateTimeElement(xml, "real-processed-on", sitemap.getRealProcessedDate());
            if (sitemap.getSitemapFormat() != null) {
                xml.element("format", sitemap.getSitemapFormat());
            }
            if (sitemap.getSitemapType() != null) {
                xml.element("type", sitemap.getSitemapType());
            }
        }
    }

    @Override
    public ApiOAuthPermission getRequiredPermission() {
        return ApiOAuthPermission.COMMON_OPERATIONS;
    }

    @Required
    public void setSitemapService2(SitemapService2 sitemapService2) {
        this.sitemapService2 = sitemapService2;
    }

    @Required
    public void setHostDbHostInfoService(HostDbHostInfoService hostDbHostInfoService) {
        this.hostDbHostInfoService = hostDbHostInfoService;
    }

    @Required
    public void setApiUrl(ApiUrl apiUrl) {
        this.apiUrl = apiUrl;
    }
}
