package ru.yandex.wmconsole.viewer.sitemap;

import java.net.URL;
import java.util.Collections;
import java.util.List;

import org.apache.http.Header;
import org.apache.http.HttpResponse;
import org.apache.http.protocol.HTTP;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;

import ru.yandex.misc.db.q.SqlLimits;
import ru.yandex.wmconsole.data.info.HostDbHostInfo;
import ru.yandex.wmconsole.data.info.MergedSitemapInfo;
import ru.yandex.wmconsole.data.info.sitemap.SitemapIndexInfo;
import ru.yandex.wmconsole.service.error.WMCUserProblem;
import ru.yandex.wmconsole.viewer.sitemap.dao.TblSitemapIndexMetaDao;
import ru.yandex.wmconsole.viewer.sitemap.dao.TblSitemapMetaDao;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.error.UserProblem;
import ru.yandex.wmtools.common.servantlet.AbstractServantlet;
import ru.yandex.wmtools.common.sita.DocumentFormatEnum;
import ru.yandex.wmtools.common.sita.SitaException;
import ru.yandex.wmtools.common.sita.SitaIncompleteResponseException;
import ru.yandex.wmtools.common.sita.SitaRedirectCycleException;
import ru.yandex.wmtools.common.sita.SitaRedirectService;
import ru.yandex.wmtools.common.sita.SitaRequestTimeout;
import ru.yandex.wmtools.common.sita.SitaUrlFetchRequest;
import ru.yandex.wmtools.common.sita.SitaUrlFetchRequestBuilder;
import ru.yandex.wmtools.common.sita.SitaUrlFetchResponse;
import ru.yandex.wmtools.common.sita.UserAgentEnum;
import ru.yandex.wmtools.common.util.URLUtil;
import ru.yandex.wmtools.common.util.http.YandexHttpStatus;

/**
 * @author aherman
 */
public class SitemapService2 {
    private static final Logger log = LoggerFactory.getLogger(SitemapService2.class);

    public static final int MAX_SITEMAP_LEGHT = 10 * 1024 * 1024;

    private TblSitemapMetaDao tblSitemapMetaDao;
    private TblSitemapIndexMetaDao tblSitemapIndexMetaDao;

    private SitaRedirectService sitaRedirectService;

    public SitaUrlFetchResponse getAndValidateSitemap(URL sitemapUrl) throws UserException, InternalException {
        if (URLUtil.isHomePage(sitemapUrl)) {
            throw new UserException(WMCUserProblem.SITEMAP_IS_HOMEPAGE, "Homepage url is not allowed");
        }

        SitaUrlFetchRequest sitaUrlFetchRequest = new SitaUrlFetchRequestBuilder(sitemapUrl)
                .setCheckAllowedInRobotsTxt(true)
                .setDocumentFormat(DocumentFormatEnum.DF_HTTP_RESPONSE)
                .setUserAgent(UserAgentEnum.ROBOT)
                .setRequestTimeout(SitaRequestTimeout._15_SECONDS)
                .createSitaUrlFetchRequest();

        SitaUrlFetchResponse sitaUrlFetchResponse = null;
        try {
            sitaUrlFetchResponse = sitaRedirectService.followRedirects(sitaUrlFetchRequest);
        } catch (SitaRedirectCycleException e) {
            throw new UserException(WMCUserProblem.SITEMAP_ACCESS_ERROR, "Unable to load sitemap " + sitemapUrl + " due to cyclic redirection");
        } catch (SitaIncompleteResponseException e) {
            throw new UserException(WMCUserProblem.SITEMAP_ACCESS_ERROR, "Request timeout or connection error for sitemap " + sitemapUrl);
        } catch (SitaException e) {
            log.error("Unable to load sitemap", e);
            throw new UserException(WMCUserProblem.SITEMAP_ACCESS_ERROR, "Unable to load sitemap " + sitemapUrl);
        }

        if (!sitaUrlFetchResponse.isAllowedInRobotsTxt()) {
            throw new UserException(WMCUserProblem.URL_IS_DISALLOWED_IN_ROBOTS_TXT, "Url " + sitemapUrl + " is disallowed in robots.txt.");
        }
        if (sitaUrlFetchResponse.getSitaHttpStatus() == YandexHttpStatus.HTTP_1002_BODY_TOO_LARGE) {
            throw new UserException(UserProblem.SITEMAP_TOO_LONG, "Sitemap file size limit exceeded");
        }
        if (sitaUrlFetchResponse.getSitaHttpStatus() == YandexHttpStatus.HTTP_1010_CONNECT_FAILED) {
            throw new UserException(WMCUserProblem.SITEMAP_ACCESS_ERROR, "Failed to load sitemap " + sitemapUrl);
        }
        if (sitaUrlFetchResponse.getSitaHttpStatus() != YandexHttpStatus.HTTP_200_OK) {
            throw new UserException(WMCUserProblem.SITEMAP_IS_NOT_AVAILABLE, "Not available: " + sitemapUrl);
        }
        HttpResponse httpHeaders = sitaUrlFetchResponse.getParsedHttpHeaders();
        Header contentLengthHeader = httpHeaders.getFirstHeader(HTTP.CONTENT_LEN);
        if (contentLengthHeader != null) {
            try {
                int contentLength = Integer.parseInt(contentLengthHeader.getValue());
                if (contentLength <= 0) {
                    throw new UserException(WMCUserProblem.EMPTY_SITEMAP, "File " + sitemapUrl + " is empty");
                }
                if (contentLength >= MAX_SITEMAP_LEGHT) {
                    throw new UserException(UserProblem.SITEMAP_TOO_LONG, "Sitemap file size limit exceeded");
                }
            } catch (NumberFormatException e) {
                log.error("Unable to parse Content-Length header: " + contentLengthHeader);
            }
        }
        return sitaUrlFetchResponse;
    }

    public MergedSitemapInfo getCurrentSitemap(HostDbHostInfo hostDbHostInfo, SitemapId sitemapId, Long indexId)
            throws InternalException, UserException
    {
        MergedSitemapInfo sitemap = tblSitemapMetaDao.getLatestSitemap(hostDbHostInfo, sitemapId);
        if (sitemap != null) {
            calculateIndexIdForSitemapId(hostDbHostInfo, sitemapId, indexId, sitemap);
        }

        return sitemap;
    }

    public MergedSitemapInfo getOldSitemap(HostDbHostInfo hostDbHostInfo, SitemapId sitemapId)
            throws InternalException
    {
        MergedSitemapInfo oldSitemap = tblSitemapMetaDao.getOldSitemap(hostDbHostInfo, sitemapId);

        return oldSitemap;
    }


    public MergedSitemapInfo getOldSitemap(HostDbHostInfo hostDbHostInfo, MergedSitemapInfo latestSitemap)
            throws InternalException
    {
        MergedSitemapInfo oldSitemap = tblSitemapMetaDao.getOldSitemap(hostDbHostInfo, latestSitemap);

        return oldSitemap;
    }

    private void calculateIndexIdForSitemapId(HostDbHostInfo hostDbHostInfo, SitemapId sitemapId, Long indexId,
            MergedSitemapInfo sitemap)
            throws InternalException, UserException
    {
        List<SitemapIndexInfo> indexList = tblSitemapIndexMetaDao.getSitemapIndexInfo(hostDbHostInfo, sitemapId);

        if (indexList.isEmpty()) {
            sitemap.setIndexId(null);
            sitemap.setIndexIdCount(0L);
            return;
        }

        sitemap.setIndexIdCount((long) indexList.size());

        // вначале попытаемся найти indexId, переданный через параметры (если есть)
        if (indexId != null) {
            for (SitemapIndexInfo indexInfo : indexList) {
                if (indexId.equals(indexInfo.getIndexId())) {
                    sitemap.setIndexId(indexId);
                    return;
                }
            }
        }

        // затем попытаемся найти sitemapIndex для того же хоста
        for (SitemapIndexInfo info : indexList) {
            String hostName = URLUtil.getHostName(
                    AbstractServantlet.prepareUrl(info.getHostName(), true), false);
            if (hostName.equals(hostDbHostInfo.getName())) {
                sitemap.setIndexId(info.getIndexId());
                return;
            }
        }

        sitemap.setIndexId(indexList.iterator().next().getIndexId());
    }

    public List<SitemapLink> getSitemapIndexChildren(HostDbHostInfo hostDbHostInfo, SitemapId sitemapId, SqlLimits limits)
            throws InternalException
    {
        List<SitemapId> sitemapIndexChildren = tblSitemapIndexMetaDao.getSitemapIndexChildren(hostDbHostInfo, sitemapId,
                limits);
        if (sitemapIndexChildren.isEmpty()) {
            return Collections.emptyList();
        }
        return tblSitemapMetaDao.getLatestSitemapLinks(hostDbHostInfo, sitemapIndexChildren);
    }

    @Required
    public void setTblSitemapMetaDao(TblSitemapMetaDao tblSitemapMetaDao) {
        this.tblSitemapMetaDao = tblSitemapMetaDao;
    }

    @Required
    public void setTblSitemapIndexMetaDao(TblSitemapIndexMetaDao tblSitemapIndexMetaDao) {
        this.tblSitemapIndexMetaDao = tblSitemapIndexMetaDao;
    }

    @Required
    public void setSitaRedirectService(SitaRedirectService sitaRedirectService) {
        this.sitaRedirectService = sitaRedirectService;
    }
}

