package ru.yandex.wmconsole.viewer.sitemap.dao;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.springframework.jdbc.core.simple.ParameterizedRowMapper;

import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.db.q.SqlLimits;
import ru.yandex.wmconsole.data.SeverityEnum;
import ru.yandex.wmconsole.data.info.HostDbHostInfo;
import ru.yandex.wmconsole.data.info.MergedSitemapInfo;
import ru.yandex.wmconsole.data.partition.WMCPartition;
import ru.yandex.wmconsole.data.sitemaps.SitemapFormatEnum;
import ru.yandex.wmconsole.data.sitemaps.SitemapSourceEnum;
import ru.yandex.wmconsole.data.sitemaps.SitemapTypeEnum;
import ru.yandex.wmconsole.viewer.sitemap.SitemapId;
import ru.yandex.wmconsole.viewer.sitemap.SitemapLink;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.service.AbstractDbService;
import ru.yandex.wmtools.common.util.SqlUtil;

import static ru.yandex.misc.db.q.ConditionUtils.column;

/**
 * @author aherman
 */
public class TblSitemapMetaDao extends AbstractDbService {
    private static final String SELECT_SITEMAP_LIST_QUERY =
            "(SELECT\n" +
            "  sc.sitemap_id AS sitemap_id,\n" +
            "  sc.host_name AS host_name,\n" +
            "  sc.url_name AS url_name,\n" +
            "  " + SitemapSourceEnum.LATEST.getId() + " AS sitemap_source\n" +
            "  FROM tbl_sitemaps_cur sc\n" +
            "    LEFT JOIN tbl_sitemaps s\n" +
            "    ON (s.from_user = 1 AND s.host_id = sc.host_id AND s.host_name = sc.host_name AND s.url_name = BINARY sc.url_name)\n" +
            "  WHERE sc.host_id = ? AND (s.from_user = 1 OR sc.from_robots_txt = 1)\n" +
            ")\n" +
            "UNION\n" +
            "(SELECT\n" +
            "  s.sitemap_id AS sitemap_id,\n" +
            "  s.host_name AS host_name,\n" +
            "  s.url_name AS url_name,\n" +
            "  " + SitemapSourceEnum.IN_SEARCH.getId() + " AS sitemap_source\n" +
            "  FROM tbl_sitemaps s\n" +
            "    LEFT JOIN tbl_sitemaps_cur sc\n" +
            "    ON (s.host_id = sc.host_id AND s.host_name = sc.host_name AND s.url_name = BINARY sc.url_name)\n" +
            "  WHERE s.host_id = ? AND s.from_user = 1 AND sc.sitemap_id IS NULL\n" +
            ")"
            ;

    public List<SitemapLink> listSitemaps(HostDbHostInfo hostDbHostInfo, SqlLimits limits) throws InternalException {
        String q = SELECT_SITEMAP_LIST_QUERY + "ORDER BY sitemap_id " + limits.toMysqlLimits();
        long hostId = hostDbHostInfo.getHostDbHostId();
        return getJdbcTemplate(new WMCPartition(hostDbHostInfo, null)).query(q, getSitemapLinkMapper(), hostId, hostId);
    }

    public int countSitemaps(HostDbHostInfo hostDbHostInfo) throws InternalException {
        String q = "SELECT count(*) FROM (" + SELECT_SITEMAP_LIST_QUERY + ") c";
        long hostId = hostDbHostInfo.getHostDbHostId();
        return getJdbcTemplate(new WMCPartition(hostDbHostInfo, null)).queryForInt(q, hostId, hostId);
    }

    public MergedSitemapInfo getLatestSitemap(HostDbHostInfo hostDbHostInfo, SitemapId sitemapId)
            throws InternalException
    {
        String q =
                "SELECT" +
                "    sitemap_id," +
                "    (SELECT COUNT(*) FROM tbl_sitemap_cur_errors" +
                "        WHERE sitemap_id=sc.sitemap_id AND type IN" +
                "          (SELECT id FROM tbl_dic_sitemap_error_type WHERE severity = " + SeverityEnum.ERROR.getValue() +")" +
                "        ) AS xml_error_count," +
                "    (SELECT COUNT(*) FROM tbl_sitemap_cur_errors" +
                "        WHERE sitemap_id=sc.sitemap_id AND type IN" +
                "          (SELECT id FROM tbl_dic_sitemap_error_type WHERE severity = " + SeverityEnum.WARNING.getValue() + ")" +
                "        ) AS warning_count," +
                "    (SELECT 1 FROM tbl_sitemaps_indexes_cur sic WHERE sic.host_id=sc.host_id AND sic.index_id=sc.sitemap_id LIMIT 1" +
                "        ) AS has_children," +
                "    host_name," +
                "    url_name," +
                "    submitted_on," +
                "    processed_on," +
                "    real_processed_on," +
                "    from_robots_txt," +
                "    format," +
                "    type," +
                "    url_count" +
                "  FROM" +
                "    tbl_sitemaps_cur AS sc" +
                "  WHERE" +
                "    sitemap_id = ?" +
                "  AND" +
                "    host_id = ?" +
                ";";

        List<MergedSitemapInfo> query = getJdbcTemplate(new WMCPartition(hostDbHostInfo, null)).query(q,
                getSitemapCurMapper(), sitemapId.getSitemapId(), hostDbHostInfo.getHostDbHostId());

        return query.isEmpty() ? null : query.get(0);
    }

    public MergedSitemapInfo getOldSitemap(HostDbHostInfo hostDbHostInfo, MergedSitemapInfo latestSitemap)
            throws InternalException
    {
        String q =
                "SELECT " +
                "    sitemap_id, " +
                "    host_name," +
                "    url_name, " +
                "    submitted_on, " +
                "    processed_on, " +
                "    real_processed_on, " +
                "    from_robots_txt, " +
                "    from_user, " +
                "    format, " +
                "    type, " +
                "    url_count, " +
                "    (SELECT SUM(count) FROM tbl_code_error_sitemap" +
                "        WHERE sitemap_id=s.sitemap_id" +
                "        ) AS url_error_count, " +
                "    (SELECT COUNT(*) FROM tbl_sitemap_errors" +
                "        WHERE sitemap_id=s.sitemap_id AND type IN" +
                "          (SELECT id FROM tbl_dic_sitemap_error_type WHERE severity = " + SeverityEnum.ERROR.getValue() +")" +
                "        ) AS xml_error_count," +
                "    (SELECT COUNT(*) FROM tbl_sitemap_errors" +
                "        WHERE sitemap_id=s.sitemap_id AND type IN" +
                "          (SELECT id FROM tbl_dic_sitemap_error_type WHERE severity = " + SeverityEnum.WARNING.getValue() + ")" +
                "        ) AS warning_count" +
                "  FROM " +
                "    tbl_sitemaps AS s" +
                "  WHERE " +
                "    url_name = BINARY ? " +
                "    AND host_name = ? " + // host_name в sitemap может отличаться от имени хоста
                "    AND host_id = ?";

        List<MergedSitemapInfo> query = getJdbcTemplate(new WMCPartition(hostDbHostInfo, null))
                .query(q, getSitemapInfoRowMapper(), latestSitemap.getSitemapPath(),
                        latestSitemap.getSitemapHost(), hostDbHostInfo.getHostDbHostId());

        if (query.isEmpty()) {
            return null;
        }
        return query.get(0);
    }

    public MergedSitemapInfo getOldSitemap(HostDbHostInfo hostDbHostInfo, SitemapId sitemapId)
            throws InternalException
    {
        String q =
                "SELECT " +
                "    sitemap_id, " +
                "    host_name," +
                "    url_name, " +
                "    submitted_on, " +
                "    processed_on, " +
                "    real_processed_on, " +
                "    from_robots_txt, " +
                "    from_user, " +
                "    format, " +
                "    type, " +
                "    url_count, " +
                "    (SELECT SUM(count)" +
                "        FROM tbl_code_error_sitemap" +
                "        WHERE sitemap_id=s.sitemap_id" +
                "      ) AS url_error_count, " +
                "    (SELECT COUNT(*) FROM tbl_sitemap_errors" +
                "        WHERE sitemap_id=s.sitemap_id AND type IN" +
                "          (SELECT id FROM tbl_dic_sitemap_error_type WHERE severity = " + SeverityEnum.ERROR.getValue() +")" +
                "        ) AS xml_error_count," +
                "    (SELECT COUNT(*) FROM tbl_sitemap_errors" +
                "        WHERE sitemap_id=s.sitemap_id AND type IN" +
                "          (SELECT id FROM tbl_dic_sitemap_error_type WHERE severity = " + SeverityEnum.WARNING.getValue() + ")" +
                "        ) AS warning_count" +
                "  FROM " +
                "    tbl_sitemaps AS s" +
                "  WHERE " +
                "    sitemap_id = ?";

        List<MergedSitemapInfo> query = getJdbcTemplate(new WMCPartition(hostDbHostInfo, null))
                .query(q, getSitemapInfoRowMapper(), sitemapId.getSitemapId());

        if (query.isEmpty()) {
            return null;
        }
        return query.get(0);
    }

    public List<SitemapLink> getLatestSitemapLinks(HostDbHostInfo hostDbHostInfo, List<SitemapId> sitemapIds)
            throws InternalException
    {
        List<Long> ids = new ArrayList<Long>(sitemapIds.size());
        for (SitemapId sitemapId : sitemapIds) {
            ids.add(sitemapId.getSitemapId());
        }
        SqlCondition condition = column("host_id").eq(hostDbHostInfo.getHostDbHostId())
                .and(column("sitemap_id").inSet(ids));

        String q =
                "SELECT " +
                "    sitemap_id," +
                "    host_name," +
                "    url_name," +
                "    " +SitemapSourceEnum.LATEST.getId() + " AS sitemap_source" +
                "  FROM tbl_sitemaps_cur WHERE " + condition.sql();

        return getJdbcTemplate(new WMCPartition(hostDbHostInfo, null)).query(q, getSitemapLinkMapper(), condition.args().toArray());
    }

    private static ParameterizedRowMapper<MergedSitemapInfo> getSitemapCurMapper() {
        return new ParameterizedRowMapper<MergedSitemapInfo>() {
            @Override
            public MergedSitemapInfo mapRow(ResultSet rs, int rowNum) throws SQLException {
                long sitemapId = rs.getLong("sitemap_id");
                Integer xmlErrorsCount = rs.getInt("xml_error_count");
                Integer warningsCount = rs.getInt("warning_count");
                boolean hasChildren = rs.getBoolean("has_children");
                String sitemapHost = rs.getString("host_name");
                String sitemapPath = rs.getString("url_name");
                Date submittedOn = SqlUtil.safeGetTimestamp(rs, "submitted_on");
                Date processedOn = SqlUtil.safeGetTimestamp(rs, "processed_on");
                Date realProcessedOn = SqlUtil.safeGetTimestamp(rs, "real_processed_on");
                boolean fromRobotsTxt = rs.getBoolean("from_robots_txt");

                SitemapFormatEnum format;
                int formatId = rs.getInt("format");
                if (rs.wasNull()) {
                    format = null;
                } else {
                    format = SitemapFormatEnum.getByValue(formatId);
                }

                SitemapTypeEnum type;
                int typeId = rs.getInt("type");
                if (rs.wasNull()) {
                    type = null;
                } else {
                    type = SitemapTypeEnum.getByValue(typeId);
                }

                Integer urlsCount = SqlUtil.getIntNullable(rs, "url_count");

                if (processedOn == null) {
                    xmlErrorsCount = null;
                    warningsCount = null;
                }

                if (submittedOn != null && submittedOn.getTime() <= 0) {
                    submittedOn = null;
                }

                if (type == null && xmlErrorsCount != null && xmlErrorsCount > 0) {
                    type = SitemapTypeEnum.SITEMAP;
                }


                return new MergedSitemapInfo(
                        sitemapId,
                        null,
                        null,
                        sitemapHost,
                        sitemapPath,
                        submittedOn,
                        fromRobotsTxt,
                        false,
                        processedOn,
                        realProcessedOn,
                        format,
                        type,
                        urlsCount,
                        xmlErrorsCount,
                        0,
                        warningsCount,
                        hasChildren,
                        SitemapSourceEnum.LATEST
                );
            }
        };
    }

    private static ParameterizedRowMapper<SitemapLink> getSitemapLinkMapper() {
        return new ParameterizedRowMapper<SitemapLink>() {
            @Override
            public SitemapLink mapRow(ResultSet rs, int rowNum) throws SQLException {
                Long sitemapId = rs.getLong("sitemap_id");
                if (sitemapId == 0) {
                    if (rs.wasNull()) {
                        sitemapId = null;
                    }
                }

                int sitemapSourceId = rs.getInt("sitemap_source");
                SitemapSourceEnum sitemapSource = SitemapSourceEnum.R.fromValueOrNull(sitemapSourceId);

                String hostName = rs.getString("host_name");
                String urlName = rs.getString("url_name");

                return new SitemapLink(new SitemapId(sitemapId, sitemapSource), hostName + urlName);
            }
        };
    }

    private static ParameterizedRowMapper<MergedSitemapInfo> getSitemapInfoRowMapper() {
        return new ParameterizedRowMapper<MergedSitemapInfo>() {
            @Override
            public MergedSitemapInfo mapRow(ResultSet rs, int i) throws SQLException {
                long sitemapId = rs.getLong("sitemap_id");
                String hostName = rs.getString("host_name");
                String sitemapPath = rs.getString("url_name");

                Date submittedOn = rs.getTimestamp("submitted_on");
                Date processedOn = rs.getTimestamp("processed_on");
                Date realProcessedOn = rs.getTimestamp("real_processed_on");
                boolean fromRobotsTxt = rs.getBoolean("from_robots_txt");
                boolean fromUser = rs.getBoolean("from_user");

                Integer formatId = SqlUtil.getIntNullable(rs, "format");
                SitemapFormatEnum format = null;
                if (formatId != null) {
                    format = SitemapFormatEnum.getByValue(formatId);
                }

                SitemapTypeEnum type = null;
                Integer typeId = SqlUtil.getIntNullable(rs, "type");
                if (typeId != null) {
                    type = SitemapTypeEnum.getByValue(typeId);
                }

                Integer urlsCount = SqlUtil.getIntNullable(rs, "url_count");

                Integer urlErrorsCount = rs.getInt("url_error_count");
                Integer xmlErrorsCount = rs.getInt("xml_error_count");
                Integer warningsCount = rs.getInt("warning_count");

                if (submittedOn != null && submittedOn.getTime() <= 0) {
                    submittedOn = null;
                }

                if (processedOn == null) {
                    urlErrorsCount = null;
                }

                if (processedOn == null) {
                    xmlErrorsCount = null;
                }

                if (processedOn == null) {
                    warningsCount = null;
                }


                return new MergedSitemapInfo(
                        sitemapId,
                        null,
                        null,
                        hostName,
                        sitemapPath,
                        submittedOn,
                        fromRobotsTxt,
                        fromUser,
                        processedOn,
                        realProcessedOn,
                        format,
                        type,
                        urlsCount,
                        xmlErrorsCount,
                        urlErrorsCount,
                        warningsCount,
                        false,
                        SitemapSourceEnum.IN_SEARCH
                );
            }
        };
    }
}
