#include <util/draft/date.h>
#include <util/generic/size_literals.h>
#include <util/string/join.h>
#include <library/cpp/string_utils/url/url.h>

#include <kernel/hosts/owner/owner.h>

#include <robot/jupiter/protos/export.pb.h>
#include <robot/library/yt/static/command.h>
#include <robot/library/yt/static/tags.h>

#include <wmconsole/version3/library/jupiter/jupiter.h>
#include <wmconsole/version3/protos/queries2.pb.h>
#include <wmconsole/version3/wmcutil/log.h>
#include <wmconsole/version3/wmcutil/regex.h>
#include <wmconsole/version3/wmcutil/url.h>
#include <wmconsole/version3/wmcutil/yt/yt_utils.h>

#include <wmconsole/version3/junk/site-speed/tables.pb.h>

#include <yweb/antispam/procus/protos/hostspeed.pb.h>

using namespace NJupiter;

#define OPERATION_WEIGHT 20.0f

namespace NWebmaster {

const float TURBO_ENABLED_SHARE_THRESHOLD   = 0.1f;
const int TURBO_ENABLED_CLICKS_THRESHOLD    = 10;
const int CLICKS_WEEK_THRESHOLD             = 200;

const TString config_TABLE_SOURCE_HOSTSPEED_ANTISPAM = "//home/antispam/user_metrics/hostspeed";
//const TString config_TABLE_SOURCE_HOST2VEC = "//home/antispam/user_metrics/hostspeed";
const TString config_TABLE_SOURCE_B2BTOP_REPORT_ROOT = "//home/webmaster/prod/b2b-top/weekly/report/all/owners";
const TString config_TABLE_SOURCE_B2BTOP_CLICKS_ROOT = "//home/webmaster/prod/b2b-top/weekly/sources/clicks/hosts";
const TString config_TABLE_SITESPEED_ROOT = "//home/webmaster/users/lester/site-speed";
const TString config_TABLE_SITESPEED_HOST2VEC = NYTUtils::JoinPath(config_TABLE_SITESPEED_ROOT, "host2vec");
//const TString config_TABLE_SITESPEED_TRAFFIC = NYTUtils::JoinPath(config_TABLE_SITESPEED_ROOT, "traffic");
const TString config_TABLE_SITESPEED_HOSTSPEED = NYTUtils::JoinPath(config_TABLE_SITESPEED_ROOT, "hostspeed");
const TString config_TABLE_SITESPEED_REPORT = NYTUtils::JoinPath(config_TABLE_SITESPEED_ROOT, "report");
const TString config_MR_SERVER_B2B = "hahn.yt.yandex.net";

struct TOwners {
    TOwners() {
        OwnerCanonizer.LoadTrueOwners();
        OwnerCanonizer.LoadSerpCategOwners();
    }

    TString GetOwner(const TString &url) const {
        return TString{OwnerCanonizer.GetUrlOwner(url)};
    }

    static const TOwners& CInstance() {
        return *Singleton<TOwners>();
    }

public:
    TOwnerCanonizer OwnerCanonizer;
};

struct TTurboReport {
    Y_SAVELOAD_DEFINE(Clicks, TurboClicks)

    TTurboReport() = default;
    TTurboReport(int clicks, int turboClicks)
        : Clicks(clicks)
        , TurboClicks(turboClicks)
    {
    }

    float GetTurboShare() const {
        if (Clicks == 0) {
            return 0;
        }
        return Min(static_cast<float>(TurboClicks) / static_cast<float>(Clicks), 1.0f);
    }

    bool GetTurboConnected() const {
        return Clicks >= TURBO_ENABLED_CLICKS_THRESHOLD
            && GetTurboShare() >= TURBO_ENABLED_SHARE_THRESHOLD
        ;
    }

public:
    int Clicks = 0;
    int TurboClicks = 0;
};

struct TSpeedReport {
    Y_SAVELOAD_DEFINE(DesktopP90, MobileP90)

    TSpeedReport() = default;
    TSpeedReport(ui32 desktopP90, ui32 mobileP90)
        : DesktopP90(desktopP90)
        , MobileP90(mobileP90)
    {
    }

public:
    ui32 DesktopP90 = 0;
    ui32 MobileP90 = 0;
};

struct THostReport {
    Y_SAVELOAD_DEFINE(Clicks, Hosts, TurboEnabled, DesktopP90, MobileP90)

    THostReport() = default;
    THostReport(int clicks)
        : Clicks(clicks)
    {
    }

    TString Debug() const {
        return TStringBuilder()
            << "Clicks:" << Clicks << "\t"
            << "TurboEnabled:" << (int)TurboEnabled << "\t"
            << "DesktopP90:" << DesktopP90 << "\t"
            << "MobileP90:" << MobileP90 << "\t"
            << "Hosts:" << JoinSeq(",", Hosts);
    }

    bool IsSpeedValid() const {
        return DesktopP90 > 0
            && MobileP90;
    }

public:
    int Clicks = 0;
    TDeque<TString> Hosts;
    bool TurboEnabled = false;
    ui32 DesktopP90 = 0;
    ui32 MobileP90 = 0;
};

struct TParseSessionMapper : public NYT::IMapper<NYT::TTableReader<NProcus::TSpeedFeatures>, NYT::TTableWriter<NProto::THostSpeed>> {
    void Do(TReader *input, TWriter *output) override {
        NProto::THostSpeed dstMsg;
        for (; input->IsValid(); input->Next()) {
            auto &row = input->GetRow();
            NProcus::EDeviceType deviceType = row.GetDeviceType();
            if (row.GetEventType() == 3178) { // https://stat.yandex-team.ru/DictionaryEditor/BlockStat
                dstMsg.SetHost(TString{NUtils::GetHost2vecDomain(row.GetName())});
                if (deviceType == NProcus::MOBILE) {
                    dstMsg.SetMobileP90(row.GetPerc90());
                } else if (deviceType == NProcus::DESKTOP) {
                    dstMsg.SetDesktopP90(row.GetPerc90());
                } else {
                    ythrow yexception() << "unknown device type: " << static_cast<int>(deviceType);
                }
                output->AddRow(dstMsg);
            }
        }
    }

public:
};

REGISTER_MAPPER(TParseSessionMapper)

//ReduceBy Host
struct TParseSessionReducer : public NYT::IReducer<NYT::TTableReader<NProto::THostSpeed>, NYT::TTableWriter<NProto::THostSpeed>> {
    void Do(TReader *input, TWriter *output) override {
        NProto::THostSpeed dstMsg;
        dstMsg.SetHost(input->GetRow().GetHost());
        TSet<ui32> desktop, mobile;
        for (; input->IsValid(); input->Next()) {
            auto &row = input->GetRow();
            if (row.HasMobileP90()) {
                mobile.insert(row.GetMobileP90());
            }
            if (row.HasDesktopP90()) {
                desktop.insert(row.GetDesktopP90());
            }
        }

        if (!desktop.empty()) {
            dstMsg.SetDesktopP90(*desktop.begin());
            dstMsg.SetP90(*desktop.begin());
        }

        if (!mobile.empty()) {
            dstMsg.SetMobileP90(*mobile.begin());
            dstMsg.SetP90(*mobile.begin());
        }

        output->AddRow(dstMsg);
    }
};

REGISTER_REDUCER(TParseSessionReducer)

//ReduceBy Group
struct TReducer2 : public NYT::IReducer<NYT::TTableReader<NProto::THost2vec>, NYT::TTableWriter<NProto::TReport>> {
    Y_SAVELOAD_JOB(HostReport)

    TReducer2() = default;
    TReducer2(const THashMap<TString, THostReport> &hostReport)
        : HostReport(hostReport)
    {
    }

    void Do(TReader *input, TWriter *output) override {
        const TString host = input->GetRow().GetGroup();
        if (!HostReport.contains(host)) {
            return;
        }

        const THostReport &hostReport = HostReport.at(host);
        if (!hostReport.IsSpeedValid()) {
            return;
        }

        int turboRivals = 0;
        TSet<std::pair<ui32, TString>> desktopRankSet, mobileRankSet;
        desktopRankSet.insert(std::make_pair(hostReport.DesktopP90, host));
        mobileRankSet.insert(std::make_pair(hostReport.MobileP90, host));
        THashSet<TString> processedRivals;
        for (; input->IsValid(); input->Next()) {
            const auto &row = input->GetRow();
            const TString &rivalHost = row.GetHost();
            if (processedRivals.contains(rivalHost)) {
                continue;
            }
            processedRivals.insert(rivalHost);

            if (!HostReport.contains(rivalHost)) {
                continue;
            }

            const THostReport &rivalReport = HostReport.at(rivalHost);
            if (rivalReport.Clicks < CLICKS_WEEK_THRESHOLD) {
                continue;
            }

            if (host == rivalHost) {
                continue;
            }

            if (rivalReport.TurboEnabled) {
                turboRivals++;
            }

            if (rivalReport.DesktopP90 > 0) {
                desktopRankSet.insert(std::make_pair(rivalReport.DesktopP90, rivalHost));
            }

            if (rivalReport.MobileP90 > 0) {
                mobileRankSet.insert(std::make_pair(rivalReport.MobileP90, rivalHost));
            }
        }

        int desktopRank = 0;
        for (const auto &obj : desktopRankSet) {
            desktopRank++;
            if (obj.second == host) {
                break;
            }
        }

        int mobileRank = 0;
        for (const auto &obj : mobileRankSet) {
            mobileRank++;
            if (obj.second == host) {
                break;
            }
        }

        NProto::TReport dstMsg;
        for (const TString &hostWithScheme : hostReport.Hosts) {
            dstMsg.SetHost(hostWithScheme);
            dstMsg.SetClicks(hostReport.Clicks);
            dstMsg.SetHostInTurbo(hostReport.TurboEnabled);
            dstMsg.SetRivalsInTurbo(turboRivals);
            dstMsg.SetDesktopRank(desktopRank);
            dstMsg.SetMobileRank(mobileRank);
            output->AddRow(dstMsg);
        }
    }

public:
    THashMap<TString, THostReport> HostReport;
};

REGISTER_REDUCER(TReducer2)

TString GetLatestTableByRegex(NYT::IClientBasePtr client, const TString &prefix, const TString &regexStr) {
    TDeque<NYTUtils::TTableInfo> tableList;
    NYTUtils::GetTableList(client, prefix, tableList, Max<size_t>());
    std::sort(tableList.rbegin(), tableList.rend(), NYTUtils::TTableInfo::TNameLess());

    TString latestTable;
    TRegularExpression regex(regexStr);
    for (const NYTUtils::TTableInfo &table : tableList) {
        TVector<TString> hits;
        if (regex.GetMatches(NYTUtils::GetTableName(table.Name), hits) == 1) {
            return table.Name;
        }
    }

    ythrow yexception() << "there is no valid table: " << prefix;
}

void LoadConnectedTurboReport(NYT::IClientPtr b2bClient, THashMap<TString, TTurboReport> &ownerTurboConnected) {
    const TString latestReportTable = GetLatestTableByRegex(
        b2bClient, config_TABLE_SOURCE_B2BTOP_REPORT_ROOT, "^(\\d+_\\d+)$"
    );

    auto reader = TTable<NProto::TB2bTopReport>(b2bClient, latestReportTable).SelectFields(
        {"Name", "Clicks", "TurboClicks"}
    ).GetReader();

    for (; reader->IsValid(); reader->Next()) {
        auto &row = reader->GetRow();
        if (row.GetClicks() > 0) {
            TTurboReport report(row.GetClicks(), row.GetTurboClicks());
            if (report.GetTurboConnected()) {
                ownerTurboConnected.emplace(std::make_pair(row.GetName(), report));
            }
        }
    }
}

void LoadSpeedReport(NYT::IClientBasePtr client, THashMap<TString, TSpeedReport> &speedReport) {
    auto reader = TTable<NProto::THostSpeed>(client, config_TABLE_SITESPEED_HOSTSPEED).GetReader();
    for (; reader->IsValid(); reader->Next()) {
        auto &row = reader->GetRow();
        TSpeedReport &report = speedReport[row.GetHost()];
        report.DesktopP90 = row.GetDesktopP90();
        report.MobileP90 = row.GetMobileP90();
    }
}

void LoadHostReport(NYT::IClientBasePtr b2bClient, const THashMap<TString, TTurboReport> &ownerTurboConnected,
    const THashMap<TString, TSpeedReport> &speedReport, THashMap<TString, THostReport> &hostClicks)
{
    const TString latestClicksTable = GetLatestTableByRegex(
        b2bClient, config_TABLE_SOURCE_B2BTOP_CLICKS_ROOT, "^(\\d+_\\d+)$"
    );

    auto reader = TTable<NProto::TB2bTopReport>(b2bClient, latestClicksTable).SelectFields(
        {"Name", "Clicks"}
    ).GetReader();

    for (; reader->IsValid(); reader->Next()) {
        auto &row = reader->GetRow();
        if (row.GetClicks() > CLICKS_WEEK_THRESHOLD) {
            const TString h2vHost = TString{NUtils::GetHost2vecDomain(row.GetName())};
            THostReport &report = hostClicks[h2vHost];
            report.Clicks += row.GetClicks();
            report.Hosts.push_back(row.GetName());
            const TString owner = TOwners::CInstance().GetOwner(row.GetName());
            if (ownerTurboConnected.contains(owner)) {
                report.TurboEnabled = true;
            }
            if (speedReport.contains(h2vHost)) {
                report.DesktopP90 = speedReport.at(h2vHost).DesktopP90;
                report.MobileP90 = speedReport.at(h2vHost).MobileP90;
            }
        }
    }
}

} //namespace NWebmaster

int main(int argc, const char **argv) {
    setenv("YT_POOL", "robot-webmaster", 1);

    NYT::Initialize(argc, argv);
    using namespace NWebmaster;

    NYT::IClientPtr b2bClient = NYT::CreateClient(config_MR_SERVER_B2B);

    NYT::IClientPtr client = NYT::CreateClient("arnold.yt.yandex.net");
    NYT::ITransactionPtr tx = client->StartTransaction();

    TMapCombineReduceCmd<TParseSessionMapper, TParseSessionReducer, TParseSessionReducer>(
        tx, new TParseSessionMapper, nullptr, new TParseSessionReducer
    )
        .OperationWeight(OPERATION_WEIGHT)
        .Input<NProcus::TSpeedFeatures>(config_TABLE_SOURCE_HOSTSPEED_ANTISPAM)
        .Output(TTable<NProto::THostSpeed>(tx, config_TABLE_SITESPEED_HOSTSPEED))
        .ReduceBy({"Host"})
        .Do()
    ;

    TSortCmd<NProto::THostSpeed>(tx, TTable<NProto::THostSpeed>(tx, config_TABLE_SITESPEED_HOSTSPEED))
        .OperationWeight(OPERATION_WEIGHT)
        .By({"Host"})
        .Do()
    ;

    THashMap<TString, TTurboReport> ownerTurboReport;
    LoadConnectedTurboReport(b2bClient, ownerTurboReport);

    THashMap<TString, TSpeedReport> speedReport;
    LoadSpeedReport(tx, speedReport);

    THashMap<TString, THostReport> hostReport;
    LoadHostReport(b2bClient, ownerTurboReport, speedReport, hostReport);

    TReduceCmd<TReducer2>(tx, new TReducer2(hostReport))
        .MemoryLimit(3_GBs)
        .OperationWeight(OPERATION_WEIGHT)
        .Input(TTable<NProto::THost2vec>(tx, config_TABLE_SITESPEED_HOST2VEC))
        .Output(TTable<NProto::TReport>(tx, config_TABLE_SITESPEED_REPORT))
        .ReduceBy({"Group"})
        .Do()
    ;

    TSortCmd<NProto::TReport>(tx, TTable<NProto::TReport>(tx, config_TABLE_SITESPEED_REPORT))
        .OperationWeight(OPERATION_WEIGHT)
        .By({"Host"})
        .Do()
    ;

    tx->Commit();
}
