#include <util/string/join.h>
#include <util/string/printf.h>
#include <robot/library/yt/static/command.h>

#include <mapreduce/yt/interface/fwd.h>

#include <wmconsole/version3/wmcutil/log.h>

#include "common.h"

namespace NWebmaster {
namespace NAcceptance {

void HostinfoLog(const TString &msg, const TString &source) {
    LOG_INFO(("hostinfo, " + msg).c_str(), source.c_str());
}

void HostinfoLogMap(const THashMap<TString, NYT::TNode> &mp, const TString &name, NYT::TTableWriterPtr<NYT::TNode> writer) {
    writer->AddRow(NYT::TNode()("time", Now().TimeT())("name", name)("log", mp));
}

bool IsThresholdBroken(const TString &name,
                        TDeque<double> &diffs,
                        const TMap<double, TString> &samples,
                        double percentile,
                        double threshold) {
    Sort(diffs);
    const double value = diffs.at(diffs.size() * percentile);
    const bool accepted = value <= threshold;
    const TString statusMessage = Sprintf("%s(p%.0f) %.2f <= %.2f %s", name.c_str(), percentile * 100.0, value, threshold, (accepted ? "OK" : "FAIL"));
    if (accepted) {
        LOG_INFO("%s", statusMessage.c_str());
    } else {
        const static TDeque<double> PERCENTILES = {
            0.1,  0.2,  0.3,  0.4,  0.5,  0.6,  0.7,  0.8,  0.9,
            0.91, 0.92, 0.93, 0.94, 0.95, 0.96, 0.97, 0.98, 0.99,
            1.0
        };
        TDeque<TString> values;
        for (double p : PERCENTILES) {
            const double value = diffs.at(Min<size_t>(diffs.size() * p, diffs.size() - 1));
            values.push_back(Sprintf("p%.0f=%.2f", p * 100.0, value));
        }
        LOG_ERROR("%s", statusMessage.c_str());
        LOG_ERROR("%s [%s]", name.c_str(), JoinSeq(" ", values).c_str());
        LOG_ERROR("%s is rejected", name.c_str());

        if (!samples.empty()) {
            LOG_ERROR("Top samples:");
            for (const auto &sample : samples) {
                LOG_ERROR("  %.2f %s", sample.first, sample.second.c_str());
            }
        }

        return true;
    }
    return false;
}

void AddSample(TDeque<double> &diffs, TMap<double, TString> &samples, double diff, const TString &host, size_t keepSamples) {
    diffs.push_back(std::abs(diff));
    samples[std::abs(diff)] = (diff >= 0 ? "+ " : "- ") + host;
    while (samples.size() > keepSamples) {
        samples.erase(samples.begin()->first);
    }
}

bool ParseDate(NYT::TNode node, time_t &time) {
    auto tmp = node.AsString();
    return ParseISO8601DateTime(tmp.c_str(), tmp.length(), time);
}

}
} //namespace NWebmaster
