#include <util/draft/date.h>
#include <util/random/random.h>
#include <util/string/join.h>
#include <util/string/printf.h>

#include <mapreduce/yt/interface/protos/yamr.pb.h>

#include <robot/library/yt/static/command.h>
#include <robot/library/yt/static/table.h>
#include <robot/library/yt/static/tags.h>

#include <wmconsole/version3/library/jupiter/jupiter.h>
#include <wmconsole/version3/searchqueries-mr/conf/yt.h>
#include <wmconsole/version3/searchqueries-mr/protos/user_sessions.pb.h>
#include <wmconsole/version3/processors/acceptance/conf/config.h>
#include <wmconsole/version3/processors/acceptance/protos/acceptance.pb.h>
#include <wmconsole/version3/processors/indexing/hostinfo/conf/config.h>
#include <wmconsole/version3/processors/indexing/hostinfo/protos/hostinfo.pb.h>
#include <wmconsole/version3/wmcutil/log.h>
#include <wmconsole/version3/wmcutil/yt/triggers.h>

#include "common.h"
#include "host_regions_common.h"

namespace NWebmaster {
namespace NAcceptance {

using namespace NJupiter;

TInputTag<::NWebmaster::NProto::THostRegions> HostRegionsPrevInputTag              (1);
TInputTag<::NWebmaster::NProto::THostRegions> HostRegionsNewInputTag               (2);

TOutputTag<NProto::THostRegionsDiff> HostRegionsOutputTag                          (1);

struct THostRegionsDiffCounter {
    static double GetDiffMainHost(const ::NWebmaster::NProto::THostRegions &pr, const ::NWebmaster::NProto::THostRegions &nr) {
        return static_cast<double>(
            static_cast<TString>(pr.GetMainHost()) != static_cast<TString>(nr.GetMainHost())
            );
    }

    struct GetDictionaryRegionsVector {
        inline TVector<i64> operator()(const ::NWebmaster::NProto::THostRegions &msg) const {
            auto v = msg.GetDictionaryRegions();
            TVector<i64> res(v.size());
            Transform(v.begin(), v.end(), res.begin(), [](google::protobuf::int64 a) { return static_cast<i64>(a); });
            return res;
        }
    };

    struct GetWebmasterRegionsVector {
        inline TVector<i64> operator()(const ::NWebmaster::NProto::THostRegions &msg) const {
            auto v = msg.GetWebmasterRegions();
            TVector<i64> res(v.size());
            Transform(v.begin(), v.end(), res.begin(), [](google::protobuf::int64 a) { return static_cast<i64>(a); });
            return res;
        }
    };

    template<class TGetter>
    static double GetDifference(const ::NWebmaster::NProto::THostRegions &pr, const ::NWebmaster::NProto::THostRegions &nr) {
        const static TGetter getter;
        TVector<i64> prevVec = getter(pr);
        TVector<i64> newVec = getter(nr);
        // TODO? more complicated logic
        double prevSz = prevVec.size();
        double newSz = newVec.size();
        if (prevSz == 0) return prevSz != newSz;
        return (newSz - prevSz) / prevSz;
    }

    void Set(const ::NWebmaster::NProto::THostRegions &pr,
                                const ::NWebmaster::NProto::THostRegions &nr) {
        DiffMainHost            = GetDiffMainHost(pr, nr);
        DiffDictionaryRegions   = GetDifference<GetDictionaryRegionsVector>(pr, nr);
        DiffWebmasterRegions    = GetDifference<GetWebmasterRegionsVector>(pr, nr);
    }

public:
    double DiffMainHost             = 0;
    double DiffDictionaryRegions    = 0;
    double DiffWebmasterRegions     = 0;
};

struct THostRegionsDiffReducer : public TTaggedReducer {
    THostRegionsDiffReducer() = default;
    THostRegionsDiffReducer(time_t timestamp)
        : Timestamp(timestamp)
    {
    }

    void Save(IOutputStream& stream) const override {
        ::Save(&stream, Timestamp);
        TTaggedReducer::Save(stream);
    }

    void Load(IInputStream& stream) override {
        ::Load(&stream, Timestamp);
        TTaggedReducer::Load(stream);
    }

    void DoTagged(TTagedReader reader, TTagedWriter writer) final {
        const TMaybe<::NWebmaster::NProto::THostRegions> mbPrev = reader.GetSingleRowMaybe(HostRegionsPrevInputTag);
        const TMaybe<::NWebmaster::NProto::THostRegions> mbNew  = reader.GetSingleRowMaybe(HostRegionsNewInputTag);

        TString host;

        THostRegionsDiffCounter dc;
        ::NWebmaster::NProto::THostRegions prevRow, newRow;

        if (mbPrev.Defined()) {
            prevRow = mbPrev.GetRef();
            host = mbPrev->GetHost();
        } else {
            host = mbNew->GetHost();
        }

        if (mbNew.Defined()) {
            newRow = mbNew.GetRef();
        }

        dc.Set(prevRow, newRow);

        NProto::THostRegionsDiff dstMsg;
        dstMsg.SetHost(host);
        dstMsg.SetDiffDictionaryRegions(dc.DiffDictionaryRegions);
        dstMsg.SetDiffMainHost(dc.DiffMainHost);
        dstMsg.SetDiffWebmasterRegions(dc.DiffWebmasterRegions);
        dstMsg.SetTimestamp(Timestamp);
        writer.AddRow(dstMsg, HostRegionsOutputTag);
    }

public:
    time_t Timestamp = 0;
};
REGISTER_REDUCER(THostRegionsDiffReducer)

struct THostRegionsArchiveLimitReducer :
        public NYT::IReducer<NYT::TTableReader<NProto::THostRegionsDiff>, NYT::TTableWriter<NProto::THostRegionsDiff>> {
    void Do(TReader *input, TWriter *output) override {
        TMap<time_t, NProto::THostRegionsDiff> rows;
        for (; input->IsValid(); input->Next()) {
            const auto &row = input->GetRow();
            rows[row.GetTimestamp()] = row;
            if (rows.size() > 100) {
                rows.erase(rows.begin()->first);
            }
        }
        for (const auto &obj : rows) {
            output->AddRow(obj.second);
        }
    }
};
REGISTER_REDUCER(THostRegionsArchiveLimitReducer)

void ValidateHostRegionsDiffs(NYT::IClientBasePtr client) {
    const auto &cfg = TConfig::CInstance();

    bool rejected = false;

    TDeque<double> diffMainHost;
    TDeque<double> diffDictionaryRegions;
    TDeque<double> diffWebmasterRegions;

    TMap<double, TString> diffMainHostSamples;
    TMap<double, TString> diffDictionaryRegionsSamples;
    TMap<double, TString> diffWebmasterRegionsSamples;

    auto reader = TTable<NProto::THostRegionsDiff>(client, cfg.TABLE_ACCEPTANCE_HOSTINFO_REGIONS).GetReader();
    for (; reader->IsValid(); reader->Next()) {
        const auto &row = reader->GetRow();
        AddSample(diffMainHost,             diffMainHostSamples,            row.GetDiffMainHost(),            row.GetHost());
        AddSample(diffDictionaryRegions,    diffDictionaryRegionsSamples,   row.GetDiffDictionaryRegions(),   row.GetHost());
        AddSample(diffWebmasterRegions,     diffWebmasterRegionsSamples,    row.GetDiffWebmasterRegions(),    row.GetHost());
    }

    rejected |= IsThresholdBroken("ValidateHostRegionsDiffs diffMainHost",
            diffMainHost,           diffMainHostSamples,            0.9, 0);
    rejected |= IsThresholdBroken("ValidateHostRegionsDiffs diffDictionaryRegions",
            diffDictionaryRegions,  diffDictionaryRegionsSamples,   0.9, 0.1);
    rejected |= IsThresholdBroken("ValidateHostRegionsDiffs diffWebmasterRegions",
            diffWebmasterRegions,   diffWebmasterRegionsSamples,    0.0, 0.1);

    if (rejected) {
        ythrow yexception() << "HostRegions are rejected";
    }
}

void AcceptHostRegions(NYT::ITransactionPtr tx) {
    const static TString functionName = "AcceptHostRegions, ";

    const auto &cfg = TConfig::CInstance();
    const auto &ccfg = TCommonYTConfig::CInstance();
    const auto &hicfg = NHostInfo::TConfig::CInstance();

    TYtModificationTimeTrigger newStateTrigger(hicfg.TABLE_EXPORT_HOST_REGIONS);
    TYtModificationTimeTrigger oldStateTrigger(cfg.TABLE_ACCEPTANCE_HOSTINFO_REGIONS);

    if (!(oldStateTrigger.NeedUpdate(tx, "dictionary_regions", hicfg.TABLE_EXPORT_HOST_REGIONS) ||
          oldStateTrigger.NeedUpdate(tx, "webmaster_regions", hicfg.TABLE_EXPORT_HOST_REGIONS))) {
        HostinfoLog(functionName + "table %s is already processed", hicfg.TABLE_EXPORT_HOST_REGIONS);
        return;
    }

    const TString stageInputTable = hicfg.TABLE_EXPORT_HOST_REGIONS;
    const TString acceptedInputTable = ccfg.GetAcceptedPath(hicfg.TABLE_EXPORT_HOST_REGIONS);

    HostinfoLog(functionName + "input %s", acceptedInputTable);
    HostinfoLog(functionName + "input %s", stageInputTable);
    HostinfoLog(functionName + "output %s", oldStateTrigger.TargetTable);

    time_t timeStamp = 0;
    ParseDate(GetYtAttr(tx, hicfg.TABLE_EXPORT_HOST_REGIONS, "modification_time"), timeStamp);

    if (!tx->Exists(acceptedInputTable)) {
        const auto opts = NYT::TCopyOptions().Recursive(true);
        tx->Copy(stageInputTable, acceptedInputTable, opts);
        HostinfoLog(functionName + "first launch, created %s", acceptedInputTable);
    }

    TReduceCmd<THostRegionsDiffReducer>(tx, new THostRegionsDiffReducer(timeStamp))
        .Input(TTable<::NWebmaster::NProto::THostRegions>(tx, acceptedInputTable), HostRegionsPrevInputTag)
        .Input(TTable<::NWebmaster::NProto::THostRegions>(tx, stageInputTable), HostRegionsNewInputTag)
        .Output(TTable<NProto::THostRegionsDiff>(tx, cfg.TABLE_ACCEPTANCE_HOSTINFO_REGIONS)
            .AsSortedOutput({"Host", "Timestamp"}), HostRegionsOutputTag
        )
        .ReduceBy({"Host"})
        .Do()
    ;

    if (!tx->Exists(cfg.TABLE_ACCEPTANCE_HOSTINFO_REGIONS_ARCHIVE)) {
        tx->Copy(TTable<NProto::THostRegionsDiff>(tx, cfg.TABLE_ACCEPTANCE_HOSTINFO_REGIONS), TTable<NProto::THostRegionsDiff>(tx, cfg.TABLE_ACCEPTANCE_HOSTINFO_REGIONS_ARCHIVE));
    }

    TReduceCmd<THostRegionsArchiveLimitReducer>(tx)
        .Input(TTable<NProto::THostRegionsDiff>(tx, cfg.TABLE_ACCEPTANCE_HOSTINFO_REGIONS_ARCHIVE))
        .Input(TTable<NProto::THostRegionsDiff>(tx, cfg.TABLE_ACCEPTANCE_HOSTINFO_REGIONS))
        .Output(TTable<NProto::THostRegionsDiff>(tx, cfg.TABLE_ACCEPTANCE_HOSTINFO_REGIONS_ARCHIVE)
            .AsSortedOutput({"Host", "Timestamp"})
        )
        .SortBy({"Host", "Timestamp"})
        .ReduceBy({"Host"})
        .Do()
    ;

    oldStateTrigger.Update(tx, "dictionary_regions", hicfg.TABLE_EXPORT_HOST_REGIONS);
    oldStateTrigger.Update(tx, "webmaster_regions", hicfg.TABLE_EXPORT_HOST_REGIONS);
    ValidateHostRegionsDiffs(tx);

    const auto opts = NYT::TCopyOptions().Force(true);
    tx->Copy(stageInputTable, acceptedInputTable, opts);
    HostinfoLog(functionName + "success%s");
}

} //namespace NAcceptance
} //namespace NWebmaster
