#include <util/draft/date.h>
#include <util/random/random.h>
#include <util/string/join.h>
#include <util/string/printf.h>

#include <mapreduce/yt/interface/protos/yamr.pb.h>

#include <robot/library/yt/static/command.h>
#include <robot/library/yt/static/table.h>
#include <robot/library/yt/static/tags.h>

#include <wmconsole/version3/library/jupiter/jupiter.h>
#include <wmconsole/version3/searchqueries-mr/conf/yt.h>
#include <wmconsole/version3/searchqueries-mr/protos/user_sessions.pb.h>
#include <wmconsole/version3/processors/acceptance/conf/config.h>
#include <wmconsole/version3/processors/acceptance/protos/acceptance.pb.h>
#include <wmconsole/version3/processors/indexing/hostinfo/conf/config.h>
#include <wmconsole/version3/processors/indexing/hostinfo/protos/hostinfo.pb.h>
#include <wmconsole/version3/wmcutil/log.h>
#include <wmconsole/version3/wmcutil/yt/triggers.h>

#include "common.h"
#include "host_statistics_common.h"

namespace NWebmaster {
namespace NAcceptance {

using namespace NJupiter;

TInputTag<::NWebmaster::NProto::THostStatistics> HostStatisticsPrevInputTag      (1);
TInputTag<::NWebmaster::NProto::THostStatistics> HostStatisticsNewInputTag       (2);

TOutputTag<NProto::THostStatisticsDiff> HostStatisticsOutputTag                  (1);

struct THostStatisticsDiffCounter {
    struct TAverageFetchTimeGetter {
        inline double operator()(const ::NWebmaster::NProto::THostStatistics &msg) const {
            if (msg.GetPagesWithFetchTime() > 0) {
                return static_cast<double>(msg.GetTotalFetchTime()) / static_cast<double>(msg.GetPagesWithFetchTime());
            } else {
                return 0;
            }
        }
    };

    struct TNumOfDocsGetter {
        inline double operator()(const ::NWebmaster::NProto::THostStatistics &msg) const {
            return static_cast<double>(msg.GetDocs());
        }
    };

    struct TNumOfDocsEmptyMetaDescrGetter {
        inline double operator()(const ::NWebmaster::NProto::THostStatistics &msg) const {
            return static_cast<double>(msg.GetEmptyDescriptions());
        }
    };

    struct TNumOfDocsEmptyTitleGetter {
        inline double operator()(const ::NWebmaster::NProto::THostStatistics &msg) const {
            return static_cast<double>(msg.GetEmptyTitles());
        }
    };

    struct TNumOfDocsEncodingUnknownGetter {
        inline double operator()(const ::NWebmaster::NProto::THostStatistics &msg) const {
            // TODO
            Y_UNUSED(msg);
            return 0;
        }
    };

    struct TNumOfDocsEncodingUnsupportedGetter {
        inline double operator()(const ::NWebmaster::NProto::THostStatistics &msg) const {
            // TODO
            Y_UNUSED(msg);
            return 0;
        }
    };

    struct TNumOfDocsHtmlGetter {
        inline double operator()(const ::NWebmaster::NProto::THostStatistics &msg) const {
            return static_cast<double>(msg.GetHtmlDocs());
        }
    };

    struct TNumOfDocsOnSearchGetter {
        inline double operator()(const ::NWebmaster::NProto::THostStatistics &msg) const {
            return static_cast<double>(msg.GetDocsOnSearch());
        }
    };

    template<class TGetter>
    static double GetDifference(const ::NWebmaster::NProto::THostStatistics &pr, const ::NWebmaster::NProto::THostStatistics &nr) {
        const static TGetter getter;
        const double pValue = getter(pr);
        const double nValue = getter(nr);
        if (pValue == 0) {
            if (nValue == 0) {
                return 0.0;
            }
            return 1.0;
        }

        return (nValue - pValue) / pValue;
    }

    void Set(const ::NWebmaster::NProto::THostStatistics &pr, const ::NWebmaster::NProto::THostStatistics &nr) {
        AverageFetchTime                = GetDifference<TAverageFetchTimeGetter>(pr, nr);
        NumOfDocs                       = GetDifference<TNumOfDocsGetter>(pr, nr);
        NumOfDocsEmptyMetaDescr         = GetDifference<TNumOfDocsEmptyMetaDescrGetter>(pr, nr);
        NumOfDocsEmptyTitle             = GetDifference<TNumOfDocsEmptyTitleGetter>(pr, nr);
        NumOfDocsEncodingUnknown        = GetDifference<TNumOfDocsEncodingUnknownGetter>(pr, nr);
        NumOfDocsEncodingUnsupported    = GetDifference<TNumOfDocsEncodingUnsupportedGetter>(pr, nr);
        NumOfDocsHtml                   = GetDifference<TNumOfDocsHtmlGetter>(pr, nr);
        NumOfDocsOnSearch               = GetDifference<TNumOfDocsOnSearchGetter>(pr, nr);
    }

public:
    double AverageFetchTime                 = 0;
    double NumOfDocs                        = 0;
    double NumOfDocsEmptyMetaDescr          = 0;
    double NumOfDocsEmptyTitle              = 0;
    double NumOfDocsEncodingUnknown         = 0;
    double NumOfDocsEncodingUnsupported     = 0;
    double NumOfDocsHtml                    = 0;
    double NumOfDocsOnSearch                = 0;
};

struct THostStatisticsDiffReducer : public TTaggedReducer {
    THostStatisticsDiffReducer() = default;
    THostStatisticsDiffReducer(time_t timestamp)
        : Timestamp(timestamp)
    {
    }

    void Save(IOutputStream& stream) const override {
        ::Save(&stream, Timestamp);
        TTaggedReducer::Save(stream);
    }

    void Load(IInputStream& stream) override {
        ::Load(&stream, Timestamp);
        TTaggedReducer::Load(stream);
    }

    void DoTagged(TTagedReader reader, TTagedWriter writer) final {
        const TMaybe<::NWebmaster::NProto::THostStatistics> mbPrev = reader.GetSingleRowMaybe(HostStatisticsPrevInputTag);
        const TMaybe<::NWebmaster::NProto::THostStatistics> mbNew  = reader.GetSingleRowMaybe(HostStatisticsNewInputTag);

        TString host;
        double diffCount = 0;

        //  difference calculator of previos state(s) and current state(s)
        THostStatisticsDiffCounter dc;
        ::NWebmaster::NProto::THostStatistics prevRow, newRow;

        if (mbPrev.Defined()) {
            prevRow = mbPrev.GetRef();
            host = mbPrev->GetHost();
            if (!mbNew.Defined()) {
                diffCount = -1;
            }
        } else {
            host = mbNew->GetHost();
            diffCount = 1;
        }

        if (mbNew.Defined()) {
            newRow = mbNew.GetRef();
        }

        dc.Set(prevRow, newRow);

        NProto::THostStatisticsDiff dstMsg;
        dstMsg.SetHost(host);
        dstMsg.SetDiffCount(diffCount);
        dstMsg.SetDiffAverageFetchTime(dc.AverageFetchTime);
        dstMsg.SetDiffNumOfDocs(dc.NumOfDocs);
        dstMsg.SetDiffNumOfDocsEmptyMetaDescr(dc.NumOfDocsEmptyMetaDescr);
        dstMsg.SetDiffNumOfDocsEmptyTitle(dc.NumOfDocsEmptyTitle);
        dstMsg.SetDiffNumOfDocsEncodingUnknown(dc.NumOfDocsEncodingUnknown);
        dstMsg.SetDiffNumOfDocsEncodingUnsupported(dc.NumOfDocsEncodingUnsupported);
        dstMsg.SetDiffNumOfDocsHtml(dc.NumOfDocsHtml);
        dstMsg.SetDiffNumOfDocsOnSearch(dc.NumOfDocsOnSearch);
        dstMsg.SetTimestamp(Timestamp);
        writer.AddRow(dstMsg, HostStatisticsOutputTag);
    }

public:
    time_t Timestamp = 0;
};
REGISTER_REDUCER(THostStatisticsDiffReducer)

struct THostStatisticsArchiveLimitReducer :
        public NYT::IReducer<NYT::TTableReader<NProto::THostStatisticsDiff>, NYT::TTableWriter<NProto::THostStatisticsDiff>> {
    void Do(TReader *input, TWriter *output) override {
        TMap<time_t, NProto::THostStatisticsDiff> rows;
        for (; input->IsValid(); input->Next()) {
            const auto &row = input->GetRow();
            rows[row.GetTimestamp()] = row;
            if (rows.size() > 100) {
                rows.erase(rows.begin()->first);
            }
        }
        for (const auto &obj : rows) {
            output->AddRow(obj.second);
        }
    }
};
REGISTER_REDUCER(THostStatisticsArchiveLimitReducer)

void ValidateHostStatiscticsDiffs(NYT::IClientBasePtr client) {
    const auto &cfg = TConfig::CInstance();

    TDeque<double> diffEmptyMetaDescription;
    TDeque<double> diffEmptyTitle;
    TDeque<double> diffAverageFetchTime;

    TMap<double, TString> diffEmptyMetaDescriptionSamples;
    TMap<double, TString> diffEmptyTitleSamples;
    TMap<double, TString> diffAverageFetchTimeSamples;

    auto reader = TTable<NProto::THostStatisticsDiff>(client, cfg.TABLE_ACCEPTANCE_HOSTINFO_STATISTICS).GetReader();
    for (; reader->IsValid(); reader->Next()) {
        const auto &row = reader->GetRow();
        AddSample(diffAverageFetchTime,     diffAverageFetchTimeSamples,        row.GetDiffAverageFetchTime(),          row.GetHost());
        AddSample(diffEmptyMetaDescription, diffEmptyMetaDescriptionSamples,    row.GetDiffNumOfDocsEmptyMetaDescr(),   row.GetHost());
        AddSample(diffEmptyTitle,           diffEmptyTitleSamples,              row.GetDiffNumOfDocsEmptyTitle(),       row.GetHost());
    }

    bool rejected = false;
    rejected |= IsThresholdBroken("ValidateHostStatiscticsDiffs diffAverageFetchTime",
            diffAverageFetchTime,       diffAverageFetchTimeSamples,        0.60, 0.20);
    rejected |= IsThresholdBroken("ValidateHostStatiscticsDiffs diffEmptyMetaDescription",
            diffEmptyMetaDescription,   diffEmptyMetaDescriptionSamples,    0.95, 0.15);
    rejected |= IsThresholdBroken("ValidateHostStatiscticsDiffs diffEmptyTitle",
            diffEmptyTitle,             diffAverageFetchTimeSamples,        0.95, 0.00);
    if (rejected) {
        ythrow yexception() << "HostStatistics are rejected";
    }
}

void AcceptHostStatistics(NYT::ITransactionPtr tx) {
    const TString functionName = "AcceptHostStatistics, ";

    const auto &cfg = TConfig::CInstance(); // local config of current process/acceptance
    const auto &ccfg = TCommonYTConfig::CInstance(); // includes everywhere
    const auto &hicfg = NHostInfo::TConfig::CInstance(); // prod-prod for indexing, part of indexing

    // triggers for checking that acceptance is needed
    TYtSourceTrigger newStateTrigger(tx, hicfg.TABLE_EXPORT_HOST_STATISTICS);
    TYtSourceTrigger oldStateTrigger(tx, cfg.TABLE_ACCEPTANCE_HOSTINFO_STATISTICS);

    if (!oldStateTrigger.NeedUpdate(newStateTrigger.Source)) {
        HostinfoLog(functionName + "source %s is already processed", oldStateTrigger.Source);
        return;
    }

    const TString stageInputTable = hicfg.TABLE_EXPORT_HOST_STATISTICS;
    const TString acceptedInputTable = ccfg.GetAcceptedPath(hicfg.TABLE_EXPORT_HOST_STATISTICS);

    HostinfoLog(functionName + "source %s", newStateTrigger.Source);
    HostinfoLog(functionName + "input %s", acceptedInputTable);
    HostinfoLog(functionName + "input %s", stageInputTable);
    HostinfoLog(functionName + "output %s", oldStateTrigger.TargetTable);

    if (!tx->Exists(acceptedInputTable)) {
        const auto opts = NYT::TCopyOptions().Recursive(true);
        tx->Copy(stageInputTable, acceptedInputTable, opts);
        HostinfoLog(functionName + "first launch, created %s", acceptedInputTable);
    }

    TReduceCmd<THostStatisticsDiffReducer>(tx, new THostStatisticsDiffReducer(GetJupiterTsTZFromPath(newStateTrigger.Source)))
        .Input(TTable<::NWebmaster::NProto::THostStatistics>(tx, acceptedInputTable), HostStatisticsPrevInputTag)
        .Input(TTable<::NWebmaster::NProto::THostStatistics>(tx, stageInputTable), HostStatisticsNewInputTag)
        .Output(TTable<NProto::THostStatisticsDiff>(tx, cfg.TABLE_ACCEPTANCE_HOSTINFO_STATISTICS)
            .AsSortedOutput({"Host", "Timestamp"}), HostStatisticsOutputTag
        )
        .ReduceBy({"Host"})
        .Do()
    ;

    if (!tx->Exists(cfg.TABLE_ACCEPTANCE_HOSTINFO_ARCHIVE))
        tx->Copy(TTable<NProto::THostStatisticsDiff>(tx, cfg.TABLE_ACCEPTANCE_HOSTINFO_STATISTICS), TTable<NProto::THostStatisticsDiff>(tx, cfg.TABLE_ACCEPTANCE_HOSTINFO_ARCHIVE), NYT::TCopyOptions().Recursive(true));

    // saving for debug / history purposes
    TReduceCmd<THostStatisticsArchiveLimitReducer>(tx)
        .Input(TTable<NProto::THostStatisticsDiff>(tx, cfg.TABLE_ACCEPTANCE_HOSTINFO_ARCHIVE))
        .Input(TTable<NProto::THostStatisticsDiff>(tx, cfg.TABLE_ACCEPTANCE_HOSTINFO_STATISTICS))
        .Output(TTable<NProto::THostStatisticsDiff>(tx, cfg.TABLE_ACCEPTANCE_HOSTINFO_ARCHIVE)
            .AsSortedOutput({"Host", "Timestamp"})
        )
        .SortBy({"Host", "Timestamp"})
        .ReduceBy({"Host"})
        .Do()
    ;

    oldStateTrigger.Update(tx, newStateTrigger.Source);
    ValidateHostStatiscticsDiffs(tx);

    const auto opts = NYT::TCopyOptions().Force(true);
    tx->Copy(stageInputTable, acceptedInputTable, opts);
    HostinfoLog(functionName + "success%s");
}

} //namespace NAcceptance
} //namespace NWebmaster
