#include <util/draft/date.h>
#include <util/random/random.h>
#include <util/string/join.h>
#include <util/string/printf.h>

#include <mapreduce/yt/interface/protos/yamr.pb.h>

#include <robot/library/yt/static/command.h>
#include <robot/library/yt/static/table.h>
#include <robot/library/yt/static/tags.h>

#include <wmconsole/version3/processors/indexing/robots/disallowed_urls/protos/disallowed_urls.pb.h>
#include <wmconsole/version3/library/jupiter/jupiter.h>
#include <wmconsole/version3/processors/acceptance/protos/acceptance.pb.h>
#include <wmconsole/version3/searchqueries-mr/conf/yt.h>
#include <wmconsole/version3/processors/acceptance/conf/config.h>
#include <wmconsole/version3/processors/indexing/hostinfo/conf/config.h>
#include <wmconsole/version3/wmcutil/log.h>
#include <wmconsole/version3/wmcutil/yt/triggers.h>

#include "task_disallowed_urls.h"
#include "common.h"

namespace NWebmaster {
namespace NAcceptance {

using namespace NJupiter;

TInputTag<::NWebmaster::NProto::TMeasuredHost> DisallowedUrlsPrevInputTag              (1);
TInputTag<::NWebmaster::NProto::TMeasuredHost> DisallowedUrlsNewInputTag               (2);

TOutputTag<NProto::TDisallowedUrlsDiff> DisallowedUrlsOutputTag                        (1);

struct TDisallowedUrlsDiffCounter {
    struct TDiffSamplesGetter {
        inline double operator()(const ::NWebmaster::NProto::TMeasuredHost &msg) const {
            return msg.GetUrls().size();
        }
    };

    struct TDiffAverageClicksGetter {
        inline double operator()(const ::NWebmaster::NProto::TMeasuredHost &msg) const {
            if (msg.GetUrls().empty()) return 0.0;
            double sum = 0.0;
            for (const auto &url: msg.GetUrls()) {
                sum += url.GetClicks();
            }
            return sum / msg.GetUrls().size();
        }
    };

    template<class TGetter>
    static double GetDifference(const ::NWebmaster::NProto::TMeasuredHost &pr, const ::NWebmaster::NProto::TMeasuredHost &nr) {
        const static TGetter getter;
        const double pValue = getter(pr);
        const double nValue = getter(nr);
        if (pValue == 0) {
            if (nValue == 0) {
                return 0.0;
            }
            return 1.0;
        }
        return (nValue - pValue) / pValue;
    }

    void Set(const ::NWebmaster::NProto::TMeasuredHost &pr, const ::NWebmaster::NProto::TMeasuredHost  &nr) {
        DiffSamplesSizes       = GetDifference<TDiffSamplesGetter>(pr, nr);
        DiffAverageClicks = GetDifference<TDiffAverageClicksGetter>(pr, nr);
    }

public:
    double DiffSamplesSizes  = 0;
    double DiffAverageClicks = 0;
};

struct TDisallowedUrlsDiffReducer : public TTaggedReducer {
    TDisallowedUrlsDiffReducer() = default;
    TDisallowedUrlsDiffReducer(TString state)
        : State(state)
    {
    }

    void Save(IOutputStream& stream) const override {
        ::Save(&stream, State);
        TTaggedReducer::Save(stream);
    }

    void Load(IInputStream& stream) override {
        ::Load(&stream, State);
        TTaggedReducer::Load(stream);
    }

    void DoTagged(TTagedReader reader, TTagedWriter writer) final {
        const TMaybe<::NWebmaster::NProto::TMeasuredHost> mbPrev = reader.GetSingleRowMaybe(DisallowedUrlsPrevInputTag);
        const TMaybe<::NWebmaster::NProto::TMeasuredHost> mbNew  = reader.GetSingleRowMaybe(DisallowedUrlsNewInputTag);

        TString host;
        TDisallowedUrlsDiffCounter dc;
        ::NWebmaster::NProto::TMeasuredHost prevRow, newRow;

        if (mbPrev.Defined()) {
            prevRow = mbPrev.GetRef();
            host = mbPrev->GetHost();
        } else {
            host = mbNew->GetHost();
        }

        if (mbNew.Defined()) {
            newRow = mbNew.GetRef();
        }

        dc.Set(prevRow, newRow);

        NProto::TDisallowedUrlsDiff dstMsg;
        dstMsg.SetHost(host);
        dstMsg.SetState(State);
        dstMsg.SetDiffAverageClicks(dc.DiffAverageClicks);
        dstMsg.SetDiffSamplesSizes(dc.DiffSamplesSizes);

        writer.AddRow(dstMsg, DisallowedUrlsOutputTag);
    }

public:
    TString State;
};
REGISTER_REDUCER(TDisallowedUrlsDiffReducer)

TString GetJupiterProductionState(NYT::IClientBasePtr client) {
    TString state, error;
    if (NWebmaster::GetJupiterProductionState(client, state, error)) {
        return state;
    }
    ythrow yexception() << "AcceptDisallowedUrls, " << error;
}

void ValidateDisallowedUrls(NYT::IClientBasePtr client) {
    const auto &cfg = TConfig::CInstance();

    TDeque<double> diffSamplesSize;
    TDeque<double> diffAverageClicks;

    TMap<double, TString> diffSamplesSizesExamples;
    TMap<double, TString> diffAverageClicksSamples;

    auto reader = TTable<NProto::TDisallowedUrlsDiff>(client, cfg.TABLE_ACCEPTANCE_DISALLOWED_URLS_FILTERED).GetReader();
    for (; reader->IsValid(); reader->Next()) {
        const auto &row = reader->GetRow();
        AddSample(diffSamplesSize, diffSamplesSizesExamples, row.GetDiffSamplesSizes(), row.GetHost());
        AddSample(diffAverageClicks, diffAverageClicksSamples, row.GetDiffAverageClicks(), row.GetHost());
    }

    bool rejected = false;
    rejected |= IsThresholdBroken("ValidateDisallowedUrls diffSamplesSize",
            diffSamplesSize,    diffSamplesSizesExamples, 0.70, 0.10);
    rejected |= IsThresholdBroken("ValidateDisallowedUrls diffAverageClicks",
            diffAverageClicks,  diffAverageClicksSamples, 0.60, 0.20);

    if (rejected) {
        ythrow yexception() << "ValidateDisallowedUrls are rejected";
    }
}

void ArchiveAll(NYT::IClientBasePtr client) {
    const auto &cfg = TConfig::CInstance();
    auto reader = client->CreateTableReader<NProto::TDisallowedUrlsDiff>(cfg.TABLE_ACCEPTANCE_DISALLOWED_URLS_FILTERED);
    auto writer = client->CreateTableWriter<NProto::TDisallowedUrlsDiff>(
        NYT::TRichYPath(cfg.TABLE_ACCEPTANCE_DISALLOWED_URLS_FILTERED_ARCHIVE).Append(true)
        )
    ;
    for (; reader->IsValid(); reader->Next()) {
        const auto &row = reader->GetRow();
        writer->AddRow(row);
    }
}

int AcceptDisallowedUrls(int, const char **) {
    const auto &cfg = TConfig::CInstance();
    const auto &ccfg = TCommonYTConfig::CInstance();
    const auto &hicfg = NHostInfo::TConfig::CInstance();

    NYT::IClientPtr client = NYT::CreateClient(cfg.MR_SERVER_DISALLOWED_URLS);

    NYT::ITransactionPtr tx = client->StartTransaction();

    TString stageTable = hicfg.TABLE_EXPORT_DISALLOWED_URLS_FILTERED;
    TString acceptanceTable = cfg.TABLE_ACCEPTANCE_DISALLOWED_URLS_FILTERED;
    TString acceptedTable = ccfg.GetAcceptedPath(stageTable);

    if (!client->Exists(acceptedTable)) {
        LOG_INFO("first launch, copying %s to %s", stageTable.c_str(), acceptedTable.c_str());
        client->Copy(stageTable, acceptedTable, NYT::TCopyOptions().Recursive(true));
    }

    if (!client->Exists(acceptanceTable)) {
        client->CreateTable<NProto::TDisallowedUrlsDiff>(acceptanceTable, NYT::TSortColumns(), NYT::TCreateOptions().Recursive(true));
    }

    TYtSourceTrigger newStateTrigger(tx, stageTable);
    TYtSourceTrigger oldStateTrigger(tx, acceptanceTable);

    if (!oldStateTrigger.NeedUpdate(newStateTrigger.Source)) {
        LOG_INFO("source %s is already processed", stageTable.c_str());
        return 0;
    }

    DoParallel(
        TSortCmd<::NWebmaster::NProto::TMeasuredHost>(tx)
            .Input<::NWebmaster::NProto::TMeasuredHost>(stageTable)
            .Output<::NWebmaster::NProto::TMeasuredHost>(stageTable)
            .By({"Host"})
        ,
        TSortCmd<::NWebmaster::NProto::TMeasuredHost>(tx)
            .Input<::NWebmaster::NProto::TMeasuredHost>(acceptedTable)
            .Output<::NWebmaster::NProto::TMeasuredHost>(acceptedTable)
            .By({"Host"})
        )
    ;

    TReduceCmd<TDisallowedUrlsDiffReducer>(tx, new TDisallowedUrlsDiffReducer(newStateTrigger.Source))
        .Input(TTable<::NWebmaster::NProto::TMeasuredHost>(tx, acceptedTable), DisallowedUrlsPrevInputTag)
        .Input(TTable<::NWebmaster::NProto::TMeasuredHost>(tx, stageTable), DisallowedUrlsNewInputTag)
        .Output(TTable<NProto::TDisallowedUrlsDiff>(tx, acceptanceTable), DisallowedUrlsOutputTag)
        .ReduceBy({"Host"})
        .Do()
    ;

    ArchiveAll(tx);

    oldStateTrigger.Update(tx, newStateTrigger.Source);

    ValidateDisallowedUrls(tx);

    const auto opts = NYT::TCopyOptions().Force(true).Recursive(true);
    tx->Copy(stageTable, acceptedTable, opts);

    tx->Commit();

    return 0;
}

} //namespace NAcceptance
} //namespace NWebmaster
