#include <util/draft/date.h>
#include <library/cpp/getopt/last_getopt.h>
#include <library/cpp/string_utils/url/url.h>


#include <robot/library/yt/static/command.h>
#include <robot/library/yt/static/table.h>


#include <wmconsole/version3/library/jupiter/jupiter.h>
#include <wmconsole/version3/processors/acceptance/conf/config.h>
#include <wmconsole/version3/processors/acceptance/protos/acceptance.pb.h>
#include <wmconsole/version3/wmcutil/log.h>
#include <wmconsole/version3/wmcutil/monitor.h>

#include "task_mirrors2.h"

namespace NWebmaster {
namespace NAcceptance {

using namespace NJupiter;

TString GetMirrorsTable2(const TString &state) {
    return NYTUtils::JoinPath("//home/jupiter/export", state, "mirrors/mirrors");
}

static const char *F_SOURCE_TABLE = "SourceTable";
static const char *F_MAIN_HOST = "MainHost";
static const char *F_HOST = "Host";
static const char *F_PREV_MAIN_MIRROR = "PrevMainMirror";
static const char *F_NEW_MAIN_MIRROR = "NewMainMirror";

static const char *F_DELETE_WWW = "delete_www";
static const char *F_ADD_WWW = "add_www";
static const char *F_HTTP_TO_HTTPS = "http://_https://";
static const char *F_HTTPS_TO_HTTP = "https://_http://";

static const TString CHANGES_SHARE = "CHANGES_SHARE";
static const TString DIFF_HOST_COUNT = "DIFF_HOST_COUNT";
static const TString FULL_HOST_COUNT = "FULL_HOST_COUNT";


enum ESourceTable {
    TABLE_CURRENT = 0,
    TABLE_NEXT,
    TABLE_WEBMASTER_HOSTS
};

struct TPrepareMirrorsMapper : public NYT::IMapper<NYT::TTableReader<NYT::TNode>, NYT::TTableWriter<NYT::TNode>> {
public:
    void Do(TReader *input, TWriter *output) override {

        for (; input->IsValid(); input->Next()) {
            NYT::TNode row = input->GetRow();
            row[F_SOURCE_TABLE] = input->GetTableIndex();
            output->AddRow(row);
        }
    }
};

REGISTER_MAPPER(TPrepareMirrorsMapper)


struct TMirrorsReducerComputeChange
        : public NYT::IReducer<NYT::TTableReader<NYT::TNode>, NYT::TTableWriter<NYT::TNode>> {
public:
    void Do(TReader *input, TWriter *output) override {
        NYT::TNode state;

        bool containsInWebmaster = false;
        const TString host = input->GetRow()[F_HOST].AsString();
        TString prevMainMirror = host;
        TString newMainMirror = host;

        for (; input->IsValid(); input->Next()) {
            // collect all info depending on source
            NYT::TNode row = input->GetRow();

            size_t sourceTable = row[F_SOURCE_TABLE].AsUint64();
            switch (sourceTable) {
                case TABLE_CURRENT:
                    prevMainMirror = row[F_MAIN_HOST].AsString();
                    break;
                case TABLE_NEXT:
                    newMainMirror = row[F_MAIN_HOST].AsString();
                    break;
                case TABLE_WEBMASTER_HOSTS:
                    containsInWebmaster = true;
                    break;
            }
        }


        if (containsInWebmaster && prevMainMirror != newMainMirror) {
            state[F_HOST] = host;
            state[F_PREV_MAIN_MIRROR] = prevMainMirror;
            state[F_NEW_MAIN_MIRROR] = newMainMirror;
            output->AddRow(state);
        }
    }
};

REGISTER_REDUCER(TMirrorsReducerComputeChange)

THashMap<TString, size_t> MonitorMirrorsChangesShare(NYT::IClientBasePtr client, const TString &tableName) {
//    const TConfig &config = TConfig::CInstance();


    THashMap<TString, size_t> counters;
    auto reader = client->CreateTableReader<NYT::TNode>(tableName);
    NYTUtils::TTableInfo tableInfo;
    NYTUtils::GetTableInfo(client, tableName, tableInfo);
    const size_t rows = tableInfo.RecordCount;
    for (; reader->IsValid(); reader->Next()) {
        const NYT::TNode &row = reader->GetRow();
        TStringBuf prevScheme;
        TStringBuf prevPureHost;
        TStringBuf newScheme;
        TStringBuf newPureHost;
        ui16 port;

        bool ok = TryGetSchemeHostAndPort(row[F_PREV_MAIN_MIRROR].AsString(), prevScheme, prevPureHost, port);
        ok &= TryGetSchemeHostAndPort(row[F_NEW_MAIN_MIRROR].AsString(), newScheme, newPureHost, port);

        if (ok && prevScheme != newScheme) {
            counters[ToString(prevScheme) + "_" + newScheme] += 1;
        }

        const bool prevHasWww = prevPureHost.starts_with("www");
        const bool newHasWWW = newPureHost.starts_with("www");

        if (ok && (prevHasWww ^ newHasWWW)) {
            if (prevHasWww) {
                counters[F_DELETE_WWW] += 1;
            } else {
                counters[F_ADD_WWW] += 1;
            }
        }
    }

    NYT::TNode suspParams = NYT::TNode::CreateList();

    for (const auto &obj : counters) {
        const float share = static_cast<float>(obj.second) / static_cast<float>(rows);
        MonitorPushMirrorsChangesShare(obj.first + "_share", share);
        MonitorPushMirrorsChangesShare(obj.first + "_cnt", obj.second);
        suspParams.Add(NYT::TNode()
                               (obj.first, share)
        );
    }

    NYTUtils::SetAttr(client, tableName, "stats", suspParams);

    return counters;
}

int AcceptanceMirrors2(int, const char **) {
    const TString &webmasterHostsName = TConfig::CInstance().TABLE_WEBMATER_HOSTS;
    const TString &homeJupiter = TConfig::CInstance().HOME_JUPITER;
    const TString &tableStatisticsName = TConfig::CInstance().TABLE_ACCEPTANCE_MIRRORS_STATISTICS;

    NYT::IClientPtr client = NYT::CreateClient(TConfig::CInstance().MR_SERVER_HOST);

    NYT::ITransactionPtr tx = client->StartTransaction();
    const TString newJupyterState = NYTUtils::GetAttr(tx, homeJupiter,
                                                TConfig::CInstance().ATTR_JUPITER_DESSERT_PREV_STATE).AsString();
    const TString curJupyterState = NYTUtils::GetAttr(tx, homeJupiter,
                                                TConfig::CInstance().ATTR_JUPITER_PRODUCTION_CURRENT_STATE).AsString();

    const TString newLastComparedTables = curJupyterState + '_' + newJupyterState;

    const TString curTableDiffName = NYTUtils::JoinPath(TConfig::CInstance().TABLE_ACCEPTANCE_MIRRORS_DIFF_ROOT,
                                                        newLastComparedTables);
    TTable<NYT::TNode> tableDiff(tx, curTableDiffName);

    if (tableDiff.Exists() || curJupyterState == newJupyterState) {
        LOG_ERROR("mirrors with state - %s and mirrors with state - %s already compared", curJupyterState.c_str(),
                  newJupyterState.c_str());
        LOG_ERROR("diff for this state you can see in - %s", curTableDiffName.c_str());
        tx->Abort();
        return 0;
    }

    const TString tablePriemkaMirrorsName = GetMirrorsTable2(newJupyterState);
    const TString tableCurrentMirrorsName = GetMirrorsTable2(curJupyterState);

    TTable<NYT::TNode> tableWebmasterHosts(tx, webmasterHostsName);


    NYT::TRichYPath domainsState(curTableDiffName);
    domainsState.Schema(NYT::TTableSchema()
                                .AddColumn(F_HOST, NYT::EValueType::VT_STRING)
                                .AddColumn(F_PREV_MAIN_MIRROR, NYT::EValueType::VT_STRING)
                                .AddColumn(F_NEW_MAIN_MIRROR, NYT::EValueType::VT_STRING));


    TMapReduceCmd<TPrepareMirrorsMapper, TMirrorsReducerComputeChange>
            (tx)
            .Input<NYT::TNode>(tableCurrentMirrorsName)
            .Input<NYT::TNode>(tablePriemkaMirrorsName)
            .Input<NYT::TNode>(webmasterHostsName)
            .Output<NYT::TNode>(domainsState)
            .ReduceBy("Host")
            .Do();

    TSortCmd<NYT::TNode>(tx)
            .Input<NYT::TNode>(curTableDiffName)
            .Output<NYT::TNode>(curTableDiffName)
            .By({"Host"})
            .OperationWeight(TConfig::CInstance().OPERATION_WEIGHT)
            .Do();

    const i64 fullHostCount = tableWebmasterHosts.GetRecordsCount();
    const i64 diffHostCount = tableDiff.GetRecordsCount();
    const double changesShare = (double) diffHostCount / fullHostCount;


    MonitorPushMirrorsChangesShare(CHANGES_SHARE, changesShare);
    MonitorPushMirrorsChangesShare(DIFF_HOST_COUNT, diffHostCount);
    MonitorPushMirrorsChangesShare(FULL_HOST_COUNT, fullHostCount);

    LOG_ERROR("Current: %s", tableCurrentMirrorsName.c_str());
    LOG_ERROR("Next: %s", tablePriemkaMirrorsName.c_str());
    LOG_ERROR("Diff: %s", curTableDiffName.c_str());
    LOG_ERROR("Statistics: %s", tableStatisticsName.c_str());
    LOG_ERROR("Hosts count: %ld", fullHostCount);
    LOG_ERROR("Diff size: %ld", diffHostCount);
    LOG_ERROR("changesShare(%f) over a given threshold()",
              changesShare
    );


    NYTUtils::SetAttr(tx, curTableDiffName, TConfig::CInstance().ATTR_FULL_HOST_COUNT, fullHostCount);
    NYTUtils::SetAttr(tx, curTableDiffName, TConfig::CInstance().ATTR_DIFF_HOST_COUNT, diffHostCount);
    NYTUtils::SetAttr(tx, curTableDiffName, TConfig::CInstance().ATTR_CHANGES_SHARE, changesShare);

    auto statisticsTable = TTable<NProto::TStatisticsMirrors>(tx, tableStatisticsName).ForAppend(true).GetWriter();

    THashMap<TString, size_t> statsMap = MonitorMirrorsChangesShare(tx, curTableDiffName);

    NProto::TStatisticsMirrors statisticsRow;
    statisticsRow.SetCurrentTableState(curJupyterState);
    statisticsRow.SetNextTableState(newJupyterState);
    statisticsRow.SetFullHostCount(fullHostCount);
    statisticsRow.SetDiffHostCount(diffHostCount);
    statisticsRow.SetChangesShare(changesShare);
    statisticsRow.SetTimestamp(ToString(TInstant::Now().TimeT()));
    statisticsRow.SetDeleteChangesWWW(statsMap[F_DELETE_WWW]);
    statisticsRow.SetAddChangesWWW(statsMap[F_ADD_WWW]);
    statisticsRow.SetHttpToHttps(statsMap[F_HTTP_TO_HTTPS]);
    statisticsRow.SetHttpsToHttp(statsMap[F_HTTPS_TO_HTTP]);

    statisticsTable->AddRow(statisticsRow);
    statisticsTable->Finish();
    TSortCmd<NProto::TStatisticsMirrors>(tx, TTable<NProto::TStatisticsMirrors>(tx, tableStatisticsName))
            .By({"Timestamp"})
            .OperationWeight(TConfig::CInstance().OPERATION_WEIGHT)
            .Do();


    tx->Commit();
    return 0;
}

}
} //namespace NWebmaster
