#include <util/draft/date.h>
#include <util/generic/queue.h>
#include <util/generic/size_literals.h>

#include <library/cpp/getopt/last_getopt.h>

#include <robot/library/yt/static/command.h>
#include <robot/library/yt/static/table.h>
#include <robot/jupiter/protos/acceptance.pb.h>

#include <yweb/antispam/common/owner/owner.h>

#include <wmconsole/version3/wmcutil/log.h>

#include <wmconsole/version3/library/jupiter/jupiter.h>
#include <wmconsole/version3/processors/achievements/conf/config.h>
#include <wmconsole/version3/processors/achievements/protos/achievements.pb.h>

#include "task_searchbase.h"

namespace NWebmaster {

using namespace NJupiter;

struct TJupiterMapper : public NYT::IMapper<NYT::TTableReader<NJupiter::TAcceptanceUrlForWebMasterRecord>, NYT::TTableWriter<NProto::TPrepSearchBase>> {
    void Start(TWriter *) override {
        MascotOwnerCanonizer.LoadTrueOwners();
    }

    void Do(TReader *input, TWriter *output) override {
        THashMap<TString, size_t> counters;
        for (; input->IsValid(); input->Next()) {
            const auto &row = input->GetRow();
            if (row.GetIsSearchable()) {
                counters[row.GetHost()]++;
            }
        }

        NProto::TPrepSearchBase dstRow;
        for (const auto &obj : counters) {
            const TString owner = MascotOwnerCanonizer.GetHostOwner(obj.first);
            dstRow.SetMascotOwner(owner);
            dstRow.SetDocsOnSearch(obj.second);
            output->AddRow(dstRow);
        }
    }

public:
    TMascotOwnerCanonizer MascotOwnerCanonizer;
};

REGISTER_MAPPER(TJupiterMapper)

//ReduceBy MascotOwner
struct TJupiterReducer : public NYT::IReducer<NYT::TTableReader<NProto::TPrepSearchBase>, NYT::TTableWriter<NProto::TPrepSearchBase>> {
    void Do(TReader *input, TWriter *output) override {
        const TString owner = input->GetRow().GetMascotOwner();
        size_t docsOnSearch = 0;
        for (; input->IsValid(); input->Next()) {
            docsOnSearch += input->GetRow().GetDocsOnSearch();
        }

        if (docsOnSearch >= 20) {
            NProto::TPrepSearchBase dstRow;
            dstRow.SetMascotOwner(owner);
            dstRow.SetDocsOnSearch(docsOnSearch);
            output->AddRow(dstRow);
        }
    }

public:
    TMascotOwnerCanonizer MascotOwnerCanonizer;
};

REGISTER_REDUCER(TJupiterReducer)

int TaskSearchBase(int, const char **) {
    const auto &cfg = NAchievements::TConfig::CInstance();
    NYT::IClientPtr client = NYT::CreateClient(cfg.MR_SERVER_HOST);

    NYTUtils::CreatePath(client, cfg.TABLE_ACHIEVEMENTS_PREP_ROOT);
    const TString jupiterProductionTable = GetJupiterAcceptanceInProdTable(client);

    TString updatedFrom;
    try {
        updatedFrom = NYTUtils::GetAttr(client, cfg.TABLE_ACHIEVEMENTS_PREP_SEARCHBASE, cfg.ATTR_UPDATE_SOURCE).AsString();
    } catch (yexception &) {
    }

    if (updatedFrom == jupiterProductionTable) {
        LOG_INFO("achievements, searchbase, already updated");
        return 0;
    }

    NYT::ITransactionPtr tx = client->StartTransaction();
    LOG_INFO("achievements, searchbase, previous source %s", updatedFrom.c_str());
    LOG_INFO("achievements, searchbase, input %s", jupiterProductionTable.c_str());
    LOG_INFO("achievements, searchbase, output %s", cfg.TABLE_ACHIEVEMENTS_PREP_SEARCHBASE.c_str());

    TMapReduceCmd<TJupiterMapper, TJupiterReducer>(tx)
        .Input(TTable<NJupiter::TAcceptanceUrlForWebMasterRecord>(tx, jupiterProductionTable)
            .SelectFields({"Host", "IsSearchable"}))
        .Output(TTable<NProto::TPrepSearchBase>(tx, cfg.TABLE_ACHIEVEMENTS_PREP_SEARCHBASE))
        .OperationWeight(cfg.OPERATION_WEIGHT)
        .MapperMemoryLimit(1_GBs)
        .ReducerMemoryLimit(1_GBs)
        .ReduceBy({"MascotOwner"})
        .Do()
    ;
    LOG_INFO("achievements, searchbase, sorting %s", cfg.TABLE_ACHIEVEMENTS_PREP_SEARCHBASE.c_str());

    TSortCmd<NProto::TPrepSearchBase>(tx, TTable<NProto::TPrepSearchBase>(tx, cfg.TABLE_ACHIEVEMENTS_PREP_SEARCHBASE))
        .By({"MascotOwner"})
        .OperationWeight(cfg.OPERATION_WEIGHT)
        .Do()
    ;
    NYTUtils::SetAttr(tx, cfg.TABLE_ACHIEVEMENTS_PREP_SEARCHBASE, cfg.ATTR_UPDATE_SOURCE, jupiterProductionTable);

    tx->Commit();
    LOG_INFO("achievements, searchbase, done");

    return 0;
}

} //namespace NWebmaster
