#include <util/draft/datetime.h>
#include <util/generic/set.h>
#include <util/string/join.h>

#include <wmconsole/version3/protos/geodata.pb.h>
#include <wmconsole/version3/wmcutil/datetime.h>
#include <wmconsole/version3/wmcutil/http_client.h>
#include <wmconsole/version3/wmcutil/yt/misc.h>

#include "config.h"
#include "monitor.h"
#include "geo.h"

namespace NWebmaster {

namespace {
const char *F_ERROR     = "Error";
const char *F_HOST      = "Host";
const char *F_SOURCE    = "Source";
const char *F_REGIONS   = "Regions";
const char *F_REGION    = "Region";
const char *F_VISIBLE   = "Visible";
}

void GetHostRegions(const TString &handler, const TDeque<TString> &hosts, proto::geodata::GeoDataMessage &msg) {
    const int MAX_RETRIES = 5;
    const int TIMEOUT = 10000; //10s
    //TVector<TString> HttpHeaders;
    //HttpHeaders.push_back("Content-Type: application/json");
    //HttpHeaders.push_back(TString("Authorization: OAuth ") + Token);
    const TString data = JoinSeq(";", hosts);
    for (int retry = 0; retry < MAX_RETRIES; retry++) {
        try {
            NWebmaster::TPostRequest<TString> post(handler);
            post.SetTimeout(TIMEOUT);
            //post.SetHeaders(HttpHeaders);
            post.SetData(data);
            TAtomicSharedPtr<TString> res = post.Perform();
            Y_PROTOBUF_SUPPRESS_NODISCARD msg.ParseFromString(*res);
            return;
        } catch (yexception &e) {
            LOG_WARN("source geo regions, %s, retry %d", e.what(), retry);
        } catch (...) {
            LOG_WARN("source geo regions, unknown error, retry %d", retry);
        }
        Sleep(TDuration::Seconds(5));
    }

    ythrow yexception() << "source geo regions, unable to complete http request after " << MAX_RETRIES << " retries";
}

void UploadGeoRegions(NYT::IClientBasePtr clientSearch, const THashSet<TString> &webmasterHosts) {
    const auto &config = TConfig::CInstance();

    TSet<TString> sortedWebmasterHosts(webmasterHosts.begin(), webmasterHosts.end());

    const TString geoSourceTable = NYTUtils::JoinPath(config.TABLE_DIGEST_SOURCE_GEO_REGIONS, NUtils::Date2StrTZ(Now().TimeT()));
    NYT::ITransactionPtr tx = clientSearch->StartTransaction();

    tx->Create(geoSourceTable,
        NYT::NT_TABLE,
        NYT::TCreateOptions()
            .Force(true)
            .Recursive(true)
    );

    NYT::TTableSchema geoSourceSchema;
    geoSourceSchema.Strict(true);
    geoSourceSchema.AddColumn(NYT::TColumnSchema().Name(F_HOST).Type(NYT::VT_STRING).SortOrder(NYT::SO_ASCENDING));
    geoSourceSchema.AddColumn(NYT::TColumnSchema().Name(F_VISIBLE).Type(NYT::VT_BOOLEAN));
    geoSourceSchema.AddColumn(NYT::TColumnSchema().Name(F_ERROR).Type(NYT::VT_STRING));
    geoSourceSchema.AddColumn(NYT::TColumnSchema().Name(F_REGIONS).Type(NYT::VT_ANY));

    LOG_INFO("source geo regions, uploading %lu hosts to %s", sortedWebmasterHosts.size(), geoSourceTable.data());

    auto writer = tx->CreateTableWriter<NYT::TNode>(NYT::TRichYPath(geoSourceTable).Schema(geoSourceSchema));
    for (auto it = sortedWebmasterHosts.begin(); it != sortedWebmasterHosts.end();) {
        TDeque<TString> hostsToRequest;
        for (size_t i = 0; it != sortedWebmasterHosts.end() && i < 1000; ++i, ++it) {
            hostsToRequest.push_back(*it);
        }

        proto::geodata::GeoDataMessage msg;
        GetHostRegions(config.GEO_REGIONS_HANDLER, hostsToRequest, msg);

        for (int i = 0; i < msg.hosts_size(); i++) {
            NYT::TNode regionsNode = NYT::TNode::CreateList();
            for (int j = 0; j < msg.hosts(i).attributes_size(); j++) {
                regionsNode.Add(NYT::TNode()
                    (F_SOURCE, msg.hosts(i).attributes(j).source_type())
                    (F_REGION, msg.hosts(i).attributes(j).region_id())
                );
            }

            writer->AddRow(NYT::TNode()
                (F_HOST, msg.hosts(i).hostname())
                (F_ERROR, msg.hosts(i).error())
                (F_VISIBLE, msg.hosts(i).visible())
                (F_REGIONS, regionsNode)
            );
        }
    }
    writer->Finish();
    tx->Commit();
}

} //namespace NWebmaster
