#include <util/generic/size_literals.h>
#include <util/string/split.h>

#include <kernel/urlnorm/normalize.h>
#include <kernel/yt/utils/yt_utils.h>
#include <library/cpp/string_utils/base64/base64.h>
#include <robot/library/yt/static/table.h>
#include <yweb/antispam/common/owner/owner.h>

#include <wmconsole/version3/protos/digest.pb.h>
#include <wmconsole/version3/wmcutil/log.h>
#include <wmconsole/version3/wmcutil/yt/yt_runner.h>

#include "mirrors.h"

namespace NWebmaster {

namespace {
    const char* F_HOST                          = "Host";
    const char* F_HOST_MAIN                     = "MainHost";
}

struct TMirrorsReducer : public NYT::IReducer<NYT::TTableReader<NYT::TNode>, NYT::TTableWriter<NYT::TNode>> {
    Y_SAVELOAD_JOB(WebmasterHostOwner)

    TMirrorsReducer() = default;

    TMirrorsReducer(const THashSet<TString>& webmasterHostOwner)
            : WebmasterHostOwner(webmasterHostOwner)
    {
    }

//public:
    THashSet<TString> WebmasterHostOwner;

    void Do(TReader* input, TWriter* output) override {
        for (; input->IsValid(); input->Next()) {
            const NYT::TNode& row = input->GetRow();
            TString host = row[F_HOST].AsString();
            TString mainHost = row[F_HOST_MAIN].AsString();
            if (host != mainHost && WebmasterHostOwner.contains(host)) {
                output->AddRow(NYT::TNode()
                        (F_HOST, host)
                        (F_HOST_MAIN, mainHost)
                );
            }
        }


    }
};

REGISTER_REDUCER(TMirrorsReducer)

TString GetLatestTable(const TVector<NYT::TNode> &tables) {
    std::pair<time_t, TString> latest;

    for (const auto &table : tables) {
        const TString tableName = table.AsString();

        const auto timestamp = FromString<time_t>(TStringBuf(tableName).RBefore('-'));
        if (timestamp > latest.first) {
            latest = std::make_pair(timestamp, tableName);
        }
    }

    return latest.second;
}

void PrepareMirrorsSource(NYT::IClientBasePtr clientSearch, const THashSet<TString>& webmasterHosts) {
    const auto &config = TConfig::CInstance();

    const TString ATTR_UPDATE_SOURCE = "update_source";

    TVector<NYT::TNode> tables = clientSearch->List(config.TABLE_SOURCE_JUPITER_EXPORT);

    TString latestTable = GetLatestTable(tables);
    TString sourceTable = NYTUtils::JoinPath(config.TABLE_SOURCE_JUPITER_EXPORT, latestTable, "/mirrors/mirrors");

    LOG_INFO("Source table %s", sourceTable.c_str());

    TString tableTime;

    try {
        tableTime = NJupiter::GetYtAttr(clientSearch, config.TABLE_DIGEST_SOURCE_MIRRORS, ATTR_UPDATE_SOURCE).AsString();
    }catch (yexception &){
        tableTime = "";
    }

    if (latestTable == tableTime) {
        LOG_INFO("Current date table already processed %s", config.TABLE_DIGEST_SOURCE_MIRRORS.c_str());
        return;
    }

    NYT::ITransactionPtr tx = clientSearch->StartTransaction();

    TOpRunner(tx)
        .InputNode(sourceTable)
        .OutputNode(config.TABLE_DIGEST_SOURCE_MIRRORS)
        .MemoryLimit(4_GBs)
        .ReduceBy("MainHost", "Rank", "Host")
        .Reduce(new TMirrorsReducer(webmasterHosts))
        .SortBy(F_HOST)
        .Sort(config.TABLE_DIGEST_SOURCE_MIRRORS);

    NJupiter::SetYtAttr(tx, config.TABLE_DIGEST_SOURCE_MIRRORS, ATTR_UPDATE_SOURCE, latestTable);


    tx->Commit();
}

} //namespace NWebmaster
