#include <algorithm>

#include <util/generic/maybe.h>
#include <util/generic/size_literals.h>

#include <robot/library/yt/static/table.h>
#include <robot/library/yt/static/command.h>
#include <yweb/antispam/common/owner/owner.h>

#include <wmconsole/version3/processors/tools/IKS/protos/iks.pb.h>
#include <wmconsole/version3/protos/digest.pb.h>
#include <wmconsole/version3/wmcutil/log.h>
#include <wmconsole/version3/wmcutil/regex.h>
#include <wmconsole/version3/wmcutil/thread.h>
#include <wmconsole/version3/wmcutil/yt/misc.h>
#include <wmconsole/version3/wmcutil/yt/transfer_manager.h>
#include <wmconsole/version3/wmcutil/yt/yt_runner.h>

#include "checklist.h"
#include "config.h"
#include "monitor.h"
#include "queries.h"
#include "recrawl.h"
#include "search.h"
#include "user_settings.h"
#include "user_verifications.h"

#include "task_report.h"

namespace NWebmaster {

namespace {
const char *F_BEAUTY_URL    = "BeautyUrl";

const char *F_HOST          = "Host";
const char *F_PATH          = "Path";

const char *F_CLICKS        = "Clicks";
const char *F_CLICKS_SPLIT  = "ClicksSplit";
const char *F_GROUP_ID      = "GroupId";
const char *F_GROUP_NAME    = "GroupName";
const char *F_POSITION      = "Position";
const char *F_SHOWS         = "Shows";
const char *F_PERIOD        = "Period";

const char *F_OLD           = "Old";
const char *F_NEW           = "New";
const char *F_DIFF          = "Diff";
const char *F_DIFF_PERCENT  = "DiffPercent";

const char *F_ERROR         = "Error";
const char *F_SOURCE        = "Source";
const char *F_REGIONS       = "Regions";
const char *F_REGION        = "Region";
const char *F_VISIBLE       = "Visible";

const char *F_REPORT        = "Report";

const char *F_POS_QUERIES_SHOWS     = "PosQueriesShowsSamples";
const char *F_POS_QUERIES_CLICKS    = "PosQueriesClicksSamples";
const char *F_POS_QUERIES_POSITION  = "PosQueriesPositionSamples";
const char *F_NEG_QUERIES_SHOWS     = "NegQueriesShowsSamples";
const char *F_NEG_QUERIES_CLICKS    = "NegQueriesClicksSamples";
const char *F_NEG_QUERIES_POSITION  = "NegQueriesPositionSamples";

const char *F_POS_GROUPS_SHOWS      = "PosGroupsShowsSamples";
const char *F_POS_GROUPS_CLICKS     = "PosGroupsClicksSamples";
const char *F_POS_GROUPS_POSITION   = "PosGroupsPositionSamples";
const char *F_NEG_GROUPS_SHOWS      = "NegGroupsShowsSamples";
const char *F_NEG_GROUPS_CLICKS     = "NegGroupsClicksSamples";
const char *F_NEG_GROUPS_POSITION   = "NegGroupsPositionSamples";

const char *F_URL_STATUS    = "UrlStatus";
const char *F_TITLE         = "Title";
const char *F_IS_SEARCHABLE = "IsSearchable";
const char *F_HTTP_CODE     = "HttpCode";
const char *F_SEARCH_STATUS = "SearchStatus";

const char *F_NUM_OF_DOCS               = "NumOfDocs";
const char *F_NUM_OF_DOCS_ON_SEARCH     = "NumOfDocsOnSearch";
const char *F_NUM_NEW_DOCS_ON_SEARCH    = "NewDocsOnSearch";
const char *F_NUM_GONE_DOCS_ON_SEARCH   = "GoneDocsOnSearch";

const char *F_TIMESTAMP = "Timestamp";
const char *F_SUCCESS   = "Success";

const char *F_PROBLEM   = "Problem";

const char *F_USER_SETTINGS         = "UserSettings";

const char *F_USER_ID               = "UserId";
const char *F_USER_LOGIN            = "UserLogin";
const char *F_VERIFICATION_TYPE     = "VerificationType";
const char *F_VERIFICATION_TIME     = "VerificationTime";

const char* F_ACHIEVEMENTS          = "Achievements";

const char* F_REVIEWS_COUNT         = "Count";

const char* F_MIRROR_HOST           = "MainHost";
}

struct TIKSConfig {
    Y_SAVELOAD_DEFINE(IKS, Timestamp)

    TIKSConfig() = default;
    TIKSConfig(const THashMap<TString, ui32> &iks, const time_t timestamp)
        : IKS(iks)
        , Timestamp(timestamp)
    {
    }

public:
    THashMap<TString, ui32> IKS;
    time_t Timestamp = 0;
};

struct TIKSValue {
    TIKSValue() = default;
    TIKSValue(ui32 iks, time_t timestamp)
        : IKS(iks)
        , Timestamp(timestamp)
    {
    }

public:
    ui32 IKS = 0;
    time_t Timestamp = 0;
};

using namespace NJupiter;

struct TBuildQueriesReportReducer : public NYT::IReducer<NYT::TTableReader<NYT::TNode>, NYT::TTableWriter<NYT::TNode>> {
    Y_SAVELOAD_JOB(TableConfig, OldIKS, NewIKS)

    TBuildQueriesReportReducer() = default;
    TBuildQueriesReportReducer(const THashMap<ui32, ETableType> &tableConfig)
        : TableConfig(tableConfig)
    {
    }

    TBuildQueriesReportReducer(const THashMap<ui32, ETableType> &tableConfig,
        const TIKSConfig &oldIKS, const TIKSConfig &newIKS
    )
        : TableConfig(tableConfig)
        , OldIKS(oldIKS)
        , NewIKS(newIKS)
    {
    }

    static void CopyQueriesIndicatorLong(const NYT::TNode &src, proto::digest::IndicatorLong *dst) {
        dst->set_new_value(src[F_NEW].AsInt64());
        dst->set_old_value(src[F_OLD].AsInt64());
        dst->set_diff(src[F_DIFF].AsInt64());
        dst->set_diff_percent(src[F_DIFF_PERCENT].AsDouble());
    }

    static void CopyQueriesIndicatorDouble(const NYT::TNode &src, proto::digest::IndicatorDouble *dst) {
        dst->set_new_value(src[F_NEW].AsDouble());
        dst->set_old_value(src[F_OLD].AsDouble());
        dst->set_diff(src[F_DIFF].AsDouble());
        dst->set_diff_percent(src[F_DIFF_PERCENT].AsDouble());
    }

    static void CopyAllQueries(const NYT::TNode &src, proto::digest::AllQueriesIndicators *dst) {
        CopyQueriesIndicatorLong(src[F_CLICKS], dst->mutable_clicks());
        CopyQueriesIndicatorLong(src[F_SHOWS], dst->mutable_shows());
        CopyQueriesIndicatorDouble(src[F_POSITION], dst->mutable_position());

        const NYT::TNode &clicksSplitNode = src[F_CLICKS_SPLIT];

        for (const NYT::TNode &row : clicksSplitNode[F_OLD].AsList()) {
            proto::digest::ClicksByPeriod *clicks = dst->add_old_clicks_by_period();
            clicks->set_clicks(row[F_CLICKS].AsInt64());
            clicks->set_timestamp(row[F_PERIOD].AsInt64());
        }

        for (const NYT::TNode &row : clicksSplitNode[F_NEW].AsList()) {
            proto::digest::ClicksByPeriod *clicks = dst->add_new_clicks_by_period();
            clicks->set_clicks(row[F_CLICKS].AsInt64());
            clicks->set_timestamp(row[F_PERIOD].AsInt64());
        }
    }

    static void CopyFavoritesAllQueries(const NYT::TNode &src, proto::digest::AllFavoriteQuerySamples *dst) {
        CopyQueriesIndicatorLong(src[F_CLICKS], dst->mutable_clicks());
        CopyQueriesIndicatorLong(src[F_SHOWS], dst->mutable_shows());
        CopyQueriesIndicatorDouble(src[F_POSITION], dst->mutable_position());
    }

    template<class T>
    static void CopyClicksToQueriesSample(const NYT::TNode &src, T *sample) {
        CopyQueriesIndicatorLong(src, sample->mutable_clicks());
    }

    template<class T>
    static void CopyShowsToQueriesSample(const NYT::TNode &src, T *sample) {
        CopyQueriesIndicatorLong(src, sample->mutable_shows());
    }

    template<class T>
    static void CopyPositionToQueriesSample(const NYT::TNode &src, T *sample) {
        CopyQueriesIndicatorDouble(src, sample->mutable_position());
    }

    template<class FAddSamples, class FSamplesCopy>
    void CopyFavoritesSplitTopsQueriesImpl(const NYT::TNode &src, FAddSamples addSample, FSamplesCopy copySamples) {
        if (NYTUtils::IsNodeNull(src)) {
            return;
        }

        for (const auto &queryObj : src.AsMap()) {
            const TString &query = queryObj.first;
            const NYT::TNode &indicatorNode = queryObj.second;
            auto *sample = addSample();
            sample->set_query(query);
            copySamples(indicatorNode, sample);
        }
    }

    void CopyFavoritesSplitQueries(const NYT::TNode &src, proto::digest::SplitFavoriteQuerySamples *dst) {
        using namespace proto::digest;
        CopyFavoritesSplitTopsQueriesImpl(src[F_NEG_QUERIES_CLICKS], std::bind(&SplitFavoriteQuerySamples::add_negative_query_clicks_samples, dst), CopyClicksToQueriesSample<QueryClicksSample>);
        CopyFavoritesSplitTopsQueriesImpl(src[F_POS_QUERIES_CLICKS], std::bind(&SplitFavoriteQuerySamples::add_positive_query_clicks_samples, dst), CopyClicksToQueriesSample<QueryClicksSample>);
        CopyFavoritesSplitTopsQueriesImpl(src[F_NEG_QUERIES_SHOWS], std::bind(&SplitFavoriteQuerySamples::add_negative_query_shows_samples, dst), CopyShowsToQueriesSample<QueryShowsSample>);
        CopyFavoritesSplitTopsQueriesImpl(src[F_POS_QUERIES_SHOWS], std::bind(&SplitFavoriteQuerySamples::add_positive_query_shows_samples, dst), CopyShowsToQueriesSample<QueryShowsSample>);
        CopyFavoritesSplitTopsQueriesImpl(src[F_NEG_QUERIES_POSITION], std::bind(&SplitFavoriteQuerySamples::add_negative_query_position_samples, dst), CopyPositionToQueriesSample<QueryPositionSample>);
        CopyFavoritesSplitTopsQueriesImpl(src[F_POS_QUERIES_POSITION], std::bind(&SplitFavoriteQuerySamples::add_positive_query_position_samples, dst), CopyPositionToQueriesSample<QueryPositionSample>);
    }

    template<class FAddSamples, class FSamplesCopy>
    void CopyGroupsQueriesImpl(const THashMap<TString, TString> &groupNames, const NYT::TNode &src, FAddSamples addSample, FSamplesCopy copySamples) {
        if (NYTUtils::IsNodeNull(src)) {
            return;
        }

        for (const auto &queryObj : src.AsMap()) {
            const TString &groupId = queryObj.first;
            const NYT::TNode &indicatorNode = queryObj.second;
            auto *sample = addSample();
            sample->set_group_id(groupId);
            if (groupNames.contains(groupId)) {
                sample->set_group_name(groupNames.at(groupId));
            }
            copySamples(indicatorNode, sample);
        }
    }

    void CopyGroupsQueries(const NYT::TNode &src, proto::digest::GroupQuerySamples *dst, const THashMap<TString, TString> &groupNames) {
        using namespace proto::digest;
        CopyGroupsQueriesImpl(groupNames, src[F_NEG_GROUPS_CLICKS], std::bind(&GroupQuerySamples::add_negative_query_clicks_samples, dst), CopyClicksToQueriesSample<GroupClicksSample>);
        CopyGroupsQueriesImpl(groupNames, src[F_POS_GROUPS_CLICKS], std::bind(&GroupQuerySamples::add_positive_query_clicks_samples, dst), CopyClicksToQueriesSample<GroupClicksSample>);
        CopyGroupsQueriesImpl(groupNames, src[F_NEG_GROUPS_SHOWS], std::bind(&GroupQuerySamples::add_negative_query_shows_samples, dst), CopyShowsToQueriesSample<GroupShowsSample>);
        CopyGroupsQueriesImpl(groupNames, src[F_POS_GROUPS_SHOWS], std::bind(&GroupQuerySamples::add_positive_query_shows_samples, dst), CopyShowsToQueriesSample<GroupShowsSample>);
        CopyGroupsQueriesImpl(groupNames, src[F_NEG_GROUPS_POSITION], std::bind(&GroupQuerySamples::add_negative_query_position_samples, dst), CopyPositionToQueriesSample<GroupPositionSample>);
        CopyGroupsQueriesImpl(groupNames, src[F_POS_GROUPS_POSITION], std::bind(&GroupQuerySamples::add_positive_query_position_samples, dst), CopyPositionToQueriesSample<GroupPositionSample>);
    }

    void CopyTopsQueries(const NYT::TNode &src, proto::digest::TopQuerySamples *dst) {
        using namespace proto::digest;
        CopyFavoritesSplitTopsQueriesImpl(src[F_NEG_QUERIES_CLICKS], std::bind(&TopQuerySamples::add_negative_query_clicks_samples, dst), CopyClicksToQueriesSample<QueryClicksSample>);
        CopyFavoritesSplitTopsQueriesImpl(src[F_POS_QUERIES_CLICKS], std::bind(&TopQuerySamples::add_positive_query_clicks_samples, dst), CopyClicksToQueriesSample<QueryClicksSample>);
        CopyFavoritesSplitTopsQueriesImpl(src[F_NEG_QUERIES_SHOWS], std::bind(&TopQuerySamples::add_negative_query_shows_samples, dst), CopyShowsToQueriesSample<QueryShowsSample>);
        CopyFavoritesSplitTopsQueriesImpl(src[F_POS_QUERIES_SHOWS], std::bind(&TopQuerySamples::add_positive_query_shows_samples, dst), CopyShowsToQueriesSample<QueryShowsSample>);
        CopyFavoritesSplitTopsQueriesImpl(src[F_NEG_QUERIES_POSITION], std::bind(&TopQuerySamples::add_negative_query_position_samples, dst), CopyPositionToQueriesSample<QueryPositionSample>);
        CopyFavoritesSplitTopsQueriesImpl(src[F_POS_QUERIES_POSITION], std::bind(&TopQuerySamples::add_positive_query_position_samples, dst), CopyPositionToQueriesSample<QueryPositionSample>);
    }

    void CopyGeoRegionsState(const NYT::TNode &stateNode, proto::digest::GeoWeekState *state) {
        state->set_visible(stateNode[F_VISIBLE].AsBool());
        state->set_error(stateNode[F_ERROR].AsString());
        for (const NYT::TNode &regionNode : stateNode[F_REGIONS].AsList()) {
            proto::digest::GeoRegionInfo *region = state->add_regions();
            region->set_region_id(regionNode[F_REGION].AsInt64());
            region->set_source_id(regionNode[F_SOURCE].AsInt64());
        }
    }

    void CopyGeoRegions(const TMaybe<NYT::TNode> &rowGeoOld, const TMaybe<NYT::TNode> &rowGeoNew, proto::digest::GeoState *dst) {
        if (rowGeoOld) {
            CopyGeoRegionsState(rowGeoOld.GetRef(), dst->mutable_old_state());
        }

        if (rowGeoNew) {
            CopyGeoRegionsState(rowGeoNew.GetRef(), dst->mutable_new_state());
        }

        using TRegion = std::pair<long, long>;

        TSet<TRegion> oldRegions;
        TSet<TRegion> newRegions;

        if (dst->has_old_state()) {
            for (int i = 0; i < dst->old_state().regions_size(); i++) {
                const proto::digest::GeoRegionInfo &regionInfo = dst->old_state().regions(i);
                oldRegions.insert(std::make_pair(regionInfo.source_id(), regionInfo.region_id()));
            }
        }

        if (dst->has_new_state()) {
            for (int i = 0; i < dst->new_state().regions_size(); i++) {
                const proto::digest::GeoRegionInfo &regionInfo = dst->new_state().regions(i);
                newRegions.insert(std::make_pair(regionInfo.source_id(), regionInfo.region_id()));
            }
        }

        TDeque<TRegion> unchangedRegions;
        TDeque<TRegion> diffNewRegions;
        TDeque<TRegion> diffGoneRegions;

        std::set_difference(oldRegions.begin(), oldRegions.end(), newRegions.begin(), newRegions.end(), std::back_inserter(diffGoneRegions));
        std::set_difference(newRegions.begin(), newRegions.end(), oldRegions.begin(), oldRegions.end(), std::back_inserter(diffNewRegions));
        std::set_intersection(oldRegions.begin(), oldRegions.end(), newRegions.begin(), newRegions.end(), std::back_inserter(unchangedRegions));

        for (const TRegion &obj : unchangedRegions) {
            proto::digest::GeoRegionInfo *region = dst->add_regions_unchanged();
            region->set_source_id(obj.first);
            region->set_region_id(obj.second);
        }

        for (const TRegion &obj : diffNewRegions) {
            proto::digest::GeoRegionInfo *region = dst->add_regions_new();
            region->set_source_id(obj.first);
            region->set_region_id(obj.second);
        }

        for (const TRegion &obj : diffGoneRegions) {
            proto::digest::GeoRegionInfo *region = dst->add_regions_gone();
            region->set_source_id(obj.first);
            region->set_region_id(obj.second);
        }
    }

    void CopyImportantUrlState(const NYT::TNode &node, proto::digest::ImportantUrlState *dst, bool copyPath = false) {
        if (!NYTUtils::IsNodeNull(node[F_IS_SEARCHABLE])) {
            dst->set_is_searchable(node[F_IS_SEARCHABLE].AsBool());
        }
        if (!NYTUtils::IsNodeNull(node[F_URL_STATUS])) {
            dst->set_url_status(node[F_URL_STATUS].AsUint64());
        }
        if (!NYTUtils::IsNodeNull(node[F_TITLE])) {
            dst->set_title(node[F_TITLE].AsString());
        }
        if (!NYTUtils::IsNodeNull(node[F_HTTP_CODE])) {
            dst->set_http_code(node[F_HTTP_CODE].AsUint64());
        }
        if (copyPath) {
            dst->set_path(node[F_PATH].AsString());
        }
    }

    void FindAndCopyImportantUrlEvent(const NYT::TNode &oldNode, const NYT::TNode &newNode, proto::digest::ImportantUrls *dst) {
        const NYT::TNode &oldNodeSearchable = oldNode[F_IS_SEARCHABLE];
        const NYT::TNode &newNodeSearchable = newNode[F_IS_SEARCHABLE];
        const NYT::TNode &oldNodeHttpCode = oldNode[F_HTTP_CODE];
        const NYT::TNode &newNodeHttpCode = newNode[F_HTTP_CODE];

        if (!NYTUtils::IsNodeNull(oldNodeSearchable) && !NYTUtils::IsNodeNull(newNodeSearchable)
            && oldNodeSearchable != newNodeSearchable) {
            proto::digest::ImportantUrlEvent *event = dst->add_events();
            event->set_path(oldNode[F_PATH].AsString());
            if (newNodeSearchable.AsBool()) {
                event->set_event_type(proto::digest::EVENT_ADDED_TO_SEARCH);
            } else {
                event->set_event_type(proto::digest::EVENT_REMOVED_FROM_SEARCH);
            }
            CopyImportantUrlState(oldNode, event->mutable_old_state());
            CopyImportantUrlState(newNode, event->mutable_new_state());
        }

        if (!NYTUtils::IsNodeNull(oldNodeHttpCode) && !NYTUtils::IsNodeNull(newNodeHttpCode)
            && oldNodeHttpCode != newNodeHttpCode) {
            proto::digest::ImportantUrlEvent *event = dst->add_events();
            event->set_path(oldNode[F_PATH].AsString());
            event->set_event_type(proto::digest::EVENT_HTTP_CODE_CHANGED);
            event->set_old_int(oldNodeHttpCode.AsUint64());
            event->set_new_int(newNodeHttpCode.AsUint64());
            CopyImportantUrlState(oldNode, event->mutable_old_state());
            CopyImportantUrlState(newNode, event->mutable_new_state());
        }
    }

    void CopyImportantUrls(const THashMap<TString, NYT::TNode> &oldImportantUrls, const THashMap<TString, NYT::TNode> &newImportantUrls, proto::digest::ImportantUrls *dst) {
        for (const auto &oldObj : oldImportantUrls) {
            const NYT::TNode &oldNode = oldObj.second;
            if (newImportantUrls.contains(oldObj.first)) {
                FindAndCopyImportantUrlEvent(oldNode, newImportantUrls.at(oldObj.first), dst);
            } else {
                CopyImportantUrlState(oldNode, dst->add_removed_urls(), true);
            }
        }

        for (const auto &newObj : newImportantUrls) {
            if (!oldImportantUrls.contains(newObj.first)) {
                CopyImportantUrlState(newObj.second, dst->add_added_urls(), true);
            }
        }

        dst->set_old_count(oldImportantUrls.size());
        dst->set_new_count(newImportantUrls.size());
    }

    void CopySitetreeState(const NYT::TNode &node, proto::digest::SitetreeState *dst) {
        dst->set_num_of_docs(node[F_NUM_OF_DOCS].AsInt64());
        dst->set_num_of_docs_on_search(node[F_NUM_OF_DOCS_ON_SEARCH].AsInt64());
        dst->set_new_docs_on_search(node[F_NUM_NEW_DOCS_ON_SEARCH].AsInt64());
        dst->set_gone_docs_on_search(node[F_NUM_GONE_DOCS_ON_SEARCH].AsInt64());
    }

    void CopySitetrees(const TMaybe<NYT::TNode> &oldRowSitetree, const TMaybe<NYT::TNode> &newRowSitetree, proto::digest::Sitetree *dst) {
        if (oldRowSitetree) {
            CopySitetreeState(oldRowSitetree.GetRef(), dst->mutable_old_state());
        }

        if (newRowSitetree) {
            CopySitetreeState(newRowSitetree.GetRef(), dst->mutable_new_state());
        }
    }

    void CopyIKS(const TIKSValue &iks, proto::digest::IKSState *state) {
        state->set_iks(iks.IKS);
        state->set_timestamp(iks.Timestamp);
    }

    void CopyIKS(const TString &owner, const TMaybe<TIKSValue> &oldIKS, const TMaybe<TIKSValue> &newIKS, proto::digest::IKS *dst) {
        if (oldIKS) {
            CopyIKS(oldIKS.GetRef(), dst->mutable_old_state());
            dst->set_owner(owner);
        }

        if (newIKS) {
            CopyIKS(newIKS.GetRef(), dst->mutable_new_state());
            dst->set_owner(owner);
        }
    }

    void CopyRecommendedUrls(const THashMap<TString, NYT::TNode> &newRecommendedUrls, proto::digest::RecommendedUrls *dst) {
        for (const auto &obj : newRecommendedUrls) {
            const TString &path = obj.first;
            const NYT::TNode &node = obj.second;
            proto::digest::RecommendedUrlState *state = dst->add_recommended_urls();
            state->set_path(path);
            state->set_clicks(node[F_CLICKS].AsInt64());
            state->set_shows(node[F_SHOWS].AsInt64());
        }
    }

    void CopyRecrawlUrls(const THashMap<TString, TMap<time_t, NYT::TNode>> &recrawlUrls, proto::digest::RecrawlUrls *dst) {
        for (const auto &obj : recrawlUrls) {
            const TString &path = obj.first;
            if (obj.second.empty()) {
                continue;
            }
            const NYT::TNode &latestNode = obj.second.rbegin()->second;
            proto::digest::RecrawlUrlState *state = dst->add_new_state();
            state->set_path(path);
            state->set_success(latestNode[F_SUCCESS].AsBool());
            state->set_http_code(latestNode[F_HTTP_CODE].AsUint64());
            state->set_timestamp(latestNode[F_TIMESTAMP].AsUint64());
        }
    }

    void CopyCheckListProblems(const TDeque<int> &oldCheckListProblems, const TDeque<int> &newCheckListProblems, proto::digest::CheckList *dst) {
        for (int problemId : oldCheckListProblems) {
            dst->mutable_old_status()->add_problems(problemId);
        }
        for (int problemId : newCheckListProblems) {
            dst->mutable_new_status()->add_problems(problemId);
        }
    }

    void CopyUserVerifications(const TDeque<NYT::TNode> &newVerifications, proto::digest::UserVerifications *dst) {
        for (const NYT::TNode &node : newVerifications) {
            proto::digest::UserVerificationState *verification = dst->add_new_verifications();
            if (!NYTUtils::IsNodeNull(node[F_USER_ID])) {
                verification->set_user_id(node[F_USER_ID].AsInt64());
            }
            if (!NYTUtils::IsNodeNull(node[F_USER_LOGIN])) {
                verification->set_login(node[F_USER_LOGIN].AsString());
            }
            if (!NYTUtils::IsNodeNull(node[F_VERIFICATION_TIME])) {
                verification->set_time(node[F_VERIFICATION_TIME].AsInt64());
            }
            if (!NYTUtils::IsNodeNull(node[F_VERIFICATION_TYPE])) {
                verification->set_type(node[F_VERIFICATION_TYPE].AsString());
            }
        }
    }

    void CopySearchHistory(const TDeque<NYT::TNode> &historyNodes, proto::digest::Sitetree *dst) {
        for (const NYT::TNode &node : historyNodes) {
            proto::digest::SitetreeUrlHistory *history = dst->add_url_history();
            history->set_path(node[F_PATH].AsString());
            if (!NYTUtils::IsNodeNull(node[F_TITLE])) {
                history->set_title(node[F_TITLE].AsString());
            }
            if (!NYTUtils::IsNodeNull(node[F_URL_STATUS])) {
                history->set_url_status(node[F_URL_STATUS].AsInt64());
            }
            if (!NYTUtils::IsNodeNull(node[F_SEARCH_STATUS])) {
                history->set_is_added(node[F_SEARCH_STATUS].AsString() == "+");
            }
        }
    }

    void CopyAchievements(const TMaybe<NYT::TNode>& oldAchievements, const TMaybe<NYT::TNode>& newAchievements, proto::digest::Achievements* dst) {
        if (oldAchievements) {
            proto::digest::SrcAchievements src;
            Y_PROTOBUF_SUPPRESS_NODISCARD src.ParseFromString(oldAchievements.GetRef()[F_ACHIEVEMENTS].AsString());

            *dst->mutable_achievements_old() = src.Getachievements();
        }
        if (newAchievements) {
            proto::digest::SrcAchievements src;
            Y_PROTOBUF_SUPPRESS_NODISCARD src.ParseFromString(newAchievements.GetRef()[F_ACHIEVEMENTS].AsString());

            *dst->mutable_achievements_new() = src.Getachievements();
        }
    }

    void CopyReviews(const TMaybe<NYT::TNode>& oldReviews, const TMaybe<NYT::TNode>& newReviews, proto::digest::Reviews* dst) {
        if (oldReviews) {
            dst->set_count_old(oldReviews.GetRef()[F_REVIEWS_COUNT].AsInt64());
        }
        if (newReviews) {
            dst->set_count_new(newReviews.GetRef()[F_REVIEWS_COUNT].AsInt64());
        }
    }

    void CopyMirrors(const TMaybe<NYT::TNode>& mirrors, proto::digest::Mirrors* dst) {
        if (mirrors) {
            dst->set_main_mirror(mirrors.GetRef()[F_MIRROR_HOST].AsString());
        }
    }

public:
    void Start(TWriter */*writer*/) override {
        MascotOwnerCanonizer.LoadTrueOwners();
    }

    void Do(TReader *input, TWriter *output) override {
        const TString host = input->GetRow()[F_HOST].AsString();
        const TString owner = MascotOwnerCanonizer.GetUrlOwner(host);

        proto::digest::DigestWeeklyReportMessage msg;
        THashMap<TString, TString> groupNames;
        TMaybe<NYT::TNode> oldRowGeo, newRowGeo;
        TMaybe<NYT::TNode> oldRowSitetree, newRowSitetree;
        THashMap<TString, NYT::TNode> oldImportantUrls, newImportantUrls;
        THashMap<TString, NYT::TNode> newRecommendedUrls;
        THashMap<TString, TMap<time_t, NYT::TNode>> recrawlUrls;
        TDeque<int> oldCheckListProblems, newCheckListProblems;
        TString oldBeautyUrl, newBeautyUrl;
        TDeque<NYT::TNode> newVerifications;
        TDeque<NYT::TNode> newSearchHistory;
        TMaybe<TIKSValue> oldIKS, newIKS;
        TMaybe<NYT::TNode> oldRowAchievements, newRowAchievements;
        TMaybe<NYT::TNode> oldRowReviews, newRowReviews;
        TMaybe<NYT::TNode> mirrors;

        if (OldIKS.IKS.contains(owner)) {
            oldIKS = TIKSValue(OldIKS.IKS.at(owner), OldIKS.Timestamp);
        }

        if (NewIKS.IKS.contains(owner)) {
            newIKS = TIKSValue(NewIKS.IKS.at(owner), NewIKS.Timestamp);
        }

        for (; input->IsValid(); input->Next()) {
            const NYT::TNode &row = input->GetRow();
            const ETableType tableType = TableConfig.at(input->GetTableIndex());

            switch(tableType) {
            case E_TABLE_TYPE_BEAUTY_URL_OLD:
                oldBeautyUrl = row[F_BEAUTY_URL].AsString();
                break;
            case E_TABLE_TYPE_BEAUTY_URL_NEW:
                newBeautyUrl = row[F_BEAUTY_URL].AsString();
                break;
            case E_TABLE_TYPE_CHECKLIST_OLD:
                oldCheckListProblems.push_back(row[F_PROBLEM].AsInt64());
                break;
            case E_TABLE_TYPE_CHECKLIST_NEW:
                newCheckListProblems.push_back(row[F_PROBLEM].AsInt64());
                break;
            case E_TABLE_TYPE_GEO_OLD:
                oldRowGeo = row;
                break;
            case E_TABLE_TYPE_GEO_NEW:
                newRowGeo = row;
                break;
            case E_TABLE_TYPE_QUERIES_ALL:
                CopyAllQueries(row, msg.mutable_all_queries());
                break;
            case E_TABLE_TYPE_QUERIES_FAVORITES_ALL:
                CopyFavoritesAllQueries(row, msg.mutable_all_favorite_queries());
                break;
            case E_TABLE_TYPE_QUERIES_FAVORITES_SPLIT:
                CopyFavoritesSplitQueries(row, msg.mutable_split_favorite_queries());
                break;
            case E_TABLE_TYPE_QUERIES_GROUPS_SETTINGS:
                groupNames[row[F_GROUP_ID].AsString()] = row[F_GROUP_NAME].AsString();
            case E_TABLE_TYPE_QUERIES_GROUPS:
                CopyGroupsQueries(row, msg.mutable_group_queries(), groupNames);
                break;
            case E_TABLE_TYPE_QUERIES_POPULAR_URLS:
                newRecommendedUrls[row[F_PATH].AsString()] = row;
                break;
            case E_TABLE_TYPE_QUERIES_TOPS:
                CopyTopsQueries(row, msg.mutable_top_queries());
                break;
            case E_TABLE_TYPE_SEARCH_HISTORY:
                newSearchHistory.push_back(row);
                break;
            case E_TABLE_TYPE_SEARCH_IMPORTANT_URLS_OLD:
                oldImportantUrls[row[F_PATH].AsString()] = row;
                break;
            case E_TABLE_TYPE_SEARCH_IMPORTANT_URLS_NEW:
                newImportantUrls[row[F_PATH].AsString()] = row;
                break;
            case E_TABLE_TYPE_SEARCH_SITETREE_OLD:
                oldRowSitetree = row;
                break;
            case E_TABLE_TYPE_SEARCH_SITETREE_NEW:
                newRowSitetree = row;
                break;
            case E_TABLE_TYPE_RECRAWL_URLS_NEW:
                if (row[F_SUCCESS].AsBool()) {
                    recrawlUrls[row[F_PATH].AsString()][row[F_TIMESTAMP].AsUint64()] = row;
                }
                break;
            case E_TABLE_TYPE_USER_VERIFICATIONS:
                newVerifications.push_back(row);
                break;
            case E_TABLE_TYPE_ACHIEVEMENTS_OLD:
                oldRowAchievements = row;
                break;
            case E_TABLE_TYPE_ACHIEVEMENTS_NEW:
                newRowAchievements = row;
                break;
            case E_TABLE_TYPE_REVIEWS_OLD:
                oldRowReviews = row;
                break;
            case E_TABLE_TYPE_REVIEWS_NEW:
                newRowReviews = row;
                break;
            case E_TABLE_TYPE_MIRRORS:
                mirrors = row;
                break;
            default:
                ythrow yexception() << "unknown table index " << input->GetTableIndex();
            };
        }

        if (!oldBeautyUrl.empty() && !newBeautyUrl.empty() && oldBeautyUrl != newBeautyUrl) {
            proto::digest::BeautyUrlState *state = msg.mutable_beauty_urls();
            state->set_old_state(oldBeautyUrl);
            state->set_new_state(newBeautyUrl);
        }

        if (oldIKS || newIKS) {
            CopyIKS(owner, oldIKS, newIKS, msg.mutable_iks());
        }

        if (oldRowGeo || newRowGeo) {
            CopyGeoRegions(oldRowGeo, newRowGeo, msg.mutable_geo_regions());
        }

        if (!oldImportantUrls.empty() || !newImportantUrls.empty()) {
            CopyImportantUrls(oldImportantUrls, newImportantUrls, msg.mutable_important_urls());
            if (msg.important_urls().events().empty() && msg.important_urls().added_urls().empty() && msg.important_urls().removed_urls().empty()) {
                msg.clear_important_urls();
            }
        }

        if (oldRowSitetree || newRowSitetree) {
            CopySitetrees(oldRowSitetree, newRowSitetree, msg.mutable_sitetrees());
        }

        if (!newRecommendedUrls.empty()) {
            CopyRecommendedUrls(newRecommendedUrls, msg.mutable_recommended_urls());
        }

        if (!recrawlUrls.empty()) {
            CopyRecrawlUrls(recrawlUrls, msg.mutable_recrawl_urls());
        }

        if (!oldCheckListProblems.empty() || !newCheckListProblems.empty()) {
            CopyCheckListProblems(oldCheckListProblems, newCheckListProblems, msg.mutable_checklist());
        }

        if (!newVerifications.empty()) {
            CopyUserVerifications(newVerifications, msg.mutable_user_verifications());
        }

        if (!newSearchHistory.empty()) {
            CopySearchHistory(newSearchHistory, msg.mutable_sitetrees());
        }

        if (oldRowAchievements || newRowAchievements) {
            CopyAchievements(oldRowAchievements, newRowAchievements, msg.mutable_achievements());
        }

        if (oldRowReviews || newRowReviews) {
            CopyReviews(oldRowReviews, newRowReviews, msg.mutable_reviews());
        }

        if (mirrors) {
            CopyMirrors(mirrors, msg.mutable_mirrors());
        }

        const ui32 TABLENO_REPORT   = 0;
        const ui32 TABLENO_DEBUG    = 1;
        TString stream;
        Y_PROTOBUF_SUPPRESS_NODISCARD msg.SerializeToString(&stream);
        if (stream.size() < 16_MBs) {
            output->AddRow(NYT::TNode()
                (F_HOST, host)
                (F_REPORT, stream),
                TABLENO_REPORT
            );
        } else {
            output->AddRow(NYT::TNode()
                (F_HOST, host)
                ("ReportLength", stream.size()),
                TABLENO_DEBUG
            );
        }
    }

public:
    THashMap<ui32, ETableType> TableConfig;
    TIKSConfig OldIKS, NewIKS;
    TMascotOwnerCanonizer MascotOwnerCanonizer;
};

REGISTER_REDUCER(TBuildQueriesReportReducer)

struct TBuildReportReducer : public NYT::IReducer<NYT::TTableReader<NYT::TNode>, NYT::TTableWriter<NYT::TNode>> {
public:
    static void CopyPart(const proto::digest::DigestWeeklyReportMessage &src, proto::digest::DigestWeeklyReportMessage &dst) {
        if (src.has_all_favorite_queries()) {
            *dst.mutable_all_favorite_queries() = src.all_favorite_queries();
        }
        if (src.has_all_queries()) {
            *dst.mutable_all_queries() = src.all_queries();
        }
        if (src.has_beauty_urls()) {
            *dst.mutable_beauty_urls() = src.beauty_urls();
        }
        if (src.has_checklist()) {
            *dst.mutable_checklist() = src.checklist();
        }
        if (src.has_geo_regions()) {
            *dst.mutable_geo_regions() = src.geo_regions();
        }
        if (src.has_group_queries()) {
            *dst.mutable_group_queries() = src.group_queries();
        }
        if (src.has_important_urls()) {
            *dst.mutable_important_urls() = src.important_urls();
        }
        if (src.has_recommended_urls()) {
            *dst.mutable_recommended_urls() = src.recommended_urls();
        }
        if (src.has_recrawl_urls()) {
            *dst.mutable_recrawl_urls() = src.recrawl_urls();
        }
        if (src.has_sitetrees()) {
            *dst.mutable_sitetrees() = src.sitetrees();
        }
        if (src.has_split_favorite_queries()) {
            *dst.mutable_split_favorite_queries() = src.split_favorite_queries();
        }
        if (src.has_top_queries()) {
            *dst.mutable_top_queries() = src.top_queries();
        }
        if (src.has_user_verifications()) {
            *dst.mutable_user_verifications() = src.user_verifications();
        }
        if (src.has_iks()) {
            *dst.mutable_iks() = src.iks();
        }
        if (src.has_achievements()) {
            *dst.mutable_achievements() = src.achievements();
        }
        if (src.has_reviews()) {
            *dst.mutable_reviews() = src.reviews();
        }
        if (src.has_mirrors()) {
            *dst.mutable_mirrors() = src.mirrors();
        }
    }

    void Do(TReader *input, TWriter *output) override {
        const int TABLENO_USER_SETTINGS_FULL = 0;
        const int TABLENO_USER_SETTINGS_LITE = 1;
        const int TABLENO_QUERIES_PART  = 2;
        const int TABLENO_SEARCH_PART   = 3;

        TMaybe<proto::digest::DigestWeeklyReportMessage> queriesPartMsg;
        TMaybe<proto::digest::DigestWeeklyReportMessage> searchPartMsg;
        TDeque<NYT::TNode> userSettingsNodes;
        TDeque<NYT::TNode> userLiteSettingsNodes;
        TSet<long> userIdsWithFullDigest;

        const TString host = input->GetRow()[F_HOST].AsString();
        for (; input->IsValid(); input->Next()) {
            const NYT::TNode &row = input->GetRow();
            switch(input->GetTableIndex()) {
            case TABLENO_USER_SETTINGS_FULL:
                userSettingsNodes.push_back(row[F_USER_SETTINGS]);
                userIdsWithFullDigest.insert(row[F_USER_SETTINGS]["user_id"].AsUint64());
                break;
            case TABLENO_USER_SETTINGS_LITE:
                userLiteSettingsNodes.push_back(row[F_USER_SETTINGS]);
                break;
            case TABLENO_QUERIES_PART: {
                queriesPartMsg.ConstructInPlace();
                Y_PROTOBUF_SUPPRESS_NODISCARD queriesPartMsg->ParseFromString(row[F_REPORT].AsString());
                break;
            }
            case TABLENO_SEARCH_PART: {
                searchPartMsg.ConstructInPlace();
                Y_PROTOBUF_SUPPRESS_NODISCARD searchPartMsg->ParseFromString(row[F_REPORT].AsString());
                break;
            }
            }
        }

        proto::digest::DigestWeeklyReportMessage msg;
        if (queriesPartMsg) {
            CopyPart(queriesPartMsg.GetRef(), msg);
        }

        if (searchPartMsg) {
            CopyPart(searchPartMsg.GetRef(), msg);
        }

        //Add lite digest settings, filter duplicates
        for (const NYT::TNode &userSettingsNode : userLiteSettingsNodes) {
            if (!userIdsWithFullDigest.contains(userSettingsNode["user_id"].AsUint64())){
                userSettingsNodes.push_back(userSettingsNode);
            }
        }

        TString stream;
        Y_PROTOBUF_SUPPRESS_NODISCARD msg.SerializeToString(&stream);
        for (const NYT::TNode &userSettingsNode : userSettingsNodes) {
            output->AddRow(NYT::TNode()
                (F_HOST, host)
                (F_REPORT, stream)
                (F_USER_SETTINGS, userSettingsNode)
                (F_USER_ID, userSettingsNode["user_id"].AsUint64())
            );
        }
    }
};

REGISTER_REDUCER(TBuildReportReducer)

bool ValidateQueriesSources(NYT::IClientBasePtr clientQueries, const TDigestWeekConfig &weekConfig) {
    bool valid = true;
    const TString groupsSourceTable = GetGroupsQueriesTableName(weekConfig);
    if (!clientQueries->Exists(groupsSourceTable)) {
        LOG_WARN("report groups queries, table is not processed");
        valid = false;
    }

    const TString favoritesSplitSourceTable = GetSplitFavoritesQueriesTableName(weekConfig);
    const TString favoritesAllSourceTable = GetAllFavoritesQueriesTableName(weekConfig);
    if (!clientQueries->Exists(favoritesAllSourceTable) || !clientQueries->Exists(favoritesSplitSourceTable)) {
        LOG_WARN("report favorites queries, tables are not processed");
        valid = false;
    }

    const TString allQueriesSourceTable = GetAllQueriesTableName(weekConfig);
    if (!clientQueries->Exists(allQueriesSourceTable)) {
        LOG_WARN("report all queries, table is not processed");
        valid = false;
    }

    const TString topsSourceTable = GetTopsQueriesTableName(weekConfig);
    if (!clientQueries->Exists(topsSourceTable)) {
        LOG_WARN("report top queries, table is not processed");
        valid = false;
    }

    const TString popularUrlsSourceTable = GetPopularUrlsTableName(weekConfig);
    if (!clientQueries->Exists(topsSourceTable)) {
        LOG_WARN("report popular urls, table is not processed");
        valid = false;
    }

    return valid;
}

bool ValidateSearchSources(NYT::IClientBasePtr clientSearch, const TDigestWeekConfig &weekConfig) {
    bool valid = true;
    const TString recrawlUrlsSourceTable = GetRecrawlUrlsTableName(weekConfig);
    if (!clientSearch->Exists(recrawlUrlsSourceTable)) {
        LOG_WARN("report popular urls, table is not processed");
        valid = false;
    }

    const TString searchHistorySourceTable = GetSearchHistoryTableName(weekConfig);
    if (!clientSearch->Exists(searchHistorySourceTable)) {
        LOG_WARN("report search history, table is not processed");
        valid = false;
    }

    const TString userVerificationsSourceTable = GetUserVerificationsTableName(weekConfig);
    if (!clientSearch->Exists(userVerificationsSourceTable)) {
        LOG_WARN("report user verifications, table is not processed");
        valid = false;
    }

    return valid;
}

TString GetPartQueriesReportTableName(const TDigestWeekConfig &weekConfig) {
    const auto &config = TConfig::CInstance();
    return NYTUtils::JoinPath(config.TABLE_DIGEST_REPORT_PART_QUERIES, weekConfig.RangeName());
}

TString GetPartSearchReportTableName(const TDigestWeekConfig &weekConfig) {
    const auto &config = TConfig::CInstance();
    return NYTUtils::JoinPath(config.TABLE_DIGEST_REPORT_PART_SEARCH, weekConfig.RangeName());
}

TString GetReportTableName(const TDigestWeekConfig &weekConfig) {
    const auto &config = TConfig::CInstance();
    return NYTUtils::JoinPath(config.TABLE_DIGEST_REPORT, weekConfig.RangeName());
}

bool GetDailyTables(NYT::IClientBasePtr client, const TDigestWeekConfig &weekConfig, const TString &prefix, const TString &regexStr, const TString &comment, TString &oldTable, TString &newTable) {
    TDeque<NYTUtils::TTableInfo> tables;
    NYTUtils::GetTableList(client, prefix, tables);

    TRegularExpression regex(regexStr);
    TMap<time_t, TString> oldTables;
    TMap<time_t, TString> newTables;
    for (const NYTUtils::TTableInfo &table : tables) {
        TVector<TString> period;
        if (regex.GetMatches(NYTUtils::GetTableName(table.Name), period) != 1) {
            continue;
        }

        time_t timestamp = str2date(period[0]);
        if (weekConfig.OldWeek.In(timestamp)) {
            oldTables[timestamp] = table.Name;
        } else if (weekConfig.NewWeek.In(timestamp)) {
            newTables[timestamp] = table.Name;
        }
    }

    if (oldTables.empty()) {
        LOG_ERROR("report %s, empty old table set", comment.data());
    } else {
        oldTable = oldTables.rbegin()->second;
        LOG_INFO("report %s, selected old table %s", comment.data(), oldTable.data());
    }

    if (newTables.empty()) {
        LOG_ERROR("report %s, empty new table set", comment.data());
    } else {
        newTable = newTables.rbegin()->second;
        LOG_INFO("report %s, selected new table %s", comment.data(), newTable.data());
    }

    return !newTables.empty() && !newTables.empty();
}

bool IsDigestReportProcessed(NYT::IClientBasePtr clientSearch, const TDigestWeekConfig &weekConfig) {
    const TString reportTableName = GetReportTableName(weekConfig);
    return clientSearch->Exists(reportTableName);
}

void LoadIKS(NYT::IClientBasePtr client, const TString &source, TIKSConfig &ownerCY) {
    const char *ATTR_UPDATE_TIME = "update_time";
    auto reader = TTable<NIks::NProto::TIKS>(client, source).SelectFields({"Host", "MainMirror", "IKS"}).GetReader();
    for (; reader->IsValid(); reader->Next()) {
        const auto &row = reader->GetRow();
        if (row.GetIKS() > 0) {
            ownerCY.IKS[row.GetHost()] = row.GetIKS();
        }
    }
    ownerCY.Timestamp = NYTUtils::GetAttr(client, source, ATTR_UPDATE_TIME).AsInt64();
}

void BuildDigestReport(NYT::IClientBasePtr clientQueries, NYT::IClientBasePtr clientSearch, const TDigestWeekConfig &weekConfig) {
    const auto &config = TConfig::CInstance();
    NYT::ITransactionPtr txSearch = clientSearch->StartTransaction();
    NYT::ITransactionPtr txQueries = clientQueries->StartTransaction();

    if (!ValidateQueriesSources(txQueries, weekConfig) || !ValidateSearchSources(txSearch, weekConfig)) {
        LOG_ERROR("report, some source tables are not processed");
        return;
    }

    const TString queriesReportPartTableName = GetPartQueriesReportTableName(weekConfig);
    const TString searchReportPartTableName = GetPartSearchReportTableName(weekConfig);
    const TString reportTableName = GetReportTableName(weekConfig);
    bool reportSearchPartPrepared = false;
    bool reportQueriesPartPrepared = false;

    auto taskSearch = [&]() {
        try {
            const char *REGEX_DATE_ONLY = "^(\\d+)$";

            TString oldBeautyUrlsTable, newBeautyUrlsTable;
            TString oldCheckListTable, newCheckListTable;
            TString oldGeoTable, newGeoTable;
            TString oldImportantUrlsTable, newImportantUrlsTable;
            TString oldSitetreeTable, newSitetreeTable;
            TString oldIksTable, newIksTable;
            TString oldAchievementsTable, newAchievementsTable;
            TString oldReviewsTable, newReviewsTable;
            TString mirrorsTable;

            if (!GetDailyTables(txSearch, weekConfig, config.TABLE_DIGEST_SOURCE_BEAUTYURL, REGEX_DATE_ONLY, "beauty urls", oldBeautyUrlsTable, newBeautyUrlsTable)) {
                ythrow yexception() << "report beauty urls, there is no tables";
            }

            if (!GetDailyTables(txSearch, weekConfig, config.TABLE_SOURCE_CHECKLIST_PREFIX, REGEX_DATE_ONLY, "check list", oldCheckListTable, newCheckListTable)) {
                ythrow yexception() << "report check list, there is no tables";
            }

            if (!GetDailyTables(txSearch, weekConfig, config.TABLE_DIGEST_SOURCE_GEO_REGIONS, REGEX_DATE_ONLY, "geo regions", oldGeoTable, newGeoTable)) {
                ythrow yexception() << "report geo regions, there is no tables";
            }

            if (!GetDailyTables(txSearch, weekConfig, config.TABLE_DIGEST_SOURCE_SEARCH_IMPORTANT_URLS, REGEX_DATE_ONLY, "search important urls", oldImportantUrlsTable, newImportantUrlsTable)) {
                ythrow yexception() << "report search important urls, there is no tables";
            }

            if (!GetDailyTables(txSearch, weekConfig, config.TABLE_DIGEST_SOURCE_SEARCH_SITETREE, REGEX_DATE_ONLY, "search sitetree", oldSitetreeTable, newSitetreeTable)) {
                ythrow yexception() << "report search sitetree, there is no tables";
            }

            if (!GetDailyTables(txSearch, weekConfig, config.TABLE_DIGEST_SOURCE_IKS, REGEX_DATE_ONLY, "iks", oldIksTable, newIksTable)) {
                ythrow yexception() << "report iks, there is no tables";
            }

            if (!GetDailyTables(txSearch, weekConfig, config.TABLE_DIGEST_SOURCE_ACHIEVEMENTS, REGEX_DATE_ONLY, "achievements", oldAchievementsTable, newAchievementsTable)) {
                ythrow yexception() << "report achievements, there is no tables";
            }

            if (!GetDailyTables(txSearch, weekConfig, config.TABLE_DIGEST_SOURCE_REVIEWS, REGEX_DATE_ONLY, "reviews", oldReviewsTable, newReviewsTable)) {
                ythrow yexception() << "report reviews, there is no tables " +  config.TABLE_DIGEST_SOURCE_REVIEWS;
            }

            TIKSConfig oldIks, newIks;
            LoadIKS(txSearch, oldIksTable, oldIks);
            LoadIKS(txSearch, newIksTable, newIks);

            TOpRunner runnerSearch(txSearch);

            ui32 searchTableNo = 0;
            THashMap<ui32, ETableType> searchTableConfig;

            runnerSearch.InputNode(oldBeautyUrlsTable);
            searchTableConfig[searchTableNo++] = E_TABLE_TYPE_BEAUTY_URL_OLD;

            runnerSearch.InputNode(newBeautyUrlsTable);
            searchTableConfig[searchTableNo++] = E_TABLE_TYPE_BEAUTY_URL_NEW;

            runnerSearch.InputNode(newCheckListTable);
            searchTableConfig[searchTableNo++] = E_TABLE_TYPE_CHECKLIST_NEW;

            runnerSearch.InputNode(oldCheckListTable);
            searchTableConfig[searchTableNo++] = E_TABLE_TYPE_CHECKLIST_OLD;

            runnerSearch.InputNode(oldGeoTable);
            searchTableConfig[searchTableNo++] = E_TABLE_TYPE_GEO_OLD;

            runnerSearch.InputNode(newGeoTable);
            searchTableConfig[searchTableNo++] = E_TABLE_TYPE_GEO_NEW;

            runnerSearch.InputNode(GetSearchHistoryTableName(weekConfig));
            searchTableConfig[searchTableNo++] = E_TABLE_TYPE_SEARCH_HISTORY;

            runnerSearch.InputNode(oldImportantUrlsTable);
            searchTableConfig[searchTableNo++] = E_TABLE_TYPE_SEARCH_IMPORTANT_URLS_OLD;

            runnerSearch.InputNode(newImportantUrlsTable);
            searchTableConfig[searchTableNo++] = E_TABLE_TYPE_SEARCH_IMPORTANT_URLS_NEW;

            runnerSearch.InputNode(oldSitetreeTable);
            searchTableConfig[searchTableNo++] = E_TABLE_TYPE_SEARCH_SITETREE_OLD;

            runnerSearch.InputNode(newSitetreeTable);
            searchTableConfig[searchTableNo++] = E_TABLE_TYPE_SEARCH_SITETREE_NEW;

            runnerSearch.InputNode(GetRecrawlUrlsTableName(weekConfig));
            searchTableConfig[searchTableNo++] = E_TABLE_TYPE_RECRAWL_URLS_NEW;

            runnerSearch.InputNode(GetUserVerificationsTableName(weekConfig));
            searchTableConfig[searchTableNo++] = E_TABLE_TYPE_USER_VERIFICATIONS;

            runnerSearch.InputNode(oldAchievementsTable);
            searchTableConfig[searchTableNo++] = E_TABLE_TYPE_ACHIEVEMENTS_OLD;

            runnerSearch.InputNode(newAchievementsTable);
            searchTableConfig[searchTableNo++] = E_TABLE_TYPE_ACHIEVEMENTS_NEW;

            runnerSearch.InputNode(oldReviewsTable);
            searchTableConfig[searchTableNo++] = E_TABLE_TYPE_REVIEWS_OLD;

            runnerSearch.InputNode(newReviewsTable);
            searchTableConfig[searchTableNo++] = E_TABLE_TYPE_REVIEWS_NEW;

            runnerSearch.InputNode(config.TABLE_DIGEST_SOURCE_MIRRORS);
            searchTableConfig[searchTableNo++] = E_TABLE_TYPE_MIRRORS;

            runnerSearch
                .OutputNode(NYT::TRichYPath(searchReportPartTableName).SortedBy(F_HOST))
                .OutputNode(NYT::TRichYPath(config.TABLE_DIGEST_REPORT_PART_SEARCH_DEBUG).SortedBy(F_HOST))
                .MemoryLimit(2_GBs)
                .MaxRowWeight(128_MBs)
                .ReduceBy(F_HOST)
                .Reduce(new TBuildQueriesReportReducer(searchTableConfig, oldIks, newIks))
            ;

            txSearch->Commit();
            reportSearchPartPrepared = true;
        } catch (yexception &e) {
            LOG_ERROR("report search part, unable to process: %s", e.what());
        }
    };

    auto taskQueries = [&]() {
        try {
            ui32 queriesTableNo = 0;
            THashMap<ui32, ETableType> tableConfig;

            TOpRunner runnerQueries(txQueries);

            runnerQueries.InputNode(GetAllFavoritesQueriesTableName(weekConfig));
            tableConfig[queriesTableNo++] = E_TABLE_TYPE_QUERIES_FAVORITES_ALL;

            runnerQueries.InputNode(GetAllQueriesTableName(weekConfig));
            tableConfig[queriesTableNo++] = E_TABLE_TYPE_QUERIES_ALL;

            runnerQueries.InputNode(config.TABLE_DIGEST_SOURCE_USER_SETTINGS_QUERIES_GROUPS); //should precede groups table
            tableConfig[queriesTableNo++] = E_TABLE_TYPE_QUERIES_GROUPS_SETTINGS;

            runnerQueries.InputNode(GetGroupsQueriesTableName(weekConfig));
            tableConfig[queriesTableNo++] = E_TABLE_TYPE_QUERIES_GROUPS;

            runnerQueries.InputNode(GetSplitFavoritesQueriesTableName(weekConfig));
            tableConfig[queriesTableNo++] = E_TABLE_TYPE_QUERIES_FAVORITES_SPLIT;

            runnerQueries.InputNode(GetPopularUrlsTableName(weekConfig));
            tableConfig[queriesTableNo++] = E_TABLE_TYPE_QUERIES_POPULAR_URLS;

            runnerQueries.InputNode(GetTopsQueriesTableName(weekConfig));
            tableConfig[queriesTableNo++] = E_TABLE_TYPE_QUERIES_TOPS;

            runnerQueries
                .OutputNode(NYT::TRichYPath(queriesReportPartTableName).SortedBy(F_HOST))
                .OutputNode(NYT::TRichYPath(config.TABLE_DIGEST_REPORT_PART_QUERIES_DEBUG).SortedBy(F_HOST))
                .ReduceBy(F_HOST)
                .MaxRowWeight(128_MBs)
                .Reduce(new TBuildQueriesReportReducer(tableConfig))
            ;

            txQueries->Commit();
            reportQueriesPartPrepared = true;
        } catch (yexception &e) {
            LOG_ERROR("report queries part, unable to process: %s", e.what());
        }
    };

    //Y_UNUSED(taskQueries);
    //Y_UNUSED(taskSearch);
    //taskQueries();
    //taskSearch();
    NUtils::RunAsync(taskQueries, taskSearch);

    if (reportQueriesPartPrepared && reportSearchPartPrepared) {
        //LOG_INFO("report, transmitting table %s from %s to %s", queriesReportPartTableName.data(), config.MR_SERVER_HOST_QUERIES.data(), config.MR_SERVER_HOST_SEARCH.data());
        //TTransferManager(config.GetYTToken()).PostTaskAndWait(
        //    config.MR_SERVER_HOST_QUERIES, queriesReportPartTableName,
        //    config.MR_SERVER_HOST_SEARCH, queriesReportPartTableName
        //);
        //LOG_INFO("report, transmitting table - done");
        NYT::TTableSchema tableSchema;
        tableSchema.Strict(true);
        tableSchema.AddColumn(NYT::TColumnSchema().Name(F_HOST).Type(NYT::VT_STRING).SortOrder(NYT::SO_ASCENDING));
        tableSchema.AddColumn(NYT::TColumnSchema().Name(F_REPORT).Type(NYT::VT_STRING));
        tableSchema.AddColumn(NYT::TColumnSchema().Name(F_USER_SETTINGS).Type(NYT::VT_ANY));
        tableSchema.AddColumn(NYT::TColumnSchema().Name(F_USER_ID).Type(NYT::VT_UINT64));

        txSearch = clientSearch->StartTransaction();

         TOpRunner(txSearch)
            .InputNode(config.TABLE_DIGEST_SOURCE_USER_SETTINGS_CHANNELS)
            .InputNode(config.TABLE_DIGEST_LITE_SOURCE_USER_SETTINGS_CHANNELS)
            .InputNode(queriesReportPartTableName)
            .InputNode(searchReportPartTableName)
            .OutputNode(NYT::TRichYPath(reportTableName).Schema(tableSchema))
            .ReduceBy(F_HOST)
            .Reduce(new TBuildReportReducer);

        txSearch->Commit();
    }
}

int TaskBuildReport(int, const char **) {
    const auto &config = TConfig::CInstance();
    TDigestWeekConfig weekConfig(TWeekConfig(Now().TimeT())/*.PrevWeek()*/.WeekStart);
    NYT::IClientPtr clientQueries = NYT::CreateClient(config.MR_SERVER_HOST_QUERIES);
    NYT::IClientPtr clientSearch = NYT::CreateClient(config.MR_SERVER_HOST_SEARCH);

    if (IsDigestReportProcessed(clientSearch, weekConfig)) {
        LOG_INFO("report %s is already processed", weekConfig.RangeName().data());
        return 0;
    }

    NYTUtils::CreatePath(clientQueries, config.TABLE_DIGEST_ROOT);
    NYTUtils::CreatePath(clientSearch, config.TABLE_DIGEST_ROOT);

    LOG_INFO("processing weeks %s - %s", weekConfig.OldWeek.WeekName().data(), weekConfig.NewWeek.WeekName().data());

    LOG_INFO("loading webmaster hosts");
    THashSet<TString> webmasterHosts;
    if (!NYTUtils::LoadWebmastersHosts(clientQueries, config.TABLE_SOURCE_WEBMASTER_HOSTS, webmasterHosts)) {
        ythrow yexception() << "webmaster hosts table is empty";
    }
    //webmasterHosts.insert("https://lenta.r");
    LOG_INFO("loaded %lu webmaster hosts", webmasterHosts.size());

    bool allQueriesProcessed = false;
    bool checkListProcessed = false;
    bool favoritesQueriesProcessed = false;
    bool groupsQueriesProcessed = false;
    bool popularUrlsProcessed = false;
    bool searchHistoryProcessed = false;
    bool recrawlUrlsProcessed = false;
    bool topsQueriesProcessed = false;
    bool userSettingsProcessed = false;
    bool userVerificationsProcessed = false;

    auto taskAllQueries = [&]() {
        try {
            PrepareAllQueriesSource(clientQueries, webmasterHosts, weekConfig);
            allQueriesProcessed = true;
        } catch (yexception &e) {
            LOG_ERROR("source all queries, unable to process: %s", e.what());
        }
    };

    auto taskCheckList = [&]() {
        try {
            PrepareCheckListSource(clientSearch);
            checkListProcessed = true;
        } catch (yexception &e) {
            LOG_ERROR("source check list, unable to process: %s", e.what());
        }
    };

    auto taskGroupsQueries = [&]() {
        try {
            PrepareGroupsQueriesSource(clientQueries, weekConfig);
            groupsQueriesProcessed = true;
        } catch (yexception &e) {
            LOG_ERROR("source groups queries, unable to process: %s", e.what());
        }
    };

    auto taskFavoritesQueries = [&]() {
        try {
            PrepareFavoritesQueriesSource(clientQueries, weekConfig);
            favoritesQueriesProcessed = true;
        } catch (yexception &e) {
            LOG_ERROR("source favorites queries, unable to process: %s", e.what());
        }
    };

    auto taskTopsQueries = [&]() {
        try {
            PrepareTopsQueriesSource(clientQueries, weekConfig);
            topsQueriesProcessed = true;
        } catch (yexception &e) {
            LOG_ERROR("source tops queries, unable to process: %s", e.what());
        }
    };

    auto taskPopularUrls = [&]() {
        try {
            PreparePopularUrlsSource(clientQueries, webmasterHosts, weekConfig);
            popularUrlsProcessed = true;
        } catch (yexception &e) {
            LOG_ERROR("source popular urls, unable to process: %s", e.what());
        }
    };

    auto taskSearchHistory = [&]() {
        try {
            PrepareSearchHistorySource(clientSearch, weekConfig);
            searchHistoryProcessed = true;
        } catch (yexception &e) {
            LOG_ERROR("source search history, unable to process: %s", e.what());
        }
    };

    auto taskRecrawlUrls = [&]() {
        try {
            PrepareRecrawlUrls(clientSearch, weekConfig);
            recrawlUrlsProcessed = true;
        } catch (yexception &e) {
            LOG_ERROR("source recrawl urls, unable to process: %s", e.what());
        }
    };

    auto taskUserSettings = [&]() {
        try {
            PrepareUserSettings(clientSearch);
            userSettingsProcessed = true;
        } catch (yexception &e) {
            LOG_ERROR("source user settings, unable to process: %s", e.what());
        }
    };

    auto taskUserVerifications = [&]() {
        try {
            PrepareUserVerifications(clientSearch, weekConfig);
            userVerificationsProcessed = true;
        } catch (yexception &e) {
            LOG_ERROR("source user verifications, unable to process: %s", e.what());
        }
    };

    //Y_UNUSED(taskAllQueries);
    //Y_UNUSED(taskFavoritesQueries);
    //Y_UNUSED(taskGroupsQueries);
    //Y_UNUSED(taskTopsQueries);
    //Y_UNUSED(taskPopularUrls);

    NUtils::RunAsync(
        taskAllQueries,
        taskCheckList,
        taskFavoritesQueries,
        taskGroupsQueries,
        taskPopularUrls,
        taskRecrawlUrls,
        taskSearchHistory,
        taskTopsQueries,
        taskUserSettings,
        taskUserVerifications
    );

    if (allQueriesProcessed
        && checkListProcessed
        && favoritesQueriesProcessed
        && groupsQueriesProcessed
        && popularUrlsProcessed
        && recrawlUrlsProcessed
        && searchHistoryProcessed
        && topsQueriesProcessed
        && userSettingsProcessed
        && userVerificationsProcessed
    ) {
        LOG_INFO("all source tables were prepared, building report");
        BuildDigestReport(clientQueries, clientSearch, weekConfig);
    } else {
        LOG_ERROR("some source tables were not prepared");
    }

    return 0;
}

} //namespace NWebmaster
