#pragma once

#include <util/datetime/base.h>
#include <util/draft/date.h>
#include <util/draft/datetime.h>
#include <util/string/builder.h>
#include <util/ysaveload.h>

#include <wmconsole/version3/wmcutil/datetime.h>

namespace NWebmaster {

struct TWeekConfig {
    TWeekConfig() = default;
    TWeekConfig(time_t ts)
        : InstantTimestamp(TInstant::Seconds(ts))
        , InstantWeekStart(GetStartOfWeek(ts))
        , InstantWeekEnd(InstantWeekStart + TDuration::Days(7) - TDuration::Seconds(1))
        , WeekStart(InstantWeekStart.Seconds())
        , WeekEnd(InstantWeekEnd.Seconds())
    {
    }

    TString WeekName() const {
        return TStringBuilder() << NUtils::Date2StrTZ(WeekStart) << "_" << NUtils::Date2StrTZ(WeekEnd);
    }

    bool operator<(const TWeekConfig &rhs) const {
        return WeekStart < rhs.WeekStart;
    }

    bool In(time_t ts) const {
        return ts >= WeekStart && ts <= WeekEnd;
    }

    TWeekConfig PrevWeek() const {
        return TWeekConfig(WeekStart - 3600 * 24 * 7);
    }

    static TInstant GetStartOfWeek(time_t ts) {
        const static int WEEK_START_OFFSET_DAYS = 6;
        int weekDay = (GetWeekDay(ts) + WEEK_START_OFFSET_DAYS) % 7;
        return TInstant::Seconds(TDate(ts).GetStart() - weekDay * 86400);
    }

    static unsigned GetWeekDay(time_t ts) {
        struct tm *ti = localtime(&ts);
        return ti->tm_wday;
    }

private:
    TInstant InstantTimestamp;

public:
    TInstant InstantWeekStart;
    TInstant InstantWeekEnd;
    time_t WeekStart = 0;
    time_t WeekEnd = 0;

    Y_SAVELOAD_DEFINE(
        InstantTimestamp,
        InstantWeekStart,
        InstantWeekEnd,
        WeekStart,
        WeekEnd
    )
};

struct TDigestWeekConfig {
    using Ptr = TAtomicSharedPtr<TDigestWeekConfig>;

public:
    TDigestWeekConfig(time_t now = Now().TimeT())
        : ThisWeek(now)
        , NewWeek(ThisWeek.PrevWeek())
        , OldWeek(NewWeek.PrevWeek())
    {
    }

    TString RangeName() const {
        return NUtils::Date2StrTZ(OldWeek.WeekStart) + "_" + NUtils::Date2StrTZ(NewWeek.WeekEnd);
    }

public:
    TWeekConfig ThisWeek;
    TWeekConfig NewWeek;
    TWeekConfig OldWeek;

    Y_SAVELOAD_DEFINE(
        ThisWeek,
        NewWeek,
        OldWeek
    )
};

enum EWeekPeriod {
    E_WEEK_PERIOD_OLD,
    E_WEEK_PERIOD_NEW,
};

} //namespace NWebmaster
