#include <wmconsole/version3/wmcutil/log.h>
#include <wmconsole/version3/wmcutil/url.h>
#include <wmconsole/version3/wmcutil/yt/yt_runner.h>
#include <wmconsole/version3/wmcutil/yt/yt_utils.h>
#include <wmconsole/version3/library/jupiter/jupiter.h>

#include <wmconsole/version3/processors/indexing/checkurl/conf/config.h>

#include "fields.h"
#include "schemes.h"
#include "task_merge.h"

namespace NWebmaster {

class TMapCheckUrlTableUpdates : public NYT::IMapper<NYT::TTableReader<NYT::TNode>, NYT::TTableWriter<NYT::TNode>> {
public:
    Y_SAVELOAD_JOB(TablesByIndexes)

    TMapCheckUrlTableUpdates() = default;

    TMapCheckUrlTableUpdates(const THashMap<ui32, ui32> &tablesByIndexes)
        : TablesByIndexes(tablesByIndexes) {
    }

    void Do(TReader *reader, TWriter *writer) override {
        for (; reader->IsValid(); reader->Next()) {
            NYT::TNode row = reader->GetRow();
            writer->AddRow(row(F_TABLE_TIMESTAMP, TablesByIndexes[reader->GetTableIndex()]));
        }
    }
public:
    THashMap<ui32, ui32> TablesByIndexes;
};

REGISTER_MAPPER(TMapCheckUrlTableUpdates)

int TaskMergeUpdates(int, const char **) {
    const auto &config = NCheckurl::TConfig::CInstance();
    NYT::IClientPtr client = NYT::CreateClient(config.MR_SERVER_HOST);
    NYT::ITransactionPtr tx = client->StartTransaction();

    NYT::TTableSchema schema = CreateCheckUrlMergeSchema(CreateCheckUrlOutcomeSchema(tx));

    TDeque<NYTUtils::TTableInfo> tablesForMerge;
    TDeque<NYTUtils::TResolvedNode> dest;
    NYTUtils::GetTableList(tx, config.TABLE_CHECKURL_OUTCOME_FOR_MERGE_PREFIX, tablesForMerge);

    if (!tablesForMerge.empty()) {
        TOpRunner mergeRunner(tx);
        NYT::TRichYPath mergedTempTable = NYT::TRichYPath(config.TABLE_CHECKURL_OUTCOME_MERGED + ".temp").Schema(schema);
        NYT::TRichYPath mergedTable = NYT::TRichYPath(config.TABLE_CHECKURL_OUTCOME_MERGED).Schema(schema);

        THashMap<ui32, ui32> tablesByIndexes;
        for (const auto &tableInfo : tablesForMerge) {
            ui32 tableTimestamp;
            const TString &tableName = NYTUtils::GetTableName(tableInfo.Name);
            if (TryFromString(tableName, tableTimestamp)) {
                mergeRunner.InputNode(tableInfo.Name);
                tablesByIndexes[tablesByIndexes.size()] = tableTimestamp;
            } else {
                LOG_ERROR("Unexpected table %s", tableName.data());
            }
        }

        mergeRunner
            .Comment("Merging tables")
            .OutputNode(mergedTempTable)
            .Map(new TMapCheckUrlTableUpdates(tablesByIndexes))

            .Comment("Sorting merged table")
            .SortBy(F_HOST, F_PATH, F_TABLE_TIMESTAMP)
            .Sort(mergedTempTable.Path_);

        if (!tx->Exists(mergedTable.Path_)) {
            // just rename for first time
            mergeRunner.Move(mergedTempTable.Path_, mergedTable.Path_);
        } else {
            mergeRunner
                .Comment("Merging results")
                .InputNode(mergedTempTable.Path_)
                .InputNode(mergedTable)
                .OutputNode(mergedTable)
                .MergeBy(F_HOST, F_PATH, F_TABLE_TIMESTAMP)
                .Merge()
                .Drop(mergedTempTable.Path_);
        }
    }

    // remove old update tables
    for (auto &tableInfo : tablesForMerge) {
        tx->Remove(tableInfo.Name);
    }

    tx->Commit();

    return 0;
}

} //namespace NWebmaster
