#pragma once

#include <util/draft/datetime.h>

#include <robot/jupiter/protos/hostdat.pb.h>
#include <robot/jupiter/library/host/hostinfo.h>

#include <wmconsole/version3/processors/indexing/library/sitetree.h>
#include <wmconsole/version3/wmcutil/datetime.h>
#include <wmconsole/version3/wmcutil/yt/yt_runner.h>
#include <wmconsole/version3/wmcutil/yt/yt_utils.h>

#include "config.h"

namespace NWebmaster {

struct TUrldatTableConfig {
    enum ETableType {
        E_TABLE_UNKNOWN = -1,
        E_TABLE_IMPORTANT_URLS,
        E_TABLE_JUPITER_ACCEPTANCE,
        E_TABLE_EVENT_HISTORY_WEEKS,
        E_TABLE_EVENT_HISTORY_TAIL,
        E_TABLE_SPREAD_PROCESSED,
        E_TABLE_SPREAD_NEW,
        E_TABLE_RECENT_HISTORY,
        E_TABLE_SPREAD_COUNTER,
        E_TABLE_JUPITER_HOSTDAT,
        E_TABLE_JUPITER_HOSTTABLE,
    };

public:
    TUrldatTableConfig() = default;
    TUrldatTableConfig(ETableType type, time_t timestamp)
        : Type(type)
        , Timestamp(timestamp)
    {
    }

public:
    ETableType Type = E_TABLE_UNKNOWN;
    time_t Timestamp = 0;

    Y_SAVELOAD_DEFINE(Type, Timestamp)
};

struct TWeekTableConfig {
    TWeekTableConfig() = default;
    TWeekTableConfig(time_t ts)
        : InstantTimestamp(TInstant::Seconds(ts))
        , InstantWeekStart(GetStartOfWeek(ts))
        , InstantWeekEnd(InstantWeekStart + TDuration::Days(7) - TDuration::Seconds(1))
        , WeekStart(InstantWeekStart.Seconds())
        , WeekEnd(InstantWeekEnd.Seconds())
    {
    }

    TString WeekName() const {
        return TStringBuilder() << NUtils::Date2StrTZ(WeekStart) << "_" << NUtils::Date2StrTZ(WeekEnd);
    }

    bool operator<(const TWeekTableConfig &rhs) const {
        return WeekStart < rhs.WeekStart;
    }

    bool In(time_t ts) const {
        return ts >= WeekStart && ts <= WeekEnd;
    }

    /*
    static int GetDayOfWeek(time_t ts) {
        struct tm *ti = localtime(&ts);
        return ti->tm_wday;
    }
    */
    static TInstant GetStartOfWeek(time_t ts) {
        struct tm *ti = localtime(&ts);
        ts -= (ti->tm_wday * 3600 * 24);

        ti = localtime(&ts);
        ti->tm_hour = 0;
        ti->tm_min = 0;
        ti->tm_sec = 0;

        return TInstant::Seconds(mktime(ti));
    }

public:
    TInstant InstantTimestamp;
    TInstant InstantWeekStart;
    TInstant InstantWeekEnd;
    time_t WeekStart = 0;
    time_t WeekEnd = 0;

    Y_SAVELOAD_DEFINE(
        InstantTimestamp,
        InstantWeekStart,
        InstantWeekEnd,
        WeekStart,
        WeekEnd
    )
};

struct TCrawlerEvent {
    enum EEventType {
        E_URL_NEW       = 0,
        E_URL_CHANGED   = 1,
        E_URL_CRAWLED   = 2,
    };

    bool operator<(const TCrawlerEvent &rhs) const;

    static TCrawlerEvent FromJupiterBase(const TUrldatTableConfig &config, const NYT::TNode &row);
    static TCrawlerEvent FromJupiterSpread(const TUrldatTableConfig &config, const NYT::TNode &row);
    static TCrawlerEvent FromDiff(const NYT::TNode &row);

    bool Empty() const {
        return Timestamp == -1;
    }

public:
    time_t LastAccess = -1;
    time_t Timestamp = -1;
    int HttpCode = -1;
    int PrevHttpCode = -1;
    //TString Title;
};

struct TRobotsRecord {
    Y_SAVELOAD_DEFINE(Robots, LastAccess, BannedByRobotsTxtSince)

    TRobotsRecord() = default;
    TRobotsRecord(const TString &robots, time_t lastAccess, time_t bannedSince);
    void Release();

public:
    TString Robots;
    time_t LastAccess = 0;
    time_t BannedByRobotsTxtSince = 0;
};

struct TDisallowByRobots {
    using Ptr = TAtomicSharedPtr<TDisallowByRobots>;

    TDisallowByRobots() = default;
    TDisallowByRobots(const TString &host, TRobotsRecord &robots);
    void AddRobotsRecord(const TString &host, const TString &robots, time_t lastAccess, bool bannedByRobotsTxtSince);
    bool IsPathDisallowed(const TString &path);

public:
    bool BanByDisallowedRootAge = false;
    TAtomicSharedPtr<NJupiter::THostInfo> HostInfo;
};

struct TJupiterUrldatReducer : public NYT::IReducer<NYT::TTableReader<NYT::TNode>, NYT::TTableWriter<NYT::TNode>> {
    Y_SAVELOAD_JOB(
        TablesInputConfig,
        EventSamplesOutputConfig,
        RecentSamplesOutputConfig,
        CountersOutputConfig,
        WebmasterHosts,
        ImportantUrlsJupiterTableNo,
        ImportantUrlsSpreadTableNo,
        Robots
    )

    TJupiterUrldatReducer() = default;
    TJupiterUrldatReducer(const TDeque<TUrldatTableConfig> &tablesInputConfig, const TMap<TWeekTableConfig, int> &eventSamplesOutputConfig,
        const TMap<TWeekTableConfig, int> &recentSamplesOutputConfig, const THashMap<time_t, int> &countersOutputConfig, const THashSet<TString> &webmasterHosts,
        int importantUrlsJupiterTableNo, int importantUrlsSpreadTableNo, const THashMap<TString, TRobotsRecord> &robots);

    //void Start(TWriter* output) override;
    void Do(TReader *input, TWriter *output) override;
    //void Finish(TWriter *output) override;

    void OutputHistoryCounters(TWriter *output, const TString &host, const TString &path, const THashMap<time_t, THashMap<int, TCrawlerEventCounter>> &httpCounters) const;
    void OutputChange(TWriter *output, const TCrawlerEvent &record, const TString &host, const TString &path, int prevHttpCode) const;
    void OutputNew(TWriter *output, const TCrawlerEvent &record, const TString &host, const TString &path) const;
    void OutputEventSamples(TWriter *output, const TCrawlerEvent &record, const TString &host, const TString &path, TCrawlerEvent::EEventType diffType, int prevHttpCode = -1) const;

public:
    TDeque<TUrldatTableConfig> TablesInputConfig;
    TMap<TWeekTableConfig, int> EventSamplesOutputConfig;
    TMap<TWeekTableConfig, int> RecentSamplesOutputConfig;
    THashMap<time_t, int> CountersOutputConfig;
    THashSet<TString> WebmasterHosts;
    int ImportantUrlsJupiterTableNo = -1;
    int ImportantUrlsSpreadTableNo = -1;
    TMutex RobotsMutex;
    THashMap<TString, TRobotsRecord> Robots;
    THashMap<TString, TDisallowByRobots::Ptr> Banners;
};

struct TCrawlerCountersReducer : public NYT::IReducer<NYT::TTableReader<NYT::TNode>, NYT::TTableWriter<NYT::TNode>> {
    const int TABLENO_COUNTERS_HTTP_CODES = 0;
    const int TABLENO_COUNTERS_EVENTS = 1;
    void Do(TReader *input, TWriter *output) override;
    void OutputHttpCodes(TWriter *output, const TString &host, const TString &path, time_t timestamp, const TWrapperNode::Ptr &node, bool isUserNode);
    void OutputEvents(TWriter *output, const TString &host, const TString &path, time_t timestamp, const TWrapperNode::Ptr &node, bool isUserNode);
    void OutputCrawlerCounters(TWriter *output, const TString &host, time_t timestamp, TSiteTreeShard::Ptr sitetreeShardPtr);
};

struct TCrawlerSitetreePrepareReducer : public NYT::IReducer<NYT::TTableReader<NYT::TNode>, NYT::TTableWriter<NYT::TNode>> {
    void Do(TReader *input, TWriter *output) override;
};

struct TCrawlerWeeklySamplesMapper : public NYT::IMapper<NYT::TTableReader<NYT::TNode>, NYT::TTableWriter<NYT::TNode>> {
    void Do(TReader *input, TWriter *output) override;
};

struct TCrawlerWeeklySamplesReducer : public NYT::IReducer<NYT::TTableReader<NYT::TNode>, NYT::TTableWriter<NYT::TNode>> {
    void Do(TReader *input, TWriter *output) override;
};

struct TCrawlerDailyHttpCountersReducer : public NYT::IReducer<NYT::TTableReader<NYT::TNode>, NYT::TTableWriter<NYT::TNode>> {
    void Do(TReader *input, TWriter *output) override;
};

struct TCrawlerDailyEventCountersReducer : public NYT::IReducer<NYT::TTableReader<NYT::TNode>, NYT::TTableWriter<NYT::TNode>> {
    void Do(TReader *input, TWriter *output) override;
};

struct TCrawlerPrepareRobotsReducer : public NYT::IReducer<NYT::TTableReader<NYT::TNode>, NYT::TTableWriter<NYT::TNode>> {
    Y_SAVELOAD_JOB(WebmasterHosts)

    TCrawlerPrepareRobotsReducer() = default;
    TCrawlerPrepareRobotsReducer(const THashSet<TString> &webmasterHosts);
    void Do(TReader *input, TWriter *output) override;

public:
    THashSet<TString> WebmasterHosts;
};

int TaskUpdateCrawlerSamples(int, const char **);
int TaskCompactEventSamples(int, const char **);
int TaskCompactRecentSamples(int, const char **);

} //namespace NWebmaster
