#include "monitor.h"

namespace NWebmaster {

void MonitorPushSearchBase(const TString &suffix, size_t historyRecords, size_t historyBytes) {
    TSolomonPushBackend pushBackend("webmaster", "webmaster_performance_" + suffix);
    TMonitoringAgent<TSolomonPushBackend> agent(pushBackend);

    agent
        .Next("Sitetree_SearchHistoryRecords", historyRecords)
        .Next("Sitetree_SearchHistoryBytes", historyBytes)
        .GlobalService("webmaster_counters")
        .Commit();
}

void MonitorPushSitetrees(const TString &suffix, size_t urls, size_t urlsInSearch, size_t hosts) {
    TSolomonPushBackend pushBackend("webmaster", "webmaster_performance_" + suffix);
    TMonitoringAgent<TSolomonPushBackend> agent(pushBackend);

    agent
        .Next("Sitetree_Urls", urls)
        .Next("Sitetree_UrlsInSearch", urlsInSearch)
        .Next("Sitetree_Hosts", hosts)
        .GlobalService("webmaster_counters")
        .Commit();
}

void MonitorPushSitetreeAnomalies(const TString &suffix, const TMap<time_t, TMap<TString, double>> &history) {
    TSolomonPushBackend pushBackend("webmaster", "webmaster_performance_" + suffix);
    TMonitoringAgent<TSolomonPushBackend> agent(pushBackend);

    for (const auto &day : history) {
        const time_t timestamp = day.first;

        for (const auto &record : day.second) {
            const TString &source = record.first;
            const double &value = record.second;

            agent
                .Next("Sitetree_" + source, value)
                .Timestamp(TInstant::Seconds(timestamp));
        }
    }

    agent
        .GlobalService("webmaster_quality")
        .Commit();
}

template<class T>
void MonitorPushLabel(const TString &suffix, const TString &service, const TString &label, T value) {
    TSolomonPushBackend pushBackend("webmaster", "webmaster_performance_" + suffix);
    TMonitoringAgent<TSolomonPushBackend> agent(pushBackend);

    agent
        .Next(label, value)
        .GlobalService(service)
        .Commit();
}

template<class T>
void MonitorPushCountersLabel(const TString &suffix, const TString &label, T value) {
    MonitorPushLabel(suffix, "webmaster_counters", label, value);
}

template<class T>
void MonitorPushQualityLabel(const TString &suffix, const TString &label, T value) {
    MonitorPushLabel(suffix, "webmaster_quality", label, value);
}

template<class T>
void MonitorPushStateLabel(const TString &suffix, const TString &label, T value) {
    MonitorPushLabel(suffix, "webmaster_state", label, value);
}

void MonitorPushCheckurlProcessed(const TString &suffix, size_t value) {
    MonitorPushQualityLabel(suffix, "CheckUrl_UrlsProcessed", value);
}

void MonitorPushCheckurlSpamness(const TString &suffix, double value) {
    MonitorPushQualityLabel(suffix, "CheckUrl_HostnamesSpamness", value);
}

void MonitorPushCrawlerSpreads(const TString &suffix, size_t value) {
    MonitorPushQualityLabel(suffix, "Crawler_SpreadsProcessed", value);
}

void MonitorPushHoststat(const TString &suffix, size_t value) {
    MonitorPushCountersLabel(suffix, "Hoststat_Records", value);
}

void MonitorPushImportantUrlsProcessed(const TString &suffix, size_t value) {
    MonitorPushQualityLabel(suffix, "ImportantUrls_UrlsProcessed", value);
}

void MonitorPushImportantUrlsUpdated(const TString &suffix, size_t value) {
    MonitorPushQualityLabel(suffix, "ImportantUrls_UrlsUpdated", value);
}

void MonitorPushImportantUrlsUpdateShare(const TString &suffix, const TString &counter, float value) {
    MonitorPushQualityLabel(suffix, "ImportantUrls_Share" + counter, value);
}

void MonitorPushJupiterLag(const TString &suffix, time_t accpState, time_t prodState) {
/*
    const time_t now = Now().Seconds();
    const double DAY_SECONDS = 86400.0;
    MonitorPushQualityLabel(suffix, "Sitetree_JupiterAcceptanceStateLag", static_cast<double>(now - accpState) / DAY_SECONDS);
    MonitorPushQualityLabel(suffix, "Sitetree_JupiterProductionStateLag", static_cast<double>(now - prodState) / DAY_SECONDS);
*/
    MonitorPushQualityLabel(suffix, "Sitetree_JupiterAcceptanceState", accpState);
    MonitorPushQualityLabel(suffix, "Sitetree_JupiterProductionState", prodState);
}

void MonitorPushSitetreeSuspiciousTables(const TString &suffix, size_t value) {
    MonitorPushQualityLabel(suffix, "Sitetree_SuspiciousTables", value);
}

void MonitorPushXmlSearchMatch(const TString &suffix, double value) {
    MonitorPushQualityLabel(suffix, "Sitetree_XmlSearchMatch", value);
}

} //namespace NWebmaster
