
#include <util/draft/date.h>
#include <util/generic/queue.h>

#include <library/cpp/getopt/last_getopt.h>

#include <robot/library/yt/static/command.h>
#include <robot/library/yt/static/table.h>
#include <kernel/urlnorm/normalize.h>

#include <wmconsole/version3/wmcutil/log.h>
#include <wmconsole/version3/library/jupiter/jupiter.h>
#include <wmconsole/version3/wmcutil/yt/transfer_manager.h>
#include <wmconsole/version3/processors/siteservices/conf/config.h>
#include <wmconsole/version3/processors/siteservices/protos/services.pb.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/json/json_writer.h>

#include "import_util.h"
#include "task_about_alice_skill.h"

namespace NWebmaster {
using namespace NJupiter;

struct TAliceSkillMapper : public NYT::IMapper<NYT::TTableReader<NYT::TNode>, NYT::TTableWriter<NProto::TPrepAlice>> {

    bool Filter(const NYT::TNode &node) {
        return node["onAir"].AsBool() && !node["deleteAt"].IsNull()
               && node["channel"].AsString().equal("aliceSkill")
               && node["publishingSettings"].UncheckedAsMap().contains("brandIsVerified")
               && node["publishingSettings"].UncheckedAsMap().at("brandIsVerified").AsBool();
    }

    void Do(TReader *input, TWriter *output) override {

        for (; input->IsValid(); input->Next()) {
            const auto &row = input->GetRow();
            if (Filter(row)) {
                NProto::TPrepAlice prepAlice;
                NJson::TJsonValue value;

                value["site"] = row["brandVerificationWebsite"].AsString();
                value["name"] = row["name"].AsString();
                value["slug"] = row["slug"].AsString();
                value["hideInStore"] = row["hideInStore"].AsBool();

                prepAlice.SetMascotOwner(GetOwner(value["site"].GetString()));
                prepAlice.SetAliceInfo(JsonToString(value));

                output->AddRow(prepAlice);
            }
        }

    }
};

REGISTER_MAPPER(TAliceSkillMapper)

//ReduceBy MascotOwner
struct TAliceSkillReducer : public NYT::IReducer<NYT::TTableReader<NProto::TPrepAlice>, NYT::TTableWriter<NProto::TPrepAlice>> {
    void Do(TReader *input, TWriter *output) override {
        NProto::TPrepAlice tPrepAliceSkills;
        NJson::TJsonValue value;

        for (; input->IsValid(); input->Next()) {
            const auto &row = input->GetRow();
            TString normalized;
            if (NUrlNorm::NormalizeUrl(row.GetMascotOwner(), normalized)){
                tPrepAliceSkills.SetMascotOwner(GetOwner(normalized));
                value.AppendValue(StringToJson(row.GetAliceInfo()));
            }
        }

        tPrepAliceSkills.SetAliceInfo(JsonToString(value));

        if (!tPrepAliceSkills.GetMascotOwner().empty()){
            output->AddRow(tPrepAliceSkills);
        }
    }
};

REGISTER_REDUCER(TAliceSkillReducer)

bool Filter(const NYT::TNode &node) {
    return node["onAir"].AsBool() && !node["deleteAt"].IsNull()
           && node["channel"].AsString().equal("aliceSkill")
           && node["publishingSettings"].UncheckedAsMap().contains("brandIsVerified")
           && node["publishingSettings"].UncheckedAsMap().at("brandIsVerified").AsBool();
}

int AboutAliceSkill(int, const char **) {
    const auto &cfg = TConfig::CInstance();
    NYT::IClientPtr clientFrom = NYT::CreateClient(TConfig::CInstance().MR_SERVER_HOST_ALICE);
    NYT::IClientPtr clientTo = NYT::CreateClient(TConfig::CInstance().MR_SERVER_HOST);

    NYTUtils::CreatePath(clientFrom, cfg.TABLE_SERVICES_PREP_ROOT);
    NYTUtils::CreatePath(clientTo, cfg.TABLE_SERVICES_PREP_ROOT);


    TString tableTimestamp;
    try {
        tableTimestamp = GetYtAttr(clientTo, cfg.TABLE_SERVICES_PREP_SPRAV, cfg.ATTR_UPDATE_SOURCE).AsString();
    }catch (yexception &){
        tableTimestamp = "";
    }
    TString aliceTimestamp = GetYtAttr(clientFrom, cfg.TABLE_SOURCE_ALICE, TYtAttrName::ModificationTime).AsString();

    if (aliceTimestamp == tableTimestamp){
        LOG_INFO("Alice db already updated!");
        return 0;
    }

    NYT::ITransactionPtr tx = clientFrom->StartTransaction();

    TMapReduceCmd<TAliceSkillMapper, TAliceSkillReducer>(tx)
            .Input(TTable<NYT::TNode>(tx, cfg.TABLE_SOURCE_ALICE))
            .Output(TTable<NProto::TPrepAlice>(tx, cfg.TABLE_SERVICES_PREP_ALICE))
            .OperationWeight(cfg.OPERATION_WEIGHT)
            .ReduceBy({"MascotOwner"})
            .Do();

    LOG_INFO("Map-Reduce done");

    TSortCmd<NProto::TPrepAlice>(tx, TTable<NProto::TPrepAlice>(tx, cfg.TABLE_SERVICES_PREP_ALICE))
            .By({"MascotOwner"})
            .OperationWeight(cfg.OPERATION_WEIGHT)
            .Do();

    SetYtAttr(tx, cfg.TABLE_SERVICES_PREP_ALICE, cfg.ATTR_UPDATE_SOURCE, aliceTimestamp);

    LOG_INFO("Sort done");

    tx->Commit();

    TTransferManager tManager(TConfigBase::GetYTToken());
    tManager.PostTaskAndWait(cfg.MR_SERVER_HOST_ALICE, cfg.TABLE_SERVICES_PREP_ALICE, cfg.MR_SERVER_HOST, cfg.TABLE_SERVICES_PREP_ALICE);

    LOG_INFO("Alice task done");

    return 0;
}

} //namespace NWebmaster
