#include <util/draft/date.h>
#include <util/generic/queue.h>

#include <library/cpp/getopt/last_getopt.h>

#include <robot/library/yt/static/command.h>
#include <robot/library/yt/static/table.h>
#include <kernel/urlnorm/normalize.h>

#include <wmconsole/version3/library/jupiter/jupiter.h>
#include <wmconsole/version3/wmcutil/log.h>
#include <wmconsole/version3/processors/siteservices/conf/config.h>
#include <wmconsole/version3/processors/siteservices/protos/sources.pb.h>
#include <wmconsole/version3/processors/siteservices/protos/services.pb.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/json/json_writer.h>

#include <limits.h>

#include "import_util.h"
#include "task_about_mobile.h"

namespace NWebmaster {

using namespace NJupiter;

struct TMobileAppMapper : public NYT::IMapper<NYT::TTableReader<NProto::TSrcMobile>, NYT::TTableWriter<NProto::TPrepMobileApp>> {
    const TString APP_BNO_APAD = "app_bno_apad";
    const TString APP_BNO_ANDROID = "app_bno_android_new";
    const TString APP_BNO_IPHONE = "app_bno_iphone_new";
    const TString APP_BNO_IPAD = "app_bno_ipad";

    void Do(TReader *input, TWriter *output) override {

        for (; input->IsValid(); input->Next()) {
            const auto &row = input->GetRow();
            auto snippet = StringToJson(row.GetData_JSON())["Snippet"].GetMap();
            NJson::TJsonValue value;
            NJson::TJsonValue appInfo;

            if (snippet.contains(APP_BNO_APAD)) value = snippet[APP_BNO_APAD];
            if (snippet.contains(APP_BNO_ANDROID)) value = snippet[APP_BNO_ANDROID];
            if (snippet.contains(APP_BNO_IPHONE)) value = snippet[APP_BNO_IPHONE];
            if (snippet.contains(APP_BNO_IPAD)) value = snippet[APP_BNO_IPAD];

            if (!value["disabled"].GetBoolean()) {
                auto data = value["SerpData"];
                TString type = data["type"].GetString();

                if (type == "bno") {
                    TString normalized;
                    if (NUrlNorm::NormalizeUrl(GetOwner(row.GetSubkey_Url()), normalized)){
                        NProto::TPrepMobileApp mobileApp;
                        mobileApp.SetMascotOwner(GetOwner(normalized));
                        mobileApp.SetAppInfo(JsonToString(data["mobile_apps"]));

                        output->AddRow(mobileApp);
                    }

                }
            }
        }
    }
};

REGISTER_MAPPER(TMobileAppMapper)

//ReduceBy MascotOwner
struct TMobileAppReducer : public NYT::IReducer<NYT::TTableReader<NProto::TPrepMobileApp>, NYT::TTableWriter<NProto::TPrepMobileApp>> {
    void Do(TReader *input, TWriter *output) override {
        NProto::TPrepMobileApp tPrepMobileApp;
        NJson::TJsonValue value;

        for (; input->IsValid(); input->Next()) {
            const auto &row = input->GetRow();
            const auto arr = StringToJson(row.GetAppInfo());

            for (auto card : arr.GetMap()) {
                value[card.first] = card.second;
            }
            tPrepMobileApp.SetMascotOwner(row.GetMascotOwner());
        }
        tPrepMobileApp.SetAppInfo(JsonToString(value));
        output->AddRow(tPrepMobileApp);
    }
};

REGISTER_REDUCER(TMobileAppReducer)

TString GetLatestTable(const TDeque<NWebmaster::NYTUtils::TTableInfo> &tables, const TString &prefix) {
    std::pair<time_t, TString> latest;
    for (const auto &table : tables) {
        const TString tableName = NWebmaster::NYTUtils::GetTableName(table.Name);
        if (tableName.StartsWith(prefix)) {
            const auto timestamp = FromString<time_t>(TStringBuf(table.Name).RAfter('.'));
            if (timestamp > latest.first) {
                latest = std::make_pair(timestamp, table.Name);
            }
        }
    }

    return latest.second;
}

int AboutMobile(int, const char **) {
    const TString MASK_PREFIX = "shinyserp_app_bno_";
    const TString MASK_ANDROID = MASK_PREFIX + "android";
    const TString MASK_APAD = MASK_PREFIX + "apad";
    const TString MASK_IPAD = MASK_PREFIX + "ipad";
    const TString MASK_IPHONE = MASK_PREFIX + "iphone";

    const auto &cfg = TConfig::CInstance();
    NYT::IClientPtr client = NYT::CreateClient(cfg.MR_SERVER_HOST_MOBILE_APPS);

    NYTUtils::CreatePath(client, cfg.TABLE_SERVICES_PREP_ROOT);

    TDeque<NYTUtils::TTableInfo> tables;
    NYTUtils::GetTableList(client, TConfig::CInstance().TABLE_SOURCE_MOBILE_APPS, tables);

    long tableTimestamp;
    try {
        tableTimestamp = GetYtAttr(client, cfg.TABLE_SERVICES_PREP_MOBILE_APPS, cfg.ATTR_UPDATE_SOURCE).AsInt64();
    } catch (yexception &) {
        tableTimestamp = 0l;
    }

    long maxSourceTimestamp;
    try {
        auto tsts = {GetYtAttr(client, GetLatestTable(tables, MASK_ANDROID), cfg.ATTR_UPLOAD_TS).AsInt64(),
                     GetYtAttr(client, GetLatestTable(tables, MASK_APAD), cfg.ATTR_UPLOAD_TS).AsInt64(),
                     GetYtAttr(client, GetLatestTable(tables, MASK_IPAD), cfg.ATTR_UPLOAD_TS).AsInt64(),
                     GetYtAttr(client, GetLatestTable(tables, MASK_IPHONE), cfg.ATTR_UPLOAD_TS).AsInt64()};
        maxSourceTimestamp = *std::max_element(tsts.begin(), tsts.end());
    } catch (yexception &) {
        tableTimestamp = LONG_MAX;
    }

    if (maxSourceTimestamp <= tableTimestamp){
        LOG_INFO("Mobile db already updated!");
        return 0;
    }

    NYTUtils::CreatePath(client, cfg.TABLE_SERVICES_PREP_ROOT);

    NYT::ITransactionPtr tx = client->StartTransaction();

    TMapReduceCmd<TMobileAppMapper, TMobileAppReducer>(tx)
            .Input(TTable<NProto::TSrcMobile>(tx, GetLatestTable(tables, MASK_ANDROID)))
            .Input(TTable<NProto::TSrcMobile>(tx, GetLatestTable(tables, MASK_APAD)))
            .Input(TTable<NProto::TSrcMobile>(tx, GetLatestTable(tables, MASK_IPAD)))
            .Input(TTable<NProto::TSrcMobile>(tx, GetLatestTable(tables, MASK_IPHONE)))
            .Output(TTable<NProto::TPrepMobileApp>(tx, cfg.TABLE_SERVICES_PREP_MOBILE_APPS))
            .OperationWeight(cfg.OPERATION_WEIGHT)
            .ReduceBy({"MascotOwner"})
            .Do();

    LOG_INFO("Map-Reduce done");

    TSortCmd<NProto::TPrepMobileApp>
            (tx, TTable<NProto::TPrepMobileApp>(tx, cfg.TABLE_SERVICES_PREP_MOBILE_APPS))
            .By({"MascotOwner"})
            .OperationWeight(cfg.OPERATION_WEIGHT)
            .Do();

    SetYtAttr(tx, cfg.TABLE_SERVICES_PREP_MOBILE_APPS, cfg.ATTR_UPDATE_SOURCE, maxSourceTimestamp);

    tx->Commit();
    LOG_INFO("Mobile apps task done");

    return 0;
}

} //namespace NWebmaster
