#include <util/draft/date.h>
#include <util/generic/set.h>
#include <util/thread/pool.h>
#include <util/string/join.h>

#include <robot/library/yt/static/command.h>
#include <robot/library/yt/static/table.h>
#include <yql/library/embedded/yql_embedded.h>
#include <kernel/urlnorm/normalize.h>

#undef LOG_DEBUG
#undef LOG_INFO
#undef LOG_ERROR

#include <wmconsole/version3/library/env/yt.h>
#include <wmconsole/version3/processors/siteservices/protos/sources.pb.h>
#include <wmconsole/version3/processors/siteservices/protos/services.pb.h>
#include <wmconsole/version3/processors/siteservices/conf/config.h>
#include <wmconsole/version3/wmcutil/yt/transfer_manager.h>

#include <ctime>

#include "import_util.h"
#include "task_about_radar_hosts.h"

namespace NWebmaster {

using namespace NJupiter;

struct TRadarMapper : public NYT::IMapper<NYT::TTableReader<NProto::TSrcRadar>, NYT::TTableWriter<NProto::TPrepRadar>> {
    void Do(TReader *input, TWriter *output) override {
        NProto::TPrepRadar radar;
        for (; input->IsValid(); input->Next()) {
            const auto &row = input->GetRow();
            TString normalized;
            if (NUrlNorm::NormalizeUrl(row.GetDomain(), normalized)){
                radar.SetMascotOwner(GetOwner(normalized));
                radar.SetType(row.GetType());
                output->AddRow(radar);
            }
        }
    }
};

REGISTER_MAPPER(TRadarMapper)

struct TRadarReducer : public NYT::IReducer<NYT::TTableReader<NProto::TPrepRadar>, NYT::TTableWriter<NProto::TPrepRadar>> {
    void Do(TReader *input, TWriter *output) override {
        if (!input->IsValid()){
            return;
        }

        NProto::TPrepRadar radar;
        THashSet<TString> types;

        for (;input->IsValid();input->Next()) {
            const auto &row = input->GetRow();
            radar.SetMascotOwner(input->GetRow().GetMascotOwner());
            types.insert(row.GetType());
        }
        radar.SetType(JoinSeq(",", types));

        output->AddRow(radar);
    }
};

REGISTER_REDUCER(TRadarReducer)

int AboutRadarHosts(int, const char **) {
    const auto &cfg = TConfig::CInstance();
    NYT::IClientPtr client = NYT::CreateClient(cfg.MR_SERVER_HOST_RADAR);

    NYTUtils::CreatePath(client, cfg.TABLE_SERVICES_PREP_ROOT);

    TDeque <NYTUtils::TTableInfo> tables;
    TDeque <TString> deque;
    NYTUtils::GetTableList(client, TConfig::CInstance().TABLE_SOURCE_RADAR, tables);

    for (const auto &table: tables) {
        deque.push_back(NWebmaster::NYTUtils::GetTableName(table.Name));
    }

    TString tableTimestamp;
    TString maxSourceTimestamp = "-1";
    try {
        maxSourceTimestamp = deque.front();
        tableTimestamp = GetYtAttr(client, cfg.TABLE_SERVICES_PREP_RADAR, cfg.ATTR_UPDATE_SOURCE).AsString();
    } catch (yexception &) {
    }

    if (tableTimestamp == maxSourceTimestamp){
        LOG_INFO("achievements, radar, Db already updated!");
        return 0;
    }


    LOG_INFO("achievements, radar, Start import table");
    NYT::ITransactionPtr tx = client->StartTransaction();


    TMapReduceCmd<TRadarMapper, TRadarReducer>(tx)
            .Input(TTable<NProto::TSrcRadar>(tx, NYTUtils::JoinPath(cfg.TABLE_SOURCE_RADAR, maxSourceTimestamp)))
            .Output(TTable<NProto::TPrepRadar>(tx, cfg.TABLE_SERVICES_PREP_RADAR))
            .OperationWeight(cfg.OPERATION_WEIGHT)
            .ReduceBy({"MascotOwner"})
            .Do();


    TSortCmd<NProto::TPrepRadar>(tx, TTable<NProto::TPrepRadar>(tx, cfg.TABLE_SERVICES_PREP_RADAR))
            .By({"MascotOwner"})
            .OperationWeight(cfg.OPERATION_WEIGHT)
            .Do();

    LOG_INFO("achievements, radar, Done sort table");

    SetYtAttr(tx, cfg.TABLE_SERVICES_PREP_RADAR, cfg.ATTR_UPDATE_SOURCE, maxSourceTimestamp);

    tx->Commit();

    LOG_INFO("achievements, radar, Start copy table");

    TTransferManager tManager(TConfigBase::GetYTToken());
    tManager.PostTaskAndWait(cfg.MR_SERVER_HOST_RADAR, cfg.TABLE_SERVICES_PREP_RADAR, cfg.MR_SERVER_HOST, cfg.TABLE_SERVICES_PREP_RADAR);


    LOG_INFO("achievements, radar, Import done");
    return 0;
}
} //namespace NWebmaster
