#include <util/draft/date.h>
#include <util/draft/datetime.h>
#include <util/generic/hash.h>
#include <util/generic/singleton.h>
#include <util/generic/size_literals.h>
#include <util/stream/output.h>
#include <util/string/join.h>
#include <util/string/split.h>
#include <util/system/compiler.h>
#include <util/memory/segmented_string_pool.h>

#include <kernel/mirrors/mirrors_trie.h>

#include <library/cpp/getopt/modchooser.h>
#include <library/cpp/charset/ci_string.h>
#include <library/cpp/on_disk/st_hash/static_hash.h>

#include <mapreduce/yt/interface/protos/yamr.pb.h>

#include <robot/library/yt/static/command.h>
#include <robot/library/yt/static/tags.h>
#include <robot/jupiter/protos/acceptance.pb.h>
#include <robot/jupiter/protos/export.pb.h>
#include <robot/jupiter/protos/external/host_mirror.pb.h>

#include <yweb/antispam/mascot/dump/protos/dumps.pb.h>
#include <yweb/antispam/mascot/dump/protos/exports.pb.h>
#include <yweb/antispam/mascot/proto/import.pb.h>

#include <wmconsole/version3/processors/antiall/threats/threats-config.h>
#include <wmconsole/version3/processors/tools/IKS/conf/config.h>
#include <wmconsole/version3/processors/tools/IKS/protos/iks.pb.h>
#include <wmconsole/version3/processors/tools/IKS/utils/canonizer.h>
#include <wmconsole/version3/processors/tools/IKS/utils/utils.h>
#include <wmconsole/version3/library/date_ranges/range_config.h>
#include <wmconsole/version3/library/iks/round.h>
#include <wmconsole/version3/library/jupiter/jupiter.h>
#include <wmconsole/version3/library/sanctions/sanctions.h>
#include <wmconsole/version3/wmcutil/datetime.h>
#include <wmconsole/version3/wmcutil/log.h>
#include <wmconsole/version3/wmcutil/regex.h>
#include <wmconsole/version3/wmcutil/yt/yt_utils.h>
#include <wmconsole/version3/wmcutil/yt/yt_runner.h>

#include "counters.h"
#include "penalty.h"
#include "utils.h"
#include "vitality.h"

using namespace NJupiter;

namespace NWebmaster {
namespace NIks {

const char *ATTR_UPDATE_TIME = "update_time";
const size_t AVERAGING_PERIOD_WEEKS = 52;

float RemapIKS(float x) {
    const float Y = 0.35;
    const float Z = 0.17;

    if (x == 0.0f) {
        return 0.0f;
    }

    /**
     *  x ^ (a e ^ (-c) x ^ b) + 10 = y
     *  Where a, b, c are empirical values:
     *  a = 0.38
     *  b = 0.17
     *  c = 0.085
     *
     *  We can choose these parameters on grouped IKS by TIc distribution, get IKS as median, like:
     *  10  2,870553338133369
     *  20  19,59549424278955
     *  30  33,937349893953645
     *  ...
     *  490000  4767,585347058719
     *  500000  5255,526702749573
     *  610000  6124,16770746651
     *
     *  The equation can be simplified:
     *  Substitute a, c -> x ^ (0.38 e ^ (-0.085) x ^ b) + 10 = y -> x ^ (0.349 x ^ b) + 10 = y
     *  Substitute x, y -> 6124.16 ^ (0.349 6124.16 ^ b) + 10 = 610000 -> b = 0.169
     *  Where 6124.16 = max(IKS), 610000 = max(TIc)
     *  x ^ (0.349 x ^ 0.169) + 10 = y, round 0.349 to 0.35, 0.169 to 0.17
     *  x ^ (Y x ^ Z) + 10 = y
     *
     *  Y = 0.35, Z = 0.17
     */

    const float remapped = pow(x, Y * pow(x, Z)) + 10.0f;
    return Max(x, remapped);
}

float GetPenaltyMultiplier(float penalty) {
    const float PENALTY_BASE = 0.1;
    const float penaltyA = penalty > 0.0f ? PENALTY_BASE : 0.0f;
    const float penaltyB = penalty * PENALTY_BASE;
    return 1.0f - (penaltyA + penaltyB);
}

float GetOrgRatingMultiplier(const TCounters &counters) {
    float orgRatingMultiplier = 1.0; //https://st.yandex-team.ru/WMC-6878#5cb5dffa45c62800206940e3
    const float Ym = 100.0f * counters.UGC_org_rating_count.Get() / counters.Visitors.Get();
    const float Y = Ym < 1.0f ? Ym : 1.0f;
    if (counters.UGC_org_rating_value.Get() > 3.5f && counters.UGC_org_rating_count.Get() >= 10.0f) {
        orgRatingMultiplier = (1.0f + (counters.UGC_org_rating_value.Get() - 3.5f) * 0.2f * Y / 1.5f);
    }
    return orgRatingMultiplier;
}

float GetRawIKS(const TCounters &counters) {
    const float CY = counters.CY100.Get(); //https://st.yandex-team.ru/WMC-7417
    const float CYDamper = (std::pow(CY, 3.0f) / (std::pow(CY, 3.0f) + 500.0f));

    return
        + (1.0f + log(counters.AvgMoreNSecVisits.Get() * counters.Visitors.Get() + 1.0f))
        * (1.0f + log(Min(CY * 42.0f, 300000.0f) + 1.0f) * CYDamper)
        * (1.0f + log(counters.GV3.Get() + 1.0f))
    ;
}

float RoundIKSMin(const float value) {
    if (value < 10.0f) {
        return 10.0f;
    }
    return RoundIKS(value);
}

struct TPatch {
    Y_SAVELOAD_DEFINE(Visitors, GreenTrafficDesktopAllCount, GreenTrafficMobileAllCount)

    TPatch() = default;
    TPatch(float visitors, float gtd, float gtm)
        : Visitors(visitors)
        , GreenTrafficDesktopAllCount(gtd)
        , GreenTrafficMobileAllCount(gtm)
    {
    }

public:
    float Visitors = 0;
    float GreenTrafficDesktopAllCount = 0;
    float GreenTrafficMobileAllCount = 0;
};

TInputTag<NProto::TFrozenCY> FrozenCYInputTag     (1);
TInputTag<NMascot::TXDump> MascotTableInputTag    (2);

TOutputTag<NProto::TIKS> TIKSOutoutTag      (1);

//ReduceBy Host
struct TCalcScoreReducer : public TTaggedReducer {

public:
    TCalcScoreReducer() = default;
    TCalcScoreReducer(
        const TDeque<time_t> &tableConfig, const THashMap<TString, size_t> &tic,
        const THashMap<TString, float> &penalties, const THashSet<TString> &deadOwners,
        const THashMap<TString, TString> &mirrors, const THashMap<TString, TPatch> &fixes,
        const THashMap<TString, size_t> &subOwnersCYFixes
    )
        : TableConfig(tableConfig)
        , TIc(tic)
        , Penalties(penalties)
        , DeadOwners(deadOwners)
        , Mirrors(mirrors)
        , Fixes(fixes)
        , SubOwnersCYFixes(subOwnersCYFixes)
    {
    }

    void Save(IOutputStream& stream) const override {
        ::Save(&stream, TableConfig);
        ::Save(&stream, TIc);
        ::Save(&stream, Penalties);
        ::Save(&stream, DeadOwners);
        ::Save(&stream, Mirrors);
        ::Save(&stream, Fixes);
        ::Save(&stream, SubOwnersCYFixes);
        TTaggedReducer::Save(stream);
    }

    void Load(IInputStream& stream) override {
        ::Load(&stream, TableConfig);
        ::Load(&stream, TIc);
        ::Load(&stream, Penalties);
        ::Load(&stream, DeadOwners);
        ::Load(&stream, Mirrors);
        ::Load(&stream, Fixes);
        ::Load(&stream, SubOwnersCYFixes);
        TTaggedReducer::Load(stream);
    }

    void StartTagged(TTagedWriter) override {
        for (time_t ts : TableConfig) {
            UpdateTimestamp = Max(UpdateTimestamp, ts);
        }
    }

    void DoTagged(TTagedReader reader, TTagedWriter writer) override {
        TMaybe<NProto::TFrozenCY> frozenRow = reader.GetSingleRowMaybe(FrozenCYInputTag);
        reader.SkipRows(FrozenCYInputTag);

        if (!reader.IsValid()) {
            return;
        }

        const TString sourceOwner = reader.GetRow(MascotTableInputTag).GetOwner();
        const TString owner = TIKSCanonizer::GetOwner(sourceOwner);
        const TString ownerUrl = TIKSCanonizer::GetOwnerUrl(sourceOwner);
        const TString mainMirror = Mirrors.contains(ownerUrl) ? Mirrors.at(ownerUrl) : ownerUrl;
        if (IsIgnoredDomain(sourceOwner) || IsIgnoredDomain(mainMirror) || DeadOwners.contains(sourceOwner)) {
            return;
        }

        bool existsInRecent = false;
        TCounters counters;
        for (; reader.IsValid(); reader.Next()) {
            const auto tableIndex = reader.GetTableIndex() - 1;
            if (tableIndex == 0) {
                existsInRecent = true;
            }
            const auto &row = reader.GetRow(MascotTableInputTag);
            
            const time_t timestamp = TableConfig.at(tableIndex);
            if (row.GetVisitors() > 0) {
                counters.Add(timestamp, row);
            }
        }

        if (counters.GetCount() == 0 || !existsInRecent) {
            return;
        }

        const auto fixesIt = Fixes.find(sourceOwner);
        if (fixesIt != Fixes.end()) {
            counters.Visitors.Set(fixesIt->second.Visitors);
            //counters.SetGT(fixesIt->second.GreenTrafficDesktopAllCount + fixesIt->second.GreenTrafficMobileAllCount);
        }

        if (frozenRow.Defined()) {
            counters.SetCY(frozenRow.GetRef().Getcy_flt_100());
        }

        if (SubOwnersCYFixes.contains(sourceOwner)) {
            counters.SetCY(SubOwnersCYFixes.at(sourceOwner));
        }

        const float rawIks = GetRawIKS(counters);
        if (rawIks < 2.75f) {
            return;
        }

        const float scaledIks = RemapIKS(rawIks);
        const float penalty = Penalties.contains(sourceOwner) ? Penalties.at(sourceOwner) : 0.0f;
        const float penaltyMultiplier = GetPenaltyMultiplier(penalty);
        const float orgRatingMultiplier = GetOrgRatingMultiplier(counters);
        const float penalizedIks = scaledIks * penaltyMultiplier * orgRatingMultiplier;

        NProto::TIKS outMsg;
        outMsg.SetHost(sourceOwner);
        outMsg.SetMainMirror(mainMirror);
        outMsg.SetIKSRaw(rawIks);
        outMsg.SetIKSScaled(scaledIks);
        outMsg.SetIKSNotPenalized(RoundIKSMin(scaledIks));
        outMsg.SetIKS(RoundIKSMin(penalizedIks));
        outMsg.SetSources(counters.GetCount());
        outMsg.SetRatingMultiplier(orgRatingMultiplier);
        outMsg.SetLastUpdate(UpdateTimestamp);
        outMsg.SetRoundsSinceUpdate(0);

        if (Penalties.contains(sourceOwner)) {
            outMsg.SetPenalty(penalty);
            outMsg.SetPenaltyMultiplier(penaltyMultiplier);
        }

        if (TIc.contains(sourceOwner)) {
            outMsg.SetTIc(TIc.at(sourceOwner));
        }

        writer.AddRow(outMsg, TIKSOutoutTag);
    }

public:
    TDeque<time_t> TableConfig;
    THashMap<TString, size_t> TIc;
    THashMap<TString, float> Penalties;
    THashSet<TString> DeadOwners;
    THashMap<TString, TString> Mirrors;
    THashMap<TString, TPatch> Fixes;
    THashMap<TString, size_t> SubOwnersCYFixes;
    time_t UpdateTimestamp = 0;
};

REGISTER_REDUCER(TCalcScoreReducer)

TInputTag<NProto::TIKS> IKSAcceptedInputTag     (1);
TInputTag<NProto::TIKS> IKSStageInputTag        (2);
TOutputTag<NProto::TIKS> IKSStageOutputTag      (1);

struct TRandomIKSScoreMergeReducer : public TTaggedReducer {
    TRandomIKSScoreMergeReducer() = default;
    TRandomIKSScoreMergeReducer(const TRandomSampler &sampler)
        : Sampler(sampler)
    {
    }

    void Save(IOutputStream& stream) const override {
        ::Save(&stream, Sampler);
        TTaggedReducer::Save(stream);
    }

    void Load(IInputStream& stream) override {
        ::Load(&stream, Sampler);
        TTaggedReducer::Load(stream);
    }

    void DoTagged(TTagedReader reader, TTagedWriter writer) override {
        TMaybe<NProto::TIKS> prod = reader.GetSingleRowMaybe(IKSAcceptedInputTag);
        reader.SkipRows(IKSAcceptedInputTag);

        TMaybe<NProto::TIKS> stage = reader.GetSingleRowMaybe(IKSStageInputTag);
        reader.SkipRows(IKSStageInputTag);

        if (!stage.Defined()) {
            return;
        }

        const TString host = stage.GetRef().GetHost();
        const bool updateByDigest   = Sampler.GetFloatDigest(host) <= 0.3;
        const bool updateByRounds   = prod.Defined() && prod.GetRef().GetRoundsSinceUpdate() > 1;
        //const bool updateByHistory  = stage.GetRef().GetSources() == 1;
        const bool updateByNewMirror  = !prod.Defined();

        stage.GetRef().SetUpdateByRandom(updateByDigest);
        stage.GetRef().SetUpdateByRounds(updateByRounds);
        stage.GetRef().SetUpdateByNewMirror(updateByNewMirror);

        if (updateByNewMirror || updateByRounds || updateByDigest) {
            writer.AddRow(stage.GetRef(), IKSStageOutputTag);
            return;
        }

        if (prod.Defined()) {
            prod.GetRef().SetRoundsSinceUpdate(prod.GetRef().GetRoundsSinceUpdate() + 1);
            writer.AddRow(prod.GetRef(), IKSStageOutputTag);
        }
    }

public:
    TRandomSampler Sampler;
};

REGISTER_REDUCER(TRandomIKSScoreMergeReducer)

//SortBy Key
struct TAntispamRanksMapper : public NYT::IMapper<NYT::TTableReader<NMascot::TXDump>, NYT::TTableWriter<NMascot::TXDump>> {
    void Do(TReader *input, TWriter *output) override {
        for (; input->IsValid(); input->Next()) {
            auto row = input->GetRow();
            if (row.GetVisitors() > 0) {
                row.SetOwner(TIKSCanonizer::GetOwner(row.GetOwner()));
                output->AddRow(row);
            }
        }
    }
};

REGISTER_MAPPER(TAntispamRanksMapper)

//ReduceBy Owner
struct TAntispamRanksReducer : public NYT::IReducer<NYT::TTableReader<NMascot::TXDump>, NYT::TTableWriter<NMascot::TXDump>> {
    void Do(TReader *input, TWriter *output) override {
        TMap<size_t, NMascot::TXDump> rows;
        for (; input->IsValid(); input->Next()) {
            const auto &row = input->GetRow();
            if (row.GetVisitors() > 0) {
                rows[row.GetVisitors()] = row;
            }
        }
        if (!rows.empty()) {
            output->AddRow(rows.rbegin()->second);
        }
    }
};

REGISTER_REDUCER(TAntispamRanksReducer)

//ReduceBy Host
struct TContentPrepareReducer : public NYT::IReducer<NYT::TTableReader<NProto::TIKS>, NYT::TTableWriter<NProto::TIKSWithContent>> {
    void Do(TReader *input, TWriter *output) override {
        const ui32 TABLENO_IKS_PREV = 0;
        const ui32 TABLENO_IKS_CURR = 1;
        const NProto::TIKS firstRow = input->GetRow();
        const TString owner = TIKSCanonizer::GetOwner(firstRow.GetHost());

        TMaybe<ui64> prevIKS, currIKS;
        TString mainMirror = firstRow.GetMainMirror();

        for (; input->IsValid(); input->Next()) {
            const auto &row = input->GetRow();
            if (input->GetTableIndex() == TABLENO_IKS_PREV) {
                prevIKS = row.GetIKS();
            } else if (input->GetTableIndex() == TABLENO_IKS_CURR) {
                currIKS = row.GetIKS();
            }
            mainMirror = row.GetMainMirror();
        }

        if (IsIgnoredDomain(mainMirror)) {
            return;
        }

        NProto::TIKSWithContent dstRow;
        dstRow.SetMascotHost(firstRow.GetHost());
        dstRow.SetHost(mainMirror);
        dstRow.SetPath("/");
        if (prevIKS.Defined()) {
            dstRow.SetPreviousIKS(prevIKS.GetRef());
        }
        if (currIKS.Defined()) {
            dstRow.SetIKS(currIKS.GetRef());
        }
        dstRow.SetMainMirror(mainMirror);
        dstRow.SetOwner(owner);
        output->AddRow(dstRow);
    }
};

REGISTER_REDUCER(TContentPrepareReducer)

TInputTag<NProto::TIKSWithContent> IKSWithContentInputTag               (1);
TInputTag<NJupiter::TContentAttrsForWebmaster> ContentAttrsInputTag     (2);
TOutputTag<NProto::TIKSWithContent> IKSWithContentOutputTag             (1);

struct TJoinJupiterContentReducer : public TTaggedReducer {
    void DoTagged(TTagedReader reader, TTagedWriter writer) override {
        TMaybe<NJupiter::TContentAttrsForWebmaster> jupiterContentAttrsRow = reader.GetSingleRowMaybe(ContentAttrsInputTag);
        if (!reader.IsValid()) {
            return;
        }

        TString titleRawUTF8;
        if (jupiterContentAttrsRow.Defined()) {
            titleRawUTF8 = jupiterContentAttrsRow.GetRef().GetTitleRawUTF8();
        }

        reader.SkipRows(ContentAttrsInputTag);
        for (NProto::TIKSWithContent contentIKSRow : reader.GetRows(IKSWithContentInputTag)) {
            contentIKSRow.SetHost(contentIKSRow.GetHost());
            contentIKSRow.SetPath(contentIKSRow.GetPath());
            contentIKSRow.SetMascotHost(contentIKSRow.GetMascotHost());
            contentIKSRow.SetMainMirror(contentIKSRow.GetMainMirror());
            contentIKSRow.SetOwner(contentIKSRow.GetOwner());
            contentIKSRow.SetIKS(contentIKSRow.GetIKS());
            contentIKSRow.SetPreviousIKS(contentIKSRow.GetPreviousIKS());
            contentIKSRow.SetTitle(titleRawUTF8);
            writer.AddRow(contentIKSRow, IKSWithContentOutputTag);
        }
    }
};

REGISTER_REDUCER(TJoinJupiterContentReducer)

//SortBy Key
struct TMirrorsMapper : public NYT::IMapper<NYT::TTableReader<NJupiter::THostMirror>, NYT::TTableWriter<NProto::TIKSMirror>> {
    Y_SAVELOAD_JOB(Hosts)

    TMirrorsMapper() = default;
    TMirrorsMapper(const THashSet<TString> &hosts)
        : Hosts(hosts)
    {
    }

    void Do(TReader *input, TWriter *output) override {
        NProto::TIKSMirror dstMsg;
        for (; input->IsValid(); input->Next()) {
            const auto &row = input->GetRow();
            if (Hosts.contains(row.GetHost()) && row.GetHost() != row.GetMainHost()) {
                dstMsg.SetHost(row.GetHost());
                dstMsg.SetMainMirror(row.GetMainHost());
                output->AddRow(dstMsg);
            }
        }
    }

public:
    THashSet<TString> Hosts;
};

REGISTER_MAPPER(TMirrorsMapper)

//ReduceBy Owner
struct TOwnersCYReducer : public NYT::IReducer<NYT::TTableReader<NMascot::TXDump>, NYT::TTableWriter<NProto::TOwnersCY>> {
    void Start(TWriter *) override {
        LoadOwnersAfterJul21(Owners);
    }

    void Do(TReader *input, TWriter *output) override {
        const TString owner = input->GetRow().GetOwner();
        TVector<TStringBuf> ownerParts;
        StringSplitter(owner).Split('.').AddTo(&ownerParts);
        bool found = false;
        for (int i = 0; i < (int)ownerParts.size() - 1; i++) {
            const TStringBuf leveledDomain(ownerParts[i].begin(), owner.end());
            if (Owners.contains(leveledDomain)) {
                found = true;
                break;
            }
        }
        if (!found) {
            return;
        }

        TCounter counter;
        for (; input->IsValid(); input->Next()) {
            counter.Add(input->GetRow().Getcy_flt_100());
        }

        NProto::TOwnersCY dstMsg;
        dstMsg.SetHost(owner);
        dstMsg.Setcy_flt_100(counter.Get());
        output->AddRow(dstMsg);
    }

public:
        //THashMap<TString, size_t> ownersCY;
    THashSet<TString> Owners;
};

REGISTER_REDUCER(TOwnersCYReducer)

void LoadMascotTables(NYT::IClientBasePtr client, TDeque<NYTUtils::TTableInfo> &mascotTables) {
    NYTUtils::GetTableList(client, TConfig::CInstance().TABLE_ROOT_MASCOT_RANKS_HISTORY, mascotTables);
    std::sort(mascotTables.rbegin(), mascotTables.rend(), NYTUtils::TTableInfo::TNameLess());
    if (mascotTables.size() > AVERAGING_PERIOD_WEEKS) {
        mascotTables.resize(AVERAGING_PERIOD_WEEKS);
    }
    if (mascotTables.empty()) {
        ythrow yexception() << "There is no Mascot source tables";
    }
}

void LoadMascotOwnerUrls(NYT::IClientBasePtr client, THashSet<TString> &owners) {
    TDeque<NYTUtils::TTableInfo> mascotTables;
    LoadMascotTables(client, mascotTables);

    auto reader = TTable<NMascot::TXDump>(client, mascotTables[0].Name).SelectFields({"Owner"}).GetReader();
    for (; reader->IsValid(); reader->Next()) {
        auto &row = reader->GetRow();
        owners.insert(TIKSCanonizer::GetOwnerUrl(row.GetOwner()));
    }
}

void LoadMirrors(NYT::IClientBasePtr client, THashMap<TString, TString> &mirrors) {
    auto reader = TTable<NProto::TIKSMirror>(client, TConfig::CInstance().TABLE_IKS_MIRRORS_STAGE).GetReader();
    for (; reader->IsValid(); reader->Next()) {
        auto &row = reader->GetRow();
        mirrors[row.GetHost()] = row.GetMainMirror();
    }
}

void LoadTIc(NYT::IClientBasePtr client, THashMap<TString, size_t> &TIc) {
    auto reader = TTable<NProto::TTIcData>(client, TConfig::CInstance().TABLE_SOURCE_TIC_DATA).GetReader();
    for (; reader->IsValid(); reader->Next()) {
        auto &row = reader->GetRow();
        TIc[row.GetHost()] = row.GetTIc();
    }
}

void LoadPenalties(NYT::IClientBasePtr client, THashMap<TString, float> &penalties) {
    auto reader = TTable<NProto::TIKSPenalty>(client, TConfig::CInstance().TABLE_IKS_PENALTY).SelectFields({"Host", "Penalty"}).GetReader();
    for (; reader->IsValid(); reader->Next()) {
        auto &row = reader->GetRow();
        penalties[row.GetHost()] = row.GetPenalty();
    }
}

void LoadDeadOwners(NYT::IClientBasePtr client, THashSet<TString> &deadOwners) {
    auto reader = TTable<NProto::TIKSVitality>(client, TConfig::CInstance().TABLE_IKS_VITALITY_STATE).SelectFields({"Host", "IsDead"}).GetReader();
    for (; reader->IsValid(); reader->Next()) {
        auto &row = reader->GetRow();
        if (row.GetIsDead()) {
            deadOwners.insert(row.GetHost());
        }
    }
}

void LoadRadarTables(NYT::IClientBasePtr clientRadar, TDeque<TString> &inputTables) {
    TMap<TString, TString> lastTables;
    TDeque<NYTUtils::TTableInfo> tableList;
    NYTUtils::GetTableList(clientRadar, TConfig::CInstance().TABLE_SOURCE_RADAR_MAU_ROOT, tableList, Max<size_t>());
    std::sort(tableList.rbegin(), tableList.rend(), NYTUtils::TTableInfo::TNameLess());

    const TString currentRangeConfigStr = TMonthRangeConfig::Today().Name();

    TString latestTable;
    TRegularExpression regex("^(\\d{4}-\\d{2}-\\d{2})_(\\d{13})$");
    for (const NYTUtils::TTableInfo &table : tableList) {
        TVector<TString> hits;
        if (regex.GetMatches(NYTUtils::GetTableName(table.Name), hits) == 2) {
            const TDate date(hits[0], "%Y-%m-%d");
            if (date.GetStart() < 1541019600) { /*20181101*/
                continue;
            }
            TMonthRangeConfig rangeConfig = TMonthRangeConfig::Next(date);
            const TString rangeConfigStr = rangeConfig.Name();
            if (rangeConfigStr == currentRangeConfigStr) {
                continue;
            }
            lastTables[rangeConfigStr] = Max(lastTables[rangeConfigStr], table.Name);
        }
    }

    for (auto it = lastTables.rbegin(); it != lastTables.rend() && inputTables.size() < 12; ++it) {
        inputTables.push_back(it->second);
    }
}

void LoadRadarMeanMAU(NYT::IClientBasePtr clientRadar, const TDeque<TString> &inputTables) {
    const THashMap<TString, TString> domains = {
        {"google.ru", "mau"},
        {"yandex.ru", "metr_mau"},
    };

    for (const TString &inputTable : inputTables) {
        NYT::TRichYPath path(inputTable);
        for (const auto &obj : domains) {
            path.AddRange(NYT::TReadRange().Exact(NYT::TReadLimit().Key(NYT::TKey(obj.first))));
        }

        auto reader = clientRadar->CreateTableReader<NYT::TNode>(path);
        for (; reader->IsValid(); reader->Next()) {
            const auto &row = reader->GetRow();
            const TString domain = row["domain"].AsString();
            const TString field = domains.at(domain);
            Cout << inputTable << "\t" << domain << "\t" << row[field].AsInt64() << Endl;
        }
    }
}

void LoadYandexAchievementsDomains(NYT::IClientBasePtr client, THashSet<TString> &yandexDomains) {
    const TString state = NYTUtils::GetAttr(client, TConfig::CInstance().TABLE_SOURCE_ACHIEVEMENTS_ROOT, "production_state").AsString();
    const TString table = NYTUtils::JoinPath(TConfig::CInstance().TABLE_SOURCE_ACHIEVEMENTS_ROOT, state, "yandex");
    auto reader = TTable<NProto::TAchievementsYandex>(client, table).GetReader();
    for (; reader->IsValid(); reader->Next()) {
        yandexDomains.insert(reader->GetRow().Getowner());
    }
}

void LoadFixes(NYT::IClientBasePtr client, THashMap<TString, TPatch> &fixes) {
    const char *D_GOOGLE_RU     = "google.ru";
    const char *D_GOOGLE_COM    = "google.com";
    const char *D_YANDEX_RU     = "yandex.ru";

    TDeque<NYTUtils::TTableInfo> mascotTables;
    LoadMascotTables(client, mascotTables);

    NYT::TRichYPath path(mascotTables[0].Name);
    path.AddRange(NYT::TReadRange().Exact(NYT::TReadLimit().Key(NYT::TKey(D_GOOGLE_COM))));
    path.AddRange(NYT::TReadRange().Exact(NYT::TReadLimit().Key(NYT::TKey(D_GOOGLE_RU))));
    path.AddRange(NYT::TReadRange().Exact(NYT::TReadLimit().Key(NYT::TKey(D_YANDEX_RU))));

    THashMap<TString, float> radarMAU = {
        {D_GOOGLE_RU, 140000000.0},
        {D_YANDEX_RU, 309434200.0},
    };

    THashMap<TString, float> mascotVisitors;
    auto reader = TTable<NMascot::TXDump>(client, path).GetReader();
    for (; reader->IsValid(); reader->Next()) {
        const NMascot::TXDump &row = reader->GetRow();
        mascotVisitors[row.GetOwner()] = row.GetVisitors();
    }

    const float radarGoogleShare = radarMAU.at(D_GOOGLE_RU) / (radarMAU.at(D_YANDEX_RU) + radarMAU.at(D_GOOGLE_RU));
    const float radarYandexShare = radarMAU.at(D_YANDEX_RU) / (radarMAU.at(D_YANDEX_RU) + radarMAU.at(D_GOOGLE_RU));

    const float mascotGoogleComVisitors = mascotVisitors.at(D_GOOGLE_COM);
    const float mascotGoogleRuVisitors = mascotVisitors.at(D_GOOGLE_RU);
    const float mascotGoogleVisitors = mascotGoogleComVisitors + mascotGoogleRuVisitors;

    const float mascotGoogleComShare = mascotGoogleComVisitors / mascotGoogleVisitors;
    const float mascotGoogleRuShare = mascotGoogleRuVisitors / mascotGoogleVisitors;

    const float rebuiltGoogleVisitors = mascotVisitors.at(D_YANDEX_RU) * radarGoogleShare / radarYandexShare;
    const float rebuiltGoogleComVisitors = rebuiltGoogleVisitors * mascotGoogleComShare;
    const float rebuiltGoogleRuVisitors  = rebuiltGoogleVisitors * mascotGoogleRuShare;

    fixes[D_GOOGLE_COM] = TPatch(rebuiltGoogleComVisitors, 0, 0);
    fixes[D_GOOGLE_RU] = TPatch(rebuiltGoogleRuVisitors, 0, 0);

/*
    // fixme: source should be loaded from the latest table (when it will be ready)
    // https://yt.yandex-team.ru/hahn/navigation?path=//home/radar_top_sites/production/export/report/sites
    const static THashMap<TString, TPatch> Fixes = {
        //{ "google.com", TPatch(60954097, 35909068, 763007) },
        //{ "google.ru",  TPatch(66597122, 39233468, 833645) },
        //{ "google.com", TPatch(76698432, 61898023, 1315230) },
        //{ "google.ru",  TPatch(50852787, 41039782, 872027) },
        //{ "google.com", TPatch(165345366, 0, 0) },
        { "google.com", TPatch(70573600, 0, 0) },
        { "google.ru", TPatch(90162260, 0, 0) },
    };
*/
}

struct TSubOwner {
    bool CanBeFixed = true;
    TString Owner;
    size_t CY = 0;
    float CYShare = 0;
    size_t NewCY = 0; //calculated on frozen cy_flt_100
};

void LoadSubOwnersCYShare(NYT::IClientBasePtr client, THashMap<TString, TSubOwner> &subOwners) {
    const auto &cfg = TConfig::CInstance();

    THashMap<TString, size_t> ownersCY;
    THashSet<TString> owners;
    LoadOwnersAfterJul21(owners);

    auto reader = TTable<NProto::TOwnersCY>(client, cfg.TABLE_IKS_OWNERS_CY).GetReader();
    for (; reader->IsValid(); reader->Next()) {
        auto &row = reader->GetRow();
        const TString domain = row.GetHost();

        TVector<TStringBuf> domainParts;
        StringSplitter(domain).Split('.').AddTo(&domainParts);
        subOwners[domain].CY = row.Getcy_flt_100();

        for (int i = 0; i < (int)domainParts.size() - 1; i++) {
            const TStringBuf leveledDomain(domainParts[i].begin(), domain.end());
            if (owners.contains(leveledDomain)) {
                ownersCY[leveledDomain] += row.Getcy_flt_100();
                subOwners[domain].Owner = leveledDomain;
                break;
            }
        }
    }

    for (auto &obj : subOwners) {
        auto &subOwner = obj.second;
        const TString &owner = subOwner.Owner;
        subOwner.CanBeFixed = obj.first != owner;
        subOwner.CYShare = static_cast<float>(subOwner.CY) / static_cast<float>(ownersCY.at(owner));
    }
}

void LoadSubOwnersCYFixes(NYT::IClientBasePtr client, THashMap<TString, size_t> &subOwnersCYFixes) {
    const auto &cfg = TConfig::CInstance();

    THashMap<TString, TSubOwner> subOwners;
    LoadSubOwnersCYShare(client, subOwners);

    THashSet<TString> ownersAfterJul21;
    LoadOwnersAfterJul21(ownersAfterJul21);

    NYT::TRichYPath frozenCY(cfg.TABLE_SOURCE_FROZEN_CY_DATA);
    for (const TString &key : ownersAfterJul21) {
        frozenCY.AddRange(NYT::TReadRange().Exact(NYT::TReadLimit().Key(NYT::TKey(key))));
    }

    THashMap<TString, size_t> ownersCY;
    auto reader = TTable<NMascot::TXDump>(client, frozenCY).GetReader();
    for (; reader->IsValid(); reader->Next()) {
        auto &row = reader->GetRow();
        ownersCY[row.GetOwner()] = row.Getcy_flt_100();
    }

    for (auto &obj : subOwners) {
        TSubOwner &subOwner = obj.second;
        const TString &owner = subOwner.Owner;

        if (ownersCY.contains(owner)) {
            subOwner.NewCY = ownersCY.at(owner) * obj.second.CYShare;
            if (subOwner.CanBeFixed) {
                subOwnersCYFixes[obj.first] = subOwner.NewCY;
            }
        }
    }
}

unsigned GetWeekDay(time_t ts) {
    struct tm *ti = localtime(&ts);
    return ti->tm_wday;
}

TInstant GetStartOfWeek(time_t ts = Now().TimeT()) {
    const static int WEEK_START_OFFSET_DAYS = 6;
    int weekDay = (GetWeekDay(ts) + WEEK_START_OFFSET_DAYS) % 7;
    return TInstant::Seconds(TDate(ts).GetStart() - weekDay * 86400);
}

void UpdateLatestMascotSource(NYT::IClientBasePtr client) {
    const TString inputTable = TConfig::CInstance().TABLE_SOURCE_MASCOT_RANKS;
    const TString outputTableDateStr = GetStartOfWeek().FormatLocalTime("%Y%m%d");
    const TString outputTable = NYTUtils::JoinPath(TConfig::CInstance().TABLE_ROOT_MASCOT_RANKS_HISTORY, outputTableDateStr);

    TMapCombineReduceCmd<TAntispamRanksMapper, TAntispamRanksReducer, TAntispamRanksReducer>(client, new TAntispamRanksMapper, nullptr, new TAntispamRanksReducer)
        .OperationWeight(TConfig::CInstance().OPERATION_WEIGHT)
        .Input(TTable<NMascot::TXDump>(client, inputTable))
        .Output(TTable<NMascot::TXDump>(client, outputTable))
        .ReduceBy({"Owner"})
        .Do()
    ;

    TSortCmd<NMascot::TXDump>(client, TTable<NMascot::TXDump>(client, outputTable)
        .SetCompressionCodec(ECompressionCodec::BROTLI_6)
        .SetErasureCodec(EErasureCodec::LRC_12_2_2)
    )
        .OperationWeight(TConfig::CInstance().OPERATION_WEIGHT)
        .By({"Owner"})
        .Do()
    ;
}

void UpdateIKS(NYT::IClientBasePtr client, const TString &iksOutput, bool loadDeadOwners) {
    const auto &cfg = TConfig::CInstance();

    THashMap<TString, size_t> TIc;
    LoadTIc(client, TIc);

    THashMap<TString, float> penalties;
    LoadPenalties(client, penalties);

    THashSet<TString> deadOwners;
    if (loadDeadOwners) {
        LoadDeadOwners(client, deadOwners);
    }

    TDeque<NYTUtils::TTableInfo> mascotTables;
    LoadMascotTables(client, mascotTables);

    TDeque<TTable<NMascot::TXDump>> inputMascotTables;
    TDeque<time_t> tableConfig;
    for (auto &table : mascotTables) {
        tableConfig.push_back(str2date(NYTUtils::GetTableName(table.Name)));
        inputMascotTables.emplace_back(client, DebugPath(table.Name));
    }

    THashMap<TString, TString> mirrors;
    LoadMirrors(client, mirrors);

    THashMap<TString, TPatch> fixes;
    LoadFixes(client, fixes);

    THashMap<TString, size_t> subOwnersCYFixes;
    LoadSubOwnersCYFixes(client, subOwnersCYFixes);

    TReduceCmd<TCalcScoreReducer>(client, new TCalcScoreReducer(tableConfig, TIc, penalties, deadOwners, mirrors, fixes, subOwnersCYFixes))
        .OperationWeight(TConfig::CInstance().OPERATION_WEIGHT)
        .Input(TTable<NProto::TFrozenCY>(client, cfg.TABLE_SOURCE_FROZEN_CY_DATA), FrozenCYInputTag)
        .Inputs(inputMascotTables, MascotTableInputTag)
        .Output(TTable<NProto::TIKS>(client, iksOutput).AsSortedOutput({"Host"}), TIKSOutoutTag)
        .MemoryLimit(4_GBs)
        .ReduceBy({"Owner"})
        .Do()
    ;

    TSortCmd<NProto::TIKS>(client, TTable<NProto::TIKS>(client, iksOutput))
        .OperationWeight(TConfig::CInstance().OPERATION_WEIGHT)
        .By({"Host"})
        .Do()
    ;

    NYTUtils::SetAttr(client, iksOutput, ATTR_UPDATE_TIME, Now().TimeT());
}

void UpdateIKS(NYT::IClientBasePtr client) {
    const auto &cfg = TConfig::CInstance();
    UpdateIKS(client, cfg.TABLE_IKS_DATA_STAGE_FULL, true);

    TRandomSampler sampler;
    sampler.Load(client, cfg.TABLE_IKS_DATA_ACCEPTED);

    TReduceCmd<TRandomIKSScoreMergeReducer>(client, new TRandomIKSScoreMergeReducer(sampler))
        .OperationWeight(TConfig::CInstance().OPERATION_WEIGHT)
        .Input(TTable<NProto::TIKS>(client, cfg.TABLE_IKS_DATA_ACCEPTED), IKSAcceptedInputTag)
        .Input(TTable<NProto::TIKS>(client, cfg.TABLE_IKS_DATA_STAGE_FULL), IKSStageInputTag)
        .Output(TTable<NProto::TIKS>(client, cfg.TABLE_IKS_DATA_STAGE).AsSortedOutput({"Host"}), IKSStageOutputTag)
        .ReduceBy({"Host"})
        .Do()
    ;

    sampler.SaveNext(client, cfg.TABLE_IKS_DATA_STAGE);
    NYTUtils::SetAttr(client, cfg.TABLE_IKS_DATA_STAGE, ATTR_UPDATE_TIME, Now().TimeT());
}

void UpdateIKSForVitality(NYT::IClientBasePtr client) {
    UpdateIKS(client, TConfig::CInstance().TABLE_IKS_VITALITY_IKS, false);
}

TString GetPrevAcceptedIKSTable(NYT::IClientBasePtr client) {
    TDeque<NYTUtils::TTableInfo> tables;
    NYTUtils::GetTableList(client, TConfig::CInstance().TABLE_ROOT_IKS_PREV, tables);
    if (tables.empty()) {
        ythrow yexception() << "There is no previous table";
    }
    std::sort(tables.rbegin(), tables.rend(), NYTUtils::TTableInfo::TNameLess());
    return tables[0].Name;
}

void UpdateIKSContent(NYT::IClientBasePtr client) {
    NYT::ITransactionPtr tx = client->StartTransaction();
    const TString acceptedIKSTable = TConfig::CInstance().TABLE_IKS_DATA_ACCEPTED;
    const TString prevAcceptedIKSTable = GetPrevAcceptedIKSTable(tx);
    const TString contentIKSTable = TConfig::CInstance().TABLE_IKS_CONTENT;
    const NYT::TSortColumns KEYS_JUPITER_CONTENT_ATTRS = {"Host", "Path"};

    TReduceCmd<TContentPrepareReducer>(tx, new TContentPrepareReducer)
        .OperationWeight(TConfig::CInstance().OPERATION_WEIGHT)
        .Input(TTable<NProto::TIKS>(tx, prevAcceptedIKSTable))
        .Input(TTable<NProto::TIKS>(tx, acceptedIKSTable))
        .Output(TTable<NProto::TIKSWithContent>(tx, contentIKSTable))
        .ReduceBy({"Host"})
        .Do()
    ;

    TSortCmd<NProto::TIKSWithContent>(tx)
        .OperationWeight(TConfig::CInstance().OPERATION_WEIGHT)
        .Input(TTable<NProto::TIKSWithContent>(tx, contentIKSTable))
        .Output(TTable<NProto::TIKSWithContent>(tx, contentIKSTable))
        .By(KEYS_JUPITER_CONTENT_ATTRS)
        .Do()
    ;

    TJoinCmd<TJoinJupiterContentReducer>(tx)
        .OperationWeight(TConfig::CInstance().OPERATION_WEIGHT)
        .Input(TTable<NJupiter::TContentAttrsForWebmaster>(tx, GetJupiterContentAttrsInProdTable(tx)).AsForeign(), ContentAttrsInputTag)
        .Input(TTable<NProto::TIKSWithContent>(tx, contentIKSTable), IKSWithContentInputTag)
        .Output(TTable<NProto::TIKSWithContent>(tx, contentIKSTable).AsSortedOutput(KEYS_JUPITER_CONTENT_ATTRS), IKSWithContentOutputTag)
        .JoinBy(KEYS_JUPITER_CONTENT_ATTRS)
        .Do()
    ;

    TSortCmd<NProto::TIKSWithContent>(tx)
        .OperationWeight(TConfig::CInstance().OPERATION_WEIGHT)
        .Input(TTable<NProto::TIKSWithContent>(tx, contentIKSTable))
        .Output(TTable<NProto::TIKSWithContent>(tx, contentIKSTable))
        .By(KEYS_JUPITER_CONTENT_ATTRS)
        .Do()
    ;

    const time_t updateTime = NYTUtils::GetAttr(tx, acceptedIKSTable, ATTR_UPDATE_TIME).AsInt64();
    NYTUtils::SetAttr(tx, contentIKSTable, ATTR_UPDATE_TIME, updateTime);
    tx->Commit();
}

void UpdateMirrors(NYT::IClientBasePtr client) {
    THashSet<TString> ownerUrls;
    LoadMascotOwnerUrls(client, ownerUrls);

    TMapCmd<TMirrorsMapper>(client, new TMirrorsMapper(ownerUrls))
        .OperationWeight(TConfig::CInstance().OPERATION_WEIGHT)
        .Input(TTable<NJupiter::THostMirror>(client, GetJupiterMirrorsInProdTable(client)))
        .Output(TTable<NProto::TIKSMirror>(client, TConfig::CInstance().TABLE_IKS_MIRRORS_STAGE))
        .MemoryLimit(1_GBs)
        .Do()
    ;

    TSortCmd<NProto::TIKSMirror>(client)
        .OperationWeight(TConfig::CInstance().OPERATION_WEIGHT)
        .Input(TTable<NProto::TIKSMirror>(client, TConfig::CInstance().TABLE_IKS_MIRRORS_STAGE))
        .Output(TTable<NProto::TIKSMirror>(client, TConfig::CInstance().TABLE_IKS_MIRRORS_STAGE))
        .By({"Host"})
        .Do()
    ;
}

void BuildCYHash(NYT::IClientBasePtr client) {
    THashMap<TString, size_t> ownerCY;
    LoadOwnerCY(client, TConfig::CInstance().TABLE_IKS_DATA_STAGE, ownerCY);
    using TName2CY = THashMap<const char *, ui32, ci_hash32, ci_equal_to>;
    segmented_string_pool stringPool;
    TName2CY name2CY;
    for (const auto &obj : ownerCY) {
        name2CY[stringPool.append(obj.first.data())] = obj.second;
    }
    SaveHashToFile32(name2CY, TConfig::CInstance().FILE_CY_HASH_LOCAL.data());
    NYTUtils::UploadFile(client, TConfig::CInstance().FILE_CY_HASH_LOCAL, TConfig::CInstance().FILE_CY_HASH_YT_STAGE);
    const time_t updateTime = NYTUtils::GetAttr(client, TConfig::CInstance().TABLE_IKS_DATA_STAGE, ATTR_UPDATE_TIME).AsInt64();
    NYTUtils::SetAttr(client, TConfig::CInstance().FILE_CY_HASH_YT_STAGE, ATTR_UPDATE_TIME, updateTime);
}

void UpdateOwnersCY(NYT::IClientBasePtr client) {
    const auto &cfg = TConfig::CInstance();

    TDeque<NYTUtils::TTableInfo> mascotTables;
    LoadMascotTables(client, mascotTables);

    TDeque<TTable<NMascot::TXDump>> inputMascotTables;
    for (auto &tableInfo : mascotTables) {
        auto table = TTable<NMascot::TXDump>(client, DebugPath(tableInfo.Name));
        inputMascotTables.push_back(table.SelectFields({"Owner", "cy_flt_100"}));
    }

    TReduceCmd<TOwnersCYReducer>(client)
        .OperationWeight(cfg.OPERATION_WEIGHT)
        .Inputs(inputMascotTables)
        .Output(TTable<NProto::TOwnersCY>(client, cfg.TABLE_IKS_OWNERS_CY).AsSortedOutput({"Host"}))
        .ReduceBy({"Owner"})
        .Do()
    ;

    TSortCmd<NProto::TOwnersCY>(client, TTable<NProto::TOwnersCY>(client, cfg.TABLE_IKS_OWNERS_CY))
        .OperationWeight(cfg.OPERATION_WEIGHT)
        .By({"Host"})
        .Do()
    ;
}

int UpdateSources(int, const char **) {
    NYT::IClientPtr client = NYT::CreateClient(TConfig::CInstance().MR_SERVER_HOST);
    NYT::ITransactionPtr tx = client->StartTransaction();
    UpdateLatestMascotSource(tx);
    tx->Commit();
    return 0;
}

int UpdateStage(int, const char **) {
    NYT::IClientPtr client = NYT::CreateClient(TConfig::CInstance().MR_SERVER_HOST);
    NYT::ITransactionPtr tx = client->StartTransaction();
    UpdateMirrors(tx);
    UpdatePenaltyV2(tx);
    UpdateIKSForVitality(tx);
    UpdateVitality(tx);
    UpdateOwnersCY(tx);
    UpdateIKS(tx);
    tx->Commit();
    return 0;
}

int StageToProd(int argc, const char **argv) {
    NLastGetopt::TOpts opts;
    opts.AddLongOption("apply", "Apply described changes").NoArgument();
    NLastGetopt::TOptsParseResult res(&opts, argc, argv);
    const bool applyChanges = res.Has("apply");

    NYT::IClientPtr client = NYT::CreateClient(TConfig::CInstance().MR_SERVER_HOST);
    NYT::ITransactionPtr tx = client->StartTransaction();

    BuildCYHash(tx);

    TTable<NProto::TIKS> prodTable(tx, TConfig::CInstance().TABLE_IKS_DATA_ACCEPTED);
    const time_t prodTimestamp = prodTable.GetAttribute<i64>(ATTR_UPDATE_TIME);

    TTable<NProto::TIKS> stageTable(tx, TConfig::CInstance().TABLE_IKS_DATA_STAGE);
    const time_t stageTimestamp = stageTable.GetAttribute<i64>(ATTR_UPDATE_TIME);

    const TString todayStr = TDate::Today().ToStroka();
    const TString newPrevIKSName = NYTUtils::JoinPath(TConfig::CInstance().TABLE_ROOT_IKS_PREV, todayStr);
    const TString newPrevCYHashName = NYTUtils::JoinPath(TConfig::CInstance().FILE_CY_HASH_YT_PREV, todayStr);

    LOG_INFO("CURRENT_PROD: %s (%ld)", NUtils::Date2StrTZ(prodTimestamp).c_str(), prodTimestamp);
    LOG_INFO("Move to PREV: %s -> %s", TConfig::CInstance().TABLE_IKS_DATA_ACCEPTED.c_str(), newPrevIKSName.c_str());
    LOG_INFO("Move to PREV: %s -> %s", TConfig::CInstance().FILE_CY_HASH_YT_ACCEPTED.c_str(), newPrevCYHashName.c_str());

    LOG_INFO("NEW_PROD: %s (%ld)", NUtils::Date2StrTZ(stageTimestamp).c_str(), stageTimestamp);
    LOG_INFO("Copy to PROD: %s -> %s", TConfig::CInstance().TABLE_IKS_DATA_STAGE.c_str(), TConfig::CInstance().TABLE_IKS_DATA_ACCEPTED.c_str());
    LOG_INFO("Copy to PROD: %s -> %s", TConfig::CInstance().FILE_CY_HASH_YT_STAGE.c_str(), TConfig::CInstance().FILE_CY_HASH_YT_ACCEPTED.c_str());

    if (applyChanges) {
        tx->Move(TConfig::CInstance().TABLE_IKS_DATA_ACCEPTED, newPrevIKSName);
        tx->Move(TConfig::CInstance().FILE_CY_HASH_YT_ACCEPTED, newPrevCYHashName);
        tx->Copy(TConfig::CInstance().TABLE_IKS_DATA_STAGE, TConfig::CInstance().TABLE_IKS_DATA_ACCEPTED);
        tx->Copy(TConfig::CInstance().TABLE_IKS_DATA_STAGE, TConfig::CInstance().TABLE_IKS_EXPORT, NYT::TCopyOptions().Force(true));
        tx->Copy(TConfig::CInstance().FILE_CY_HASH_YT_STAGE, TConfig::CInstance().FILE_CY_HASH_YT_ACCEPTED);
        LOG_INFO("Changes applied");
    }

    LOG_INFO("Done");

    tx->Commit();
    return 0;
}

int UpdateContent(int, const char **) {
    NYT::IClientPtr client = NYT::CreateClient(TConfig::CInstance().MR_SERVER_HOST);
    NYT::ITransactionPtr tx = client->StartTransaction();
    UpdateIKSContent(tx);
    tx->Commit();
    return 0;
}

int Debug(int, const char **) {
    NYT::IClientPtr client = NYT::CreateClient(TConfig::CInstance().MR_SERVER_HOST);

    //UpdatePenaltyV2(client);

    THashMap<TString, float> penalties;
    {
        auto reader = TTable<NProto::TIKSPenalty>(client, DebugPath(TConfig::CInstance().TABLE_IKS_PENALTY)).SelectFields({"Host", "Penalty"}).GetReader();
        for (; reader->IsValid(); reader->Next()) {
            auto &row = reader->GetRow();
            penalties[row.GetHost()] = row.GetPenalty();
        }
    }

    const TVector<TString> dates = {
        "20190401",
        "20190501",
    };

    THashMap<TString, TCounters> ownersCounters;
    const time_t now = Now().TimeT();
    for (const TString& dateStr : dates) {
        const NYT::TRichYPath path = DebugPath(NYTUtils::JoinPath(TConfig::CInstance().TABLE_ROOT_MASCOT_RANKS_HISTORY, dateStr));
        auto reader = TTable<NMascot::TXDump>(client, path).GetReader();
        for (; reader->IsValid(); reader->Next()) {
            auto &row = reader->GetRow();
            ownersCounters[row.GetOwner()].Add(now, row);
        }
    }

    Cout << "Dates\t" << JoinSeq(", ", dates) << Endl;
    for (const auto &obj : ownersCounters) {
        const TString &sourceOwner = obj.first;
        const TCounters &counters = obj.second;
        const float rawIks = GetRawIKS(counters);
        const float scaledIks = RemapIKS(rawIks);
        const float penalty = penalties.contains(sourceOwner) ? penalties.at(sourceOwner) : 0.0f;
        const float penaltyMultiplier = GetPenaltyMultiplier(penalty);
        const float orgRatingMultiplier = GetOrgRatingMultiplier(counters);
        const float penalizedIks = scaledIks * penaltyMultiplier * orgRatingMultiplier;

        const float CY = counters.CY100.Get(); //https://st.yandex-team.ru/WMC-7417
        const float CYDamper = (std::pow(CY, 3.0f) / (std::pow(CY, 3.0f) + 500.0f));

        Cout << "Owner\t"               << sourceOwner << Endl;
        Cout << "IKSRaw\t"              << rawIks << Endl;
        Cout << "IKSScaled\t"           << scaledIks << Endl;
        Cout << "IKSNotPenalized\t"     << RoundIKSMin(scaledIks) << Endl;
        Cout << "IKS\t"                 << RoundIKSMin(penalizedIks) << Endl;
        Cout << "Sources\t"             << counters.GetCount() << Endl;
        Cout << "RatingMultiplier\t"    << orgRatingMultiplier << Endl;
        Cout << "CY = " << CY << Endl;
        Cout << "CYDamper = " << CYDamper << Endl;

        Cout << "(1.0f + log(counters.AvgMoreNSecVisits.Get() * counters.Visitors.Get() + 1.0f)) = " << (1.0f + log(counters.AvgMoreNSecVisits.Get() * counters.Visitors.Get() + 1.0f)) << Endl;
        Cout << "(1.0f + log(Min(CY * 42.0f, 300000.0f) + 1.0f) * CYDamper) = " << (1.0f + log(Min(CY * 42.0f, 300000.0f) + 1.0f) * CYDamper) << Endl;
        Cout << "(1.0f + log(counters.GV3.Get() + 1.0f)) = " << (1.0f + log(counters.GV3.Get() + 1.0f)) << Endl;
    }

    return 0;
}

} //namespace NIks
} //namespace NWebmaster

int main(int argc, const char **argv) {
    using namespace NWebmaster;
    using namespace NWebmaster::NIks;
    NYT::Initialize(argc, argv);

    TModChooser modChooser;
    modChooser.AddMode("UpdateSources", UpdateSources, "Update Mascot sources");
    modChooser.AddMode("UpdateStage", UpdateStage, "Update IKS stage version");
    modChooser.AddMode("StageToProd", StageToProd, "Push stage IKS to production");
    modChooser.AddMode("UpdateContent", UpdateContent, "Update \"content\" table");
    modChooser.AddMode("Debug", Debug, "Debug mode");
    return modChooser.Run(argc, argv);

    //NYT::IClientPtr client = NYT::CreateClient(TConfig::CInstance().MR_SERVER_HOST);
    //THashSet<TString> yandexDomains;
    //LoadYandexAchievementsDomains(client, yandexDomains);
    //Cout << yandexDomains.size() << Endl;
}
