#include <util/draft/date.h>
#include <util/generic/deque.h>
#include <util/generic/hash.h>
#include <util/generic/map.h>
#include <util/generic/hash_set.h>
#include <util/generic/set.h>
#include <util/stream/output.h>
#include <util/string/join.h>

#include <library/cpp/yson/node/node_io.h>

#include <kernel/yt/compression/compression.h>

#include <robot/library/yt/static/command.h>
#include <robot/library/yt/static/tags.h>

#include <wmconsole/version3/library/sanctions/sanctions.h>
#include <wmconsole/version3/processors/antiall/threats/threats-config.h>
#include <wmconsole/version3/processors/tools/IKS/conf/config.h>
#include <wmconsole/version3/processors/tools/IKS/protos/iks.pb.h>
#include <mapreduce/yt/interface/protos/yamr.pb.h>
#include <yweb/antispam/att/protos/owner.pb.h>

#include "penalty.h"

using namespace NJupiter;

namespace NWebmaster {
namespace NIks {

using TSpan = std::pair<time_t, time_t>;

bool IsSanctionVisibleInWebmaster(const TString &sanction, TString &threatName) {
    if (TSanctions::CInstance().GetThreatName(sanction, threatName)) {
        TThreatSettings threat;
        TThreatsConfig::CInstance().FindMatchingThreat(threatName, &threat);
        return threat.Show;
    } else {
        threatName = "NOT_FOUND";
    }
    return false;
}

bool In(const TSpan &range, time_t point) {
    return point >= range.first && point <= range.second;
}

void ExtractHistory(const THistory &historyMsg, THashMap<TString, TMultiMap<time_t, bool>> &events) {
    for (const auto &entry : historyMsg.GetHistory()) {
        for (const auto &sanction : entry.GetValue()) {
            if (!sanction.empty()) {
                events[sanction];
            }
        }
    }

    for (const auto &entry : historyMsg.GetHistory()) {
        const time_t timestamp = entry.GetTimestamp();
        const THashSet<TString> sanctions(entry.GetValue().begin(), entry.GetValue().end());
        for (auto &event : events) {
            const bool sanctionIsActive = sanctions.contains(event.first);
            event.second.emplace(timestamp, sanctionIsActive);
        }
    }
}

template<class FSanctionChecker>
void FinalizeSpans(THashMap<TString, TMultiMap<time_t, bool>> &events, TDeque<TSpan> &spans, THashMap<TString, TDeque<TSpan>> &spansBySanction, time_t lBound, time_t uBound, FSanctionChecker fSanctionChecker) {
    for (auto &event: events) {
        const auto &sanctionName = event.first;
        auto &entries = event.second;
        if (entries.rbegin()->second) {
            entries.emplace(uBound, false);
        }

        TSpan span;
        for (const auto &entry : entries) {
            if (entry.second) {
                if (!span.first) {
                    span.first = entry.first;
                }
                span.second = entry.first;
            } else if (span.second != 0) {
                span.second = entry.first;
                if (In(span, lBound)) {
                    span.first = lBound;
                }
                if (span.first >= lBound && span.first != span.second) {
                    TString threatName;
                    if (fSanctionChecker(sanctionName, threatName)) {
                        spans.emplace_back(span);
                    }
                    spansBySanction[sanctionName].emplace_back(span);
                }
                span = std::make_pair(0, 0);
            }
        }
        std::sort(spansBySanction[sanctionName].begin(), spansBySanction[sanctionName].end());
    }
    std::sort(spans.begin(), spans.end());
}

void ExtractRanges(const TDeque<TSpan> &spans, TDeque<TSpan> &ranges, time_t &lastTimestamp) {
    for (const TSpan &span : spans) {
        if (ranges.empty()) {
            lastTimestamp = span.first;
            ranges.emplace_back(span);
            continue;
        }

        if (In(ranges.back(), span.first)) {
            if (span.second > ranges.back().second) {
                ranges.back().second = span.second;
            }
        } else {
            ranges.emplace_back(span);
            lastTimestamp = std::max(lastTimestamp, span.second);
        }
    }
}

struct TIKSPenaltyMapper : public NYT::IMapper<NYT::TTableReader<NYT::TYamr>, NYT::TTableWriter<NProto::TIKSPenalty>> {
    const time_t PENALTY_RANGE_SECONDS = 30 * 6 * 86400;

    void Start(TWriter *) override {
        TThreatsConfig::CInstance();
        TSanctions::CInstance();
        UpperBound = Now().Seconds();
        LowerBound = UpperBound - PENALTY_RANGE_SECONDS;
    }

    template<class FSanctionChecker>
    static TString GetSanctionsDetailsStr(const THashMap<TString, TDeque<TSpan>> &spansBySanction, FSanctionChecker fSanctionChecker) {
        TStringBuilder result;
        bool rest = false;
        for (auto &obj : spansBySanction) {
            if (obj.second.empty()) {
                continue;
            }
            const TString &sanctionName = obj.first;
            TString threatName;
            const bool visible = fSanctionChecker(sanctionName, threatName);
            if (rest) {
                result << " / ";
            }
            result << sanctionName << "(" << threatName << ")=" << (ui32)visible;
            for (auto &span : obj.second) {
                result << " " << span.first << "-" << span.second;
            }
            rest = true;
        }
        return result;
    }

    static TString GetUsedRangesStr(const TDeque<TSpan> &ranges, size_t &seconds) {
        TStringBuilder result;
        seconds = 0;
        bool rest = false;
        for (auto &range : ranges) {
            if (rest) {
                result << " ";
            }
            result << range.first << "-" << range.second;
            seconds += range.second - range.first;
            rest = true;
        }
        return result;
    }

    void Do(TReader *input, TWriter *output) override {
        for (; input->IsValid(); input->Next()) {
            auto &row = input->GetRow();

            THistory historyMsg;
            Y_PROTOBUF_SUPPRESS_NODISCARD historyMsg.ParseFromArray(row.GetValue().data(), row.GetValue().size());

            THashMap<TString, TMultiMap<time_t, bool>> events;
            ExtractHistory(historyMsg, events);

            time_t lastSanctionTimestamp = 0;
            TDeque<TSpan> spans, ranges;
            THashMap<TString, TDeque<TSpan>> spansBySanction;
            FinalizeSpans(events, spans, spansBySanction, LowerBound, UpperBound, IsSanctionVisibleInWebmaster);
            ExtractRanges(spans, ranges, lastSanctionTimestamp);

            if (ranges.empty()) {
                continue;
            }

            NProto::TIKSPenalty outMsg;
            size_t secondsUnderSanctions;
            outMsg.SetUsedRanges(GetUsedRangesStr(ranges, secondsUnderSanctions));
            outMsg.SetHost(row.GetKey());
            outMsg.SetSecondsUnderSanctions(secondsUnderSanctions);
            outMsg.SetLastSanctionTimestamp(lastSanctionTimestamp);
            outMsg.SetDetails(GetSanctionsDetailsStr(spansBySanction, IsSanctionVisibleInWebmaster));
            outMsg.SetPenalty(std::min(static_cast<float>(secondsUnderSanctions) / static_cast<float>(PENALTY_RANGE_SECONDS), 1.0f));
            output->AddRow(outMsg);
        }
    }

public:
    time_t LowerBound = 0;
    time_t UpperBound = 0;
};

REGISTER_MAPPER(TIKSPenaltyMapper)

//SortBy owner
struct TIKSPenaltyV2Reducer : public NYT::IReducer<NYT::TTableReader<NProto::TIKSPenaltySource>, NYT::TTableWriter<NProto::TIKSPenalty>> {
    const constexpr static char *FORMAT = "%Y-%m-%d";
    const time_t PENALTY_RANGE_SECONDS = 30 * 6 * 86400;

    TIKSPenaltyV2Reducer() {
        const time_t upperBoundTimeT = Now().Seconds();
        const time_t lowerBoundTimeT = upperBoundTimeT - PENALTY_RANGE_SECONDS;
        UpperBound = TDate(upperBoundTimeT).ToStroka(FORMAT);
        LowerBound = TDate(lowerBoundTimeT).ToStroka(FORMAT);
        TThreatsConfig::CInstance();
    }

    void Do(TReader *input, TWriter *output) override {
        THashMap<TString, time_t> secondsUnderSanctionByWeek;
        THashMap<TString, time_t> secondsUnderSanctionByMethod;
        const auto firstRow = input->GetRow();

        for (; input->IsValid(); input->Next()) {
            const auto &row = input->GetRow();
            if (row.GetWeek() < LowerBound || row.GetWeek() > UpperBound) {
                continue;
            }

            TString threatName;
            TThreatSettings threat;
            if (TSanctions::CInstance().GetThreatName(row.GetMethod(), threatName)) {
                TThreatsConfig::CInstance().FindMatchingThreat(threatName, &threat);
            }

            if (!threat.Show) {
                continue;
            }

            secondsUnderSanctionByWeek[row.GetWeek()]       = Max<time_t>(secondsUnderSanctionByWeek[row.GetWeek()], row.GetTimeUnderSanction());
            secondsUnderSanctionByMethod[row.GetMethod()]   = Max<time_t>(secondsUnderSanctionByMethod[row.GetMethod()], row.GetTimeUnderSanction());
        }

        if (secondsUnderSanctionByWeek.empty()) {
            return;
        }

        TString minSanctionTimestampStr = secondsUnderSanctionByWeek.begin()->first;
        TString maxSanctionTimestampStr = secondsUnderSanctionByWeek.begin()->first;
        size_t secondsUnderSanctions = 0;
        for (const auto &obj : secondsUnderSanctionByWeek) {
            secondsUnderSanctions += obj.second;
            minSanctionTimestampStr = Min<TString>(obj.first, minSanctionTimestampStr);
            maxSanctionTimestampStr = Max<TString>(obj.first, maxSanctionTimestampStr);
        }

        TString details;
        for (const auto &obj : secondsUnderSanctionByMethod) {
            if (!details.empty()) {
                details += " ";
            }
            details += obj.first + "=" + ToString(obj.second);
        }

        NProto::TIKSPenalty outMsg;
        outMsg.SetUsedRanges(minSanctionTimestampStr + "_" + maxSanctionTimestampStr);
        outMsg.SetHost(firstRow.GetOwner());
        outMsg.SetSecondsUnderSanctions(secondsUnderSanctions);
        outMsg.SetLastSanctionTimestamp(TDate(maxSanctionTimestampStr, FORMAT).GetStart());
        outMsg.SetDetails(details);
        outMsg.SetPenalty(std::min(static_cast<float>(secondsUnderSanctions) / static_cast<float>(PENALTY_RANGE_SECONDS), 1.0f));
        output->AddRow(outMsg);
    }

public:
    TString LowerBound;
    TString UpperBound;
};

REGISTER_REDUCER(TIKSPenaltyV2Reducer)

void UpdatePenalty(NYT::IClientBasePtr client) {
    TMapCmd<TIKSPenaltyMapper>(client)
        .OperationWeight(TConfig::CInstance().OPERATION_WEIGHT)
        .Input(TTable<NYT::TYamr>(client, TConfig::CInstance().TABLE_SOURCE_IKS_PENALTY))
        .Output(TTable<NProto::TIKSPenalty>(client, TConfig::CInstance().TABLE_IKS_PENALTY).AsSortedOutput({"Host"}))
        .Ordered()
        .Do()
    ;
}

void UpdatePenaltyV2(NYT::IClientBasePtr client) {
    TReduceCmd<TIKSPenaltyV2Reducer>(client)
        .ReduceBy({"owner"})
        .SortBy({"owner", "method", "week"})
        .OperationWeight(TConfig::CInstance().OPERATION_WEIGHT)
        .Input(TTable<NProto::TIKSPenaltySource>(client, TConfig::CInstance().TABLE_SOURCE_IKS_PENALTY))
        .Output(TTable<NProto::TIKSPenalty>(client, TConfig::CInstance().TABLE_IKS_PENALTY).AsSortedOutput({"Host"}))
        .Do()
    ;
}

} //namespace NIks
} //namespace NWebmaster
