#include <util/datetime/cputimer.h>
#include <util/draft/datetime.h>
#include <util/generic/deque.h>
#include <util/generic/hash_set.h>
#include <util/generic/size_literals.h>
#include <util/generic/vector.h>
#include <util/string/reverse.h>

#include <library/cpp/containers/comptrie/comptrie.h>
#include <library/cpp/containers/comptrie/prefix_iterator.h>
#include <library/cpp/string_utils/url/url.h>

#include <mapreduce/yt/interface/client.h>
#include <mapreduce/yt/interface/protos/yamr.pb.h>

#include <robot/jupiter/protos/external/host_mirror.pb.h>
#include <robot/library/yt/static/command.h>
#include <robot/library/yt/static/tags.h>

#include <wmconsole/version3/library/jupiter/jupiter.h>
#include <wmconsole/version3/processors/user_sessions/exports/catalogia/protos/catalogia.pb.h>
#include <wmconsole/version3/processors/user_sessions/library/utils.h>
#include <wmconsole/version3/processors/user_sessions/protos/user_sessions.pb.h>
#include <wmconsole/version3/protos/queries2.pb.h>
#include <wmconsole/version3/wmcutil/log.h>
#include <wmconsole/version3/wmcutil/owners.h>
#include <wmconsole/version3/wmcutil/url.h>
#include <wmconsole/version3/wmcutil/yt/triggers.h>
#include <wmconsole/version3/wmcutil/yt/yt_utils.h>

#include "config.h"
#include "task_mirrors.h"
#include "utils.h"

namespace NWebmaster {
namespace NCatalogia {

using namespace NJupiter;

static const TInputTag<NProto::TCatalogiaHost> CatalogiaHostInputTag            (1);
static const TInputTag<NProto::TCatalogiaMirror> CatalogiaMirrorInputTag        (2);
static const TOutputTag<NProto::TCatalogiaMirror> CatalogiaMirrorOutputTag      (3);

struct TMirrorsMapper : public NYT::IMapper<NYT::TTableReader<NJupiter::THostMirror>, NYT::TTableWriter<NProto::TCatalogiaMirror>> {
    Y_SAVELOAD_JOB(TrieStream)

public:
    TMirrorsMapper() = default;
    TMirrorsMapper(const TVector<char> &trieStream)
        : TrieStream(trieStream)
    {
    }

public:
    void Start(TWriter* /*writer*/) override {
        Trie.Init(&TrieStream[0], TrieStream.size());
    }

    void Do(TReader *input, TWriter *output) override {
        NProto::TCatalogiaMirror dstMsg;
        for (; input->IsValid(); input->Next()) {
            const auto &row = input->GetRow();
            const TString &host = row.GetHost();
            TString rhost = host;
            ReverseInPlace(rhost);

            bool found = false;
            for (auto it = MakePrefixIterator(Trie, rhost.data(), rhost.size()); it; ++it) {
                const TString owner = host.substr(host.size() - it.GetPrefixLen());
                if (NUtils::IsSubdomain(host, owner)) {
                    found = true;
                    break;
                }
            }

            if (found && row.GetHost() != row.GetMainHost()) {
                dstMsg.SetHost(row.GetHost());
                dstMsg.SetMainHost(row.GetMainHost());
                output->AddRow(dstMsg);
            }
        }
    }

public:
    TVector<char> TrieStream;
    TCompactTrie<char> Trie;
};

REGISTER_MAPPER(TMirrorsMapper)

struct TMirrorsFilterReducer : public TTaggedReducer {
    void DoTagged(TTagedReader reader, TTagedWriter writer) override {
        TMaybe<NProto::TCatalogiaHost> filter = reader.GetRowMaybe(CatalogiaHostInputTag);
        reader.SkipRows(CatalogiaHostInputTag);
        if (!reader.IsValid() || !filter.Defined()) {
            return;
        }

        for (auto row : reader.GetRows(CatalogiaMirrorInputTag)) {
            writer.AddRow(row, CatalogiaMirrorOutputTag);
        }
    }
};

REGISTER_REDUCER(TMirrorsFilterReducer)

int TaskMirrors(int, const char **) {
    const auto &cfg = TConfig::CInstance();
    NYT::IClientPtr clientMain = NYT::CreateClient(cfg.MR_SERVER_HOST_MAIN);

    const TString mirrorsPath = GetJupiterMirrorsInProdTable(clientMain);
    TYtSourceTrigger mirrorsTrigger(clientMain, cfg.TABLE_CATALOGIA_SOURCE_MIRRORS);
    if (!mirrorsTrigger.NeedUpdate(mirrorsPath)) {
        LOG_INFO("user_sessions, mirrors is already updated");
        return 0;
    }

    NYT::IClientPtr clientCatalogia = NYT::CreateClient(cfg.MR_SERVER_HOST_CATALOGIA);
    THashSet<TString> domains;
    TVector<char> domainsTrieStream;
    LoadCatalogiaDomains(clientCatalogia, cfg.TABLE_SOURCE_CATALOGIA_DOMAINS, domains, domainsTrieStream);
    LOG_INFO("user_sessions, domains %lu, trie %lu bytes", domains.size(), domainsTrieStream.size());

    NYT::ITransactionPtr tx = clientMain->StartTransaction();

    LOG_INFO("user_sessions, mirrors, input %s", mirrorsPath.c_str());
    LOG_INFO("user_sessions, mirrors, output %s", cfg.TABLE_CATALOGIA_SOURCE_MIRRORS.c_str());

    TMapCmd<TMirrorsMapper>(tx, new TMirrorsMapper(domainsTrieStream))
        .OperationWeight(cfg.OPERATION_WEIGHT)
        .Input(TTable<NJupiter::THostMirror>(tx, mirrorsPath))
        .Output(TTable<NProto::TCatalogiaMirror>(tx, cfg.TABLE_CATALOGIA_SOURCE_MIRRORS))
        .CpuLimit(0.2)
        .Do()
    ;

    TSortCmd<NProto::TCatalogiaMirror>(tx, TTable<NProto::TCatalogiaMirror>(tx, cfg.TABLE_CATALOGIA_SOURCE_MIRRORS))
        .OperationWeight(cfg.OPERATION_WEIGHT)
        .By({"Host", "MainHost"})
        .Do()
    ;

    TReduceCmd<TMirrorsFilterReducer>(tx)
        .OperationWeight(cfg.OPERATION_WEIGHT)
        .Input(TTable<NProto::TCatalogiaHost>(tx, cfg.TABLE_CATALOGIA_SOURCE_HOSTS_FLT), CatalogiaHostInputTag)
        .Input(TTable<NProto::TCatalogiaMirror>(tx, cfg.TABLE_CATALOGIA_SOURCE_MIRRORS), CatalogiaMirrorInputTag)
        .Output(TTable<NProto::TCatalogiaMirror>(tx, cfg.TABLE_CATALOGIA_SOURCE_MIRRORS_FLT)
            .AsSortedOutput({"Host", "MainHost"}), CatalogiaMirrorOutputTag
        )
        .ReduceBy({"Host"})
        .CpuLimit(0.2)
        .Do()
    ;

    mirrorsTrigger.Update(tx, mirrorsPath);
    tx->Commit();

    LOG_INFO("user_sessions, mirrors, done");

    return 0;
}

} //namespace NCatalogia
} //namespace NWebmaster
