#include <util/string/reverse.h>

#include <wmconsole/version3/processors/user_sessions/exports/catalogia/protos/catalogia.pb.h>
#include <wmconsole/version3/wmcutil/regex.h>

#include "utils.h"

namespace NWebmaster {
namespace NCatalogia {

bool MatchPeriod(const TString &tableName, time_t periodBegin, time_t periodEnd, time_t &matched) {
    if (tableName.empty()) {
        return false;
    }

    NWebmaster::TRegularExpression regex("(\\d+)_(\\d+)");
    TVector<TString> period;
    if (regex.GetMatches(tableName, period) != 2) {
        LOG_ERROR("Unable to parse source table name");
    }

    time_t tableDate = str2date(period[0]);
    if (tableDate >= periodBegin && tableDate <= periodEnd) {
        matched = tableDate;
        return true;
    }

    return false;
}

void BuildTrie(const THashSet<TString> &words, TVector<char> &dest) {
    TCompactTrie<char>::TBuilder trieBuilder;
    for (const TString& word : words) {
        TString rword = word;
        ReverseInPlace(rword);
        trieBuilder.Add(rword, rword.size());
    }

    TBufferStream data;
    trieBuilder.SaveAndDestroy(data);
    dest.assign(data.Buffer().Data(), data.Buffer().Data() + data.Buffer().Size());
}

void LoadHosts(NYT::IClientBasePtr client, const TString &sourceTable, const TString &column, THashSet<TString> &hosts) {
    auto reader = client->CreateTableReader<NYT::TNode>(sourceTable);
    for (; reader->IsValid(); reader->Next()) {
        const NYT::TNode &row = reader->GetRow();
        TString host = row[column].AsString();
        host.to_lower();
        host = TString{CutWWWPrefix(host)};
        if (!host.Contains(".")) {
            continue;
        }
        TString asciiHost;
        if (NUtils::IDNHostToAscii(host, asciiHost)) {
            hosts.insert(asciiHost);
        } else {
            LOG_WARN("unable to parse %s hostname", host.data());
        }
    }
}

void LoadCatalogiaDomains(NYT::IClientBasePtr client, const TString &table, THashSet<TString> &domains, TVector<char> &trie) {
    const char *F_LINK_DOMAIN   = "Domain";
    LoadHosts(client, table, F_LINK_DOMAIN, domains);
    BuildTrie(domains, trie);
}

void LoadCatalogiaRobots(NYT::IClientBasePtr client, const TString &table, THashMap<TString, TString> &robotsMap) {
    robotsMap.clear();
    for (auto input = client->CreateTableReader<NProto::TCatalogiaRobots>(table); input->IsValid(); input->Next()) {
        const auto &row = input->GetRow();
        robotsMap[row.GetHost()] = row.GetRobots();
    }
}

void LoadCatalogiaMirrors(NYT::IClientBasePtr client, const TString &table, THashMap<TString, TString> &mirrorsMap) {
    mirrorsMap.clear();
    for (auto input = client->CreateTableReader<NProto::TCatalogiaMirror>(table); input->IsValid(); input->Next()) {
        const auto &row = input->GetRow();
        mirrorsMap[row.GetHost()] = row.GetMainHost();
    }
}

} //namespace NCatalogia
} //namespace NWebmaster
