#pragma once

#include <wmconsole/version3/processors/user_sessions/protos/user_sessions.pb.h>
#include <wmconsole/version3/protos/queries2.pb.h>

#include "regions_limiter.h"
#include "utils.h"

namespace NWebmaster {
namespace NUserSessions {

struct TCompactCounters {
    const static time_t NO_TS = -1;
    struct TCounter {
        ui32 Clicks = 0;
        ui32 Shows = 0;
    };

    using TAggKey = std::tuple<
        time_t, //TableTimestamp
        bool,   //IsMobile
        bool,   //IsPad
        ui32    //RegionId
    >;

    using TKey = std::tuple<
        time_t, //TableTimestamp
        bool,   //IsMobile
        bool,   //IsPad
        ui32,   //RegionId
        ui16    //Position
    >;

    inline TKey GetKey(const NProto::TQuery &row, const time_t tableTimestamp = NO_TS) {
        return {
            tableTimestamp,
            row.GetIsMobile(),
            row.GetIsPad(),
            row.GetRegionId(),
            row.GetPosition()
        };
    }

    inline TKey GetKey(bool isMobile, bool isPad, ui32 regionId, ui16 position, time_t tableTimestamp) {
        return {
            tableTimestamp,
            isMobile,
            isPad,
            regionId,
            position
        };
    }

    inline TAggKey GetAggKey(const NProto::TQuery &row, const time_t tableTimestamp = NO_TS) {
        return {
            tableTimestamp,
            row.GetIsMobile(),
            row.GetIsPad(),
            row.GetRegionId(),
        };
    }

    inline TAggKey GetAggKey(const TKey &key) {
        return {
            std::get<0>(key),
            std::get<1>(key),
            std::get<2>(key),
            std::get<3>(key)
        };
    }

    inline time_t GetTableTimestamp(const TKey &key) {
        return std::get<0>(key);
    }

    inline bool GetIsMobile(const TKey &key) {
        return std::get<1>(key);
    }

    inline bool GetIsPad(const TKey &key) {
        return std::get<2>(key);
    }

    inline ui32 GetRegionId(const TKey &key) {
        return std::get<3>(key);
    }

    inline ui16 GetPosition(const TKey &key) {
        return std::get<4>(key);
    }

    inline TKey SetRegionId(const TKey &key, ui32 regionId) {
        return {
            std::get<0>(key),
            std::get<1>(key),
            std::get<2>(key),
            regionId,
            std::get<4>(key)
        };
    }

    void Add(const NProto::TQuery &row, const TKey &key) {
        if (!IsVisibleQueryInWebmaster(row)) {
            return;
        }

        TCounter &counter = Counters[key];
        counter.Clicks += row.GetClicks();
        counter.Shows += row.GetShows();
        Total.Clicks += row.GetClicks();
        Total.Shows += row.GetShows();
    }

    void Add(const NProto::TQuery &row, const time_t tableTimestamp = NO_TS) {
        Add(row, GetKey(row, tableTimestamp));
    }

    void AddPatchedRegionId(const NProto::TQuery &row, const time_t tableTimestamp, ui32 regionId) {
        Add(row, GetKey(row.GetIsMobile(), row.GetIsPad(), regionId, row.GetPosition(), tableTimestamp));
    }

    void Compact(const TRegionsLimiter &limiter) {
        THashMap<TKey, TCounter> newCounters;
        for (const auto &obj : Counters) {
            const TKey &key = obj.first;
            const TCounter &oldCounter = obj.second;
            const ui32 actualRegionId = limiter.GetActualId(GetRegionId(key));
            TCounter &newCounter = newCounters[SetRegionId(key, actualRegionId)];
            newCounter.Clicks += oldCounter.Clicks;
            newCounter.Shows += oldCounter.Shows;
        }
        Counters.swap(newCounters);
    }

    bool Empty() const {
        return Counters.empty();
    }

    void Write(proto::queries2::QueryMessage &dstMsg) {
        THashMap<TAggKey, proto::queries2::QueryRegionInfo*> agg;
        for (const auto &obj : Counters) {
            const TKey &key = obj.first;
            const TCounter &counter = obj.second;

            proto::queries2::QueryRegionInfo *&qri = agg[GetAggKey(key)];
            if (!qri) {
                qri = dstMsg.add_reports_by_region();
                qri->set_region_id(GetRegionId(key));
                qri->set_is_mobile(GetIsMobile(key));
                qri->set_is_pad(GetIsPad(key));
                const time_t tts = GetTableTimestamp(key);
                if (tts != NO_TS) {
                    qri->set_timestamp(tts);
                }
            }

            proto::queries2::QueryPositionInfo *qpi = qri->add_position_info();
            qpi->set_position(GetPosition(key));
            qpi->set_clicks_count(counter.Clicks);
            qpi->set_shows_count(counter.Shows);
        }
    }

public:
    THashMap<TKey, TCounter> Counters;
    TCounter Total;
};

} //namespace NUserSessions
} //namespace NWebmaster
