#pragma once

#include <util/generic/deque.h>
#include <util/generic/hash.h>
#include <util/generic/ptr.h>

#include <library/cpp/geobase/lookup.hpp>

#include <wmconsole/version3/processors/user_sessions/conf/config.h>

namespace NWebmaster {

struct TRegionsLimiter {
    using TId = NGeobase::TId;

    struct TDescriptor {
        using Ptr = TSimpleSharedPtr<TDescriptor>;
        const NGeobase::ERegionType ROUND_LEVEL = NGeobase::ERegionType::REGION;

    public:
        TDescriptor(TId regionId)
            : RegionId(regionId)
            , ParentRegionId(regionId)
        {
            const bool canBeCompacted = Geo().GetRegionType(RegionId) > static_cast<int>(ROUND_LEVEL);
            if (canBeCompacted) {
                ParentRegionId = Geo().GetParentIdWithType(RegionId, static_cast<int>(ROUND_LEVEL));
                if (ParentRegionId == 0) {
                    ParentRegionId = Geo().GetParentId(RegionId);
                }
            }

            IsCompacted = (RegionId == ParentRegionId);
        }

        inline TId GetActualId() const {
            return IsCompacted ? ParentRegionId : RegionId;
        }

        inline void Compact() {
            IsCompacted = true;
        }

    public:
        TId RegionId;
        TId ParentRegionId;
        ui32 Traffic = 0;
        bool IsCompacted = false;
    };

    static const NGeobase::TLookup & Geo() {
        const static NGeobase::TLookup geo(TConfig::GEOBASE_FILE_LITE);
        return geo;
    }

    TRegionsLimiter(size_t limit = 3500)
        : RegionsLimit(limit)
    {
    }

    void Add(TId regionId, ui32 traffic) {
        TDescriptor::Ptr &regionDescPtr = GetRegionOrParentDescriptor(regionId);
        regionDescPtr->Traffic += traffic;
        ActualRegions.insert(regionDescPtr->GetActualId());
    }

    bool NeedCompaction() const {
        const size_t COMPCATION_THRESHOLD = 10000;
        return ActualRegions.size() > COMPCATION_THRESHOLD;
    }

    TDescriptor::Ptr & GetDescriptor(TId regionId) {
        if (Regions.contains(regionId)) {
            return Regions.at(regionId);
        }

        //We shoud be able to throw an exception
        Regions[regionId] = new TDescriptor(regionId);
        return Regions[regionId];
    }

    TDescriptor::Ptr & GetRegionOrParentDescriptor(TId regionId) {
        TDescriptor::Ptr &rdPtr = GetDescriptor(regionId);
        if (rdPtr->GetActualId() == regionId) {
            return rdPtr;
        }

        return GetDescriptor(rdPtr->GetActualId());
    }

    TId GetActualId(TId regionId) const {
        return Regions.at(regionId)->GetActualId();
    }

    void Compact(TId regionId) {
        TDescriptor::Ptr &rdPtr = GetDescriptor(regionId);
        if (rdPtr->IsCompacted) {
            return;
        }

        rdPtr->Compact();
        TDescriptor::Ptr &prdPtr = GetDescriptor(rdPtr->GetActualId());
        prdPtr->Traffic += rdPtr->Traffic;
        rdPtr->Traffic = 0;
        ActualRegions.erase(rdPtr->RegionId);
        ActualRegions.insert(rdPtr->GetActualId());
    }

    void Compact() {
        TDeque<TDescriptor::Ptr> descs;
        for (const TId regionId : ActualRegions) {
            descs.push_back(Regions.at(regionId));
        }

        std::sort(descs.begin(), descs.end(), [](const TDescriptor::Ptr &lhs, const TDescriptor::Ptr &rhs) -> bool {
            return lhs->Traffic < rhs->Traffic;
        });

        for (auto &desc : descs) {
            Compact(desc->RegionId);
            if (ActualRegions.size() <= RegionsLimit) {
                break;
            }
        }
    }

public:
    THashMap<TId, TDescriptor::Ptr> Regions;
    THashSet<TId> ActualRegions;
    size_t RegionsLimit = 0;
};

} //namespace NWebmaster
