#include <mapreduce/yt/interface/protos/yamr.pb.h>

#include <robot/library/yt/static/command.h>

#include <wmconsole/version3/wmcutil/log.h>

#include <wmconsole/version3/processors/user_sessions/conf/config.h>
#include <wmconsole/version3/processors/user_sessions/library/compact_counters.h>
#include <wmconsole/version3/processors/user_sessions/protos/user_sessions.pb.h>
#include <wmconsole/version3/protos/queries2.pb.h>

#include "counters.h"
#include "table_config.h"
#include "utils.h"

namespace NWebmaster {
namespace NUserSessions {

using namespace NJupiter;

//ReduceBy Host, Query, Path
struct TConvertV4QueriesReducer : public NYT::IReducer<NYT::TTableReader<NProto::TQuery>, NYT::TTableWriter<NYT::TYamr>> {
    Y_SAVELOAD_JOB(Timestamp)

public:
    TConvertV4QueriesReducer() = default;
    TConvertV4QueriesReducer(const time_t timestamp)
        : Timestamp(timestamp)
    {
    }

    void Do(TReader *input, TWriter *output) override {
        const NProto::TQuery firstRow = input->GetRow();

        TCompactCounters counters;
        for (; input->IsValid(); input->Next()) {
            const auto &row = input->GetRow();
            counters.Add(row);
        }

        if (counters.Empty()) {
            return;
        }

        proto::queries2::QueryMessage dstMsg;
        dstMsg.set_query(firstRow.GetQuery());
        dstMsg.set_corrected_query(firstRow.GetCorrectedQuery());
        dstMsg.set_url(firstRow.GetHost() + firstRow.GetPath());
        dstMsg.set_cm2(firstRow.GetCm2());
        dstMsg.set_is_nav(firstRow.GetIsNav());
        dstMsg.SetUpperPornoUpperPl(firstRow.GetUpperPornoUpperPl());
        dstMsg.SetIsPornoQuery(firstRow.GetIsPornoQuery());
        dstMsg.set_timestamp(Timestamp);
        counters.Write(dstMsg);

        TString stream;
        Y_PROTOBUF_SUPPRESS_NODISCARD dstMsg.SerializeToString(&stream);

        NYT::TYamr dstRow;
        dstRow.SetKey(firstRow.GetHost());
        dstRow.SetValue(stream);
        output->AddRow(dstRow);
    }

public:
    time_t Timestamp = 0;
};

REGISTER_REDUCER(TConvertV4QueriesReducer)

void ConvertUserSessionsToQueriesV4(NYT::IClientBasePtr client, const TTableConfig &ttcfg, bool force = true) {
    if (!force && client->Exists(ttcfg.OutputConvertedTable)) {
        return;
    }

    NYT::ITransactionPtr tx = client->StartTransaction();
    TReduceCmd<TConvertV4QueriesReducer>(tx, new TConvertV4QueriesReducer(ttcfg.Timestamp))
        .OperationWeight(TConfig::CInstance().OPERATION_WEIGHT)
        .ReduceBy({"Host", "CorrectedQuery", "Path"})
        .Input(TTable<NProto::TQuery>(tx, DebugPath(ttcfg.OutputTableQueries)))
        .Output(TTable<NYT::TYamr>(tx, ttcfg.OutputConvertedTable).AsSortedOutput({"key"}))
        .Do()
    ;

    TSortCmd<NYT::TYamr>(tx)
        .OperationWeight(TConfig::CInstance().OPERATION_WEIGHT)
        .Input(TTable<NYT::TYamr>(tx, ttcfg.OutputConvertedTable))
        .Output(TTable<NYT::TYamr>(tx, ttcfg.OutputConvertedTable)
            .SetCompressionCodec(ECompressionCodec::BROTLI_6)
            .SetErasureCodec(EErasureCodec::LRC_12_2_2)
        )
        .By({"key"})
        .Do()
    ;

    tx->Commit();
}

} //namespace NUserSessions
} //namespace NWebmaster
