#pragma once

#include <util/generic/hash.h>
#include <util/generic/singleton.h>
#include <util/generic/strbuf.h>
#include <util/memory/blob.h>
#include <util/string/join.h>
#include <util/string/split.h>

#include <library/cpp/archive/yarchive.h>
#include <library/cpp/json/json_reader.h>
#include <library/cpp/string_utils/url/url.h>

#include <quality/logs/baobab/api/cpp/common/all.h>

#include <wmconsole/version3/processors/user_sessions/protos/feeds.pb.h>

namespace NWebmaster {
namespace NUserSessions {

struct TCostPlusSearchProp {
    struct TPluginConfig {
        TPluginConfig() = default;
        TPluginConfig(const TString &vertical, const TString &domain)
            : Vertical(vertical)
            , Domain(domain)
        {
        }

    public:
        TString Vertical;
        TString Domain;
    };

    struct TDetected {
        void AddHostHit(const TString &vertical, const TString &domain) {
            Hits.push_back("host:" + vertical + ":" + domain);
        }

        void AddPluginHit(const TString &vertical, const TString &plugin, const TString &domain) {
            Hits.push_back("plugin/" + plugin + ":" + vertical + ":" + domain);
        }

        TString ToString() const {
            return JoinSeq(",", Hits);
        }

    public:
        TDeque<TString> Hits;
    };

    using TDetectedDomains = THashMap<TString, TDetected>;

    static const TCostPlusSearchProp &CInstance() {
        return *Singleton<TCostPlusSearchProp>();
    }

    TCostPlusSearchProp() {
        static const unsigned char COST_PLUS_PARTNERS_DATA[] = {
            #include "cost_plus_partners.h"
        };

        TArchiveReader archive(TBlob::NoCopy(COST_PLUS_PARTNERS_DATA, Y_ARRAY_SIZE(COST_PLUS_PARTNERS_DATA)));
        for (size_t i = 0; i < archive.Count(); ++i) {
            const TString key = archive.KeyByIndex(i);
            const TBlob blob = archive.ObjectBlobByKey(key);

            TMemoryInput in(blob.Data(), blob.Length());
            NJson::TJsonValue partnersJson;
            NJson::ReadJsonTree(&in, &partnersJson, true);

            THashMap<TString, NJson::TJsonValue> verticalsMap;
            partnersJson.GetMap(&verticalsMap);
            for (const auto &verticalsObj : verticalsMap) {
                THashMap<TString, NJson::TJsonValue> servicesMap;
                verticalsObj.second.GetMap(&servicesMap);
                for (const auto &servicesObj : servicesMap) {
                    if (servicesObj.second.Has("plugin")) {
                        PluginsAlias[servicesObj.second["plugin"].GetString()] = TPluginConfig(verticalsObj.first, servicesObj.first);
                    }
                }
            }
        }
    }

    void Parse(TStringBuf psf, TDetectedDomains &dst) const {
        if (psf.SkipPrefix("UPPER.PrettySerpFeatures.HasData_") && psf.ChopSuffix("_landings")) {
            if (psf.SkipPrefix("cp_")) {
                const TString vertical = TString{psf.Before('_')};
                TString domain = TString{psf.After('_')};
                SubstGlobal(domain, '_', '.');
                dst[domain].AddHostHit(vertical, domain);
            } else if (PluginsAlias.contains(psf)) {
                const TString &vertical = PluginsAlias.at(psf).Vertical;
                const TString &domain = PluginsAlias.at(psf).Domain;
                dst[domain].AddPluginHit(vertical, TString{psf}, domain);
            }
        }
    }

    static bool Lookup(const TString &url, const TDetectedDomains &detected, TString &result) {
        TStringBuf host = CutWWWPrefix(GetHost(url));
        TString domain(host.data(), host.size());
        domain.to_lower();

        TVector<TStringBuf> domainParts;
        StringSplitter(domain).Split('.').AddTo(&domainParts);

        //try to iterate over domains: a.b.c.d, b.c.d, c.d
        for (int i = 0; i < (int)domainParts.size() - 1; i++) {
            const TStringBuf leveledDomain(domainParts[i].begin(), domain.end());
            if (detected.contains(leveledDomain)) {
                result = detected.at(leveledDomain).ToString();
                return true;
            }
        }
        return false;
    }

public:
    THashMap<TString, TPluginConfig> PluginsAlias;
};


struct TCostPlusBlockStatistics {
    struct TItem {
        using Ptr = TSimpleSharedPtr<TItem>;

        enum EType {
            SET,
            OFFER,
            MORE_LINK,
        };

    public:
        TItem(EType type, const TString &blockId)
            : Type(type)
            , BaobabBlockID(blockId)
        {
        }

        void Serialize(NProto::TCostPlusItem &cpi) const {
            switch(Type) {
            case SET:
                cpi.SetType(NProto::SET);
                break;
            case OFFER:
                cpi.SetType(NProto::OFFER);
                break;
            case MORE_LINK:
                cpi.SetType(NProto::MORE_LINK);
                break;
            };

            TString host, path;
            SplitUrlToHostAndPath(Url, host, path);
            host.to_lower();
            cpi.SetBaobabBlockID(BaobabBlockID);
            cpi.SetHost(host);
            cpi.SetPath(path);
            cpi.SetTitle(Title);
            cpi.SetClicks(Clicks);
        }

    public:
        EType Type;
        TString BaobabBlockID;
        TString Url;
        TString Title;
        int Clicks = 0;
    };

public:
    TItem::Ptr CreateItem(TItem::EType type, const TString &blockId) {
        TItem::Ptr item(new TItem(type, blockId));
        ItemsByID[blockId] = item;
        return item;
    }

    void ParseMoreLink(const TMaybe<NBaobab::TBlock> &child) {
        MoreLink = CreateItem(TItem::MORE_LINK, child->GetID());
        MoreLink->Url = child->GetAttrs().at("url").Get<TString>();
        if (child->GetAttrs().has("text")) {
            MoreLink->Title = child->GetAttrs().at("text").Get<TString>();
        }
    }

    void ParseGallery(const TMaybe<NBaobab::TBlock> &block) {
        for (TMaybe<NBaobab::TBlock> child = block->GetFirstChild(); child.Defined(); child = child->GetNextSibling()) {
            if (child->GetName() == "item") {
                TItem::Ptr offer = CreateItem(TItem::OFFER, child->GetID());
                offer->Url = child->GetAttrs().at("url").Get<TString>();
                if (child->GetAttrs().has("title")) {
                    offer->Title = child->GetAttrs().at("title").Get<TString>();
                }
                Offers.push_back(offer);
            } else if (child->GetName() == "more_link") {
                ParseMoreLink(child);
            }
        }
    }

    void ParseCostPlus(const TMaybe<NBaobab::TBlock> &block) {
        for (TMaybe<NBaobab::TBlock> child = block->GetFirstChild(); child.Defined(); child = child->GetNextSibling()) {
            if (child->GetName() == "data") {
                const auto type = child->GetAttrs().at("type").Get<TString>();
                if (type == "gallery" || type == "table") {
                    ParseGallery(child);
                }
            } else if (child->GetName() == "more_link") {
                ParseMoreLink(child);
            }
        }
    }

    void ParseResult(const TMaybe<NBaobab::TBlock> &block) {
        for (TMaybe<NBaobab::TBlock> child = block->GetFirstChild(); child.Defined(); child = child->GetNextSibling()) {
            if (child->GetName() == "title") {
                Set = CreateItem(TItem::SET, child->GetID());
                Set->Url = child->GetAttrs().at("titleUrl").Get<TString>();
            }

            if (child->GetName() == "costplus") {
                Thematics = child->GetAttrs().at("type").Get<TString>();
                ParseCostPlus(child);
            }
        }
    }

    void ProcessClick(const TMaybe<NBaobab::TBlock> &block) {
        if (!block.Defined()) {
            return;
        }

        const TString &blockID = block->GetID();
        if (ItemsByID.contains(blockID)) {
            ItemsByID.at(blockID)->Clicks++;
            Clicks++;
        } else {
            ProcessClick(block->GetParent());
        }
    }

    void Serialize(TMaybe<NProto::TCostPlusBlockStatistics> &cpbs) {
        cpbs = NProto::TCostPlusBlockStatistics();
        cpbs->SetThematics(Thematics);
        cpbs->SetClicks(Clicks);

        if (Set) {
            Set->Serialize(*cpbs->MutableSet());
        }

        for (const auto &offer : Offers) {
            auto *protoOffer = cpbs->AddOffers();
            offer->Serialize(*protoOffer);
        }

        if (MoreLink) {
            MoreLink->Serialize(*cpbs->MutableMoreLink());
        }
    }

public:
    THashMap<TString, TItem::Ptr> ItemsByID;
    TItem::Ptr Set;
    TVector<TItem::Ptr> Offers;
    TItem::Ptr MoreLink;
    TString Thematics;
    int Clicks = 0;
};

} //namespace NUserSessions
} //namespace NWebmaster
