#include <util/generic/size_literals.h>

#include <metrika/uatraits/include/uatraits/detector.hpp>
#include <kernel/prs_log/serializer/serialize.h>
#include <kernel/search_query/search_query.h>
#include <library/cpp/protobuf/yql/descriptor.h>
#include <library/cpp/resource/resource.h>
#include <mapreduce/yt/interface/protos/yamr.pb.h>

#include <quality/user_sessions/createlib/qb3/parser/proto_session/proto_session.h>
#include <quality/user_sessions/createlib/qb3/proto/events.pb.h>
#include <quality/user_sessions/request_aggregate_lib/all.h>
#include <quality/user_sessions/request_aggregate_lib/mr_reader.h>
#include <yweb/verticals/cost_plus/lib/plugins_extractor/plugins_extractor.h>

#include <robot/library/yt/static/command.h>
#include <robot/library/yt/static/tags.h>

#include <wmconsole/version3/processors/user_sessions/conf/config.h>
#include <wmconsole/version3/processors/user_sessions/protos/user_sessions.pb.h>
#include <wmconsole/version3/protos/queries2.pb.h>
#include <wmconsole/version3/wmcutil/log.h>

#include "cost_plus.h"
#include "counters.h"
#include "parse.h"
#include "robots.h"
#include "utils.h"

namespace NWebmaster {
namespace NUserSessions {

using namespace NJupiter;

static const TInputTag<NYT::TYamr> CleanInputTag                    (1);
static const TInputTag<NYT::TYamr> FraudsInputTag                   (2);
static const TInputTag<NYT::TYamr> RobotsInputTag                   (3);
static const TInputTag<NProto::TFraudUIDType> UIDFraudTypesInputTag (4);
static const TInputTag<::NUserSessions::NProto::TTotalEvent> ColumnarCleanInputTag   (5);
static const TInputTag<::NUserSessions::NProto::TTotalEvent> ColumnarFraudsInputTag  (6);
static const TInputTag<::NUserSessions::NProto::TTotalEvent> ColumnarRobotsInputTag  (7);
static const TInputTag<::NUserSessions::NProto::TTotalEvent> ColumnarEventMoneyInputTag  (8);

static const TOutputTag<NProto::TQuery> QueryOutputTag  (1);
static const TOutputTag<NProto::TQuery> PrsLogOutputTag (2);

//ReduceBy key
//SortBy key, subkey
struct TParseSessionReducer : public TTaggedReducer {
    TParseSessionReducer() = default;
    TParseSessionReducer(
        const TBlockStatInfo &blockStatInfo, const TMetrikaRobotHits &metrikaRobotHits, bool enablePrsLog
    )
        : BlockStatInfo(blockStatInfo)
        , MetrikaRobotHits(metrikaRobotHits)
        , EnablePrsLog(enablePrsLog)
    {
    }

    void Save(IOutputStream& stream) const override {
        ::Save(&stream, BlockStatInfo);
        ::Save(&stream, MetrikaRobotHits);
        ::Save(&stream, EnablePrsLog);
        TTaggedReducer::Save(stream);
    }

    void Load(IInputStream& stream) override {
        ::Load(&stream, BlockStatInfo);
        ::Load(&stream, MetrikaRobotHits);
        ::Load(&stream, EnablePrsLog);
        TTaggedReducer::Load(stream);
    }

public:
    constexpr static unsigned int MAX_POSITION = 50;

    struct TClicks {
    public:
        ui32 Clicks = 0;
        ui32 TurboClicks = 0;
    };

    struct THostFactor {
        TMaybe<TString> DocMarkersSlices;
        TMaybe<TString> DocMarkersCostPlus;
        TMaybe<TString> DataAvailableCostPlus2;
        TMaybe<NProto::TCostPlusBlockStatistics> CostPlusBlockStatistics;
        bool IsFactAnswer = false;
    };

    using THostFactors = THashMap<TString, THostFactor>;

    bool ConstructFactDocUrl(const TStringBuf& docUrl, TString &fixedDocUrl) {
        NUri::TUri uri;
        NUri::TState::EParsed state = uri.Parse(
                docUrl,
                NUri::TParseFlags(NUri::TFeature::FeaturesAll | NUri::TFeature::FeatureAllowHostIDN | NUri::TFeature::FeatureEncodeSpace)
        );
        if (state != NUri::TState::ParsedOK || !uri.IsValidGlobal()) {
            return false;
        }
        TStringBuf queryString = uri.GetField(NUri::TField::FieldQuery);
        TCgiParameters params(queryString);
        if (!params.Has("utm_source") || params.Get("utm_source") != "fact") {
            params.InsertUnescaped("utm_source", "fact");
            NUri::TUriUpdate(uri).Set(NUri::TField::FieldQuery, params.Print());
        }
        fixedDocUrl = uri.PrintS();
        return true;
    }

    TMaybe<TString> GetDataAvailableCostPlus2(const NRA::TOrganicResultProperties* organicResult) {
        if (organicResult) {
            const auto &mbPlugin = NPiraPlugins::ExtractPluginData(organicResult->GetMarkers());
            if (mbPlugin.Defined()) {
                const auto &plugin = mbPlugin.Get();
                TString result;
                if (plugin->PluginName.StartsWith("cp_")) {
                    result = "host:" + plugin->Theme + ":" + plugin->Service;
                } else {
                    result = "plugin/" + plugin->PluginName + ":" + plugin->Theme + ":" + plugin->Service;
                }
                return {result};
            }
        }
        return Nothing();
    }

    void ClearHostFactors(NProto::TQuery &dstMsg) {
        dstMsg.ClearDocMarkersSlices();
        dstMsg.ClearDocMarkersCostPlus();
        dstMsg.ClearCostPlusBlockStatistics();
        dstMsg.ClearDataAvailableCostPlus2();
    }

    void WriteUrls(NProto::TQuery &dstMsg, const THashMap<TString, THashMap<EResultSource, TClicks>> &urls, const TCostPlusSearchProp::TDetectedDomains &costPlusDomains, const THostFactors &hostFactors, TTagedWriter writer) {
        for (const auto &urlObj : urls) {
            TString host, path;
            TString url = urlObj.first;
            if (!SplitAndFixUrl(url, host, path)) {
                continue;
            }

            if (hostFactors.contains(host)) {
                const THostFactor &hostFactor = hostFactors.at(host);
                if (hostFactor.IsFactAnswer) {
                    if (!ConstructFactDocUrl(url, url)) {
                        continue;
                    }
                    if (!SplitAndFixUrl(url, host, path)) {
                        continue;
                    }
                }

                if (hostFactor.DocMarkersSlices.Defined()) {
                    dstMsg.SetDocMarkersSlices(*hostFactor.DocMarkersSlices);
                } else {
                    dstMsg.ClearDocMarkersSlices();
                }

                if (hostFactor.DocMarkersCostPlus.Defined()) {
                    dstMsg.SetDocMarkersCostPlus(*hostFactor.DocMarkersCostPlus);
                } else {
                    dstMsg.ClearDocMarkersCostPlus();
                }

                if (hostFactor.CostPlusBlockStatistics.Defined()) {
                    *dstMsg.MutableCostPlusBlockStatistics() = *hostFactor.CostPlusBlockStatistics;
                } else {
                    dstMsg.ClearCostPlusBlockStatistics();
                }
                if (hostFactor.DataAvailableCostPlus2.Defined()) {
                    dstMsg.SetDataAvailableCostPlus2(*hostFactor.DataAvailableCostPlus2);
                } else {
                    dstMsg.ClearDataAvailableCostPlus2();
                }
            } else {
                ClearHostFactors(dstMsg);
            }

            TString cpAvailableData;
            if (TCostPlusSearchProp::Lookup(host, costPlusDomains, cpAvailableData)) {
                dstMsg.SetDataAvailableCostPlus(cpAvailableData);
            } else {
                dstMsg.ClearDataAvailableCostPlus();
            }

            for (const auto &resultObj : urlObj.second) {
                dstMsg.SetHost(host);
                dstMsg.SetPath(path);
                dstMsg.SetResultSource(resultObj.first);
                dstMsg.SetClicks(resultObj.second.Clicks);
                dstMsg.SetTurboClicks(resultObj.second.TurboClicks);
                writer.AddRow(dstMsg, QueryOutputTag);
            }
        }
    }

    void WritePrsLog(NProto::TQuery &dstMsg, const TMultiMap<float, TString> &urls, const TCostPlusSearchProp::TDetectedDomains &costPlusDomains, EResultSource source, TTagedWriter writer) {
        dstMsg.ClearUpperDocNavPred0();
        dstMsg.SetClicks(0);
        dstMsg.SetTurboClicks(0);
        dstMsg.SetResultSource(source);
        size_t position = urls.size();
        for (const auto &[rel, url] : urls) {
            position--;
            TString host, path;
            if (!SplitAndFixUrl(url, host, path)) {
                continue;
            }

            TString cpAvailableData;
            if (TCostPlusSearchProp::Lookup(host, costPlusDomains, cpAvailableData)) {
                dstMsg.SetDataAvailableCostPlus(cpAvailableData);
            } else {
                dstMsg.ClearDataAvailableCostPlus();
            }

            dstMsg.SetHost(host);
            dstMsg.SetPath(path);
            dstMsg.SetPosition(position);
            dstMsg.SetPrsLogL2Relevance(rel);
            writer.AddRow(dstMsg, PrsLogOutputTag);
        }
    }

    inline bool GetSearchProp(const NRA::TMiscRequestProperties *miscPropPtr, const char *key, TString &value) {
        if (const auto* const p = miscPropPtr->GetSearchPropsValues().FindPtr(key)) {
            value = *p;
            return true;
        }
        return false;
    }

    template<class TValueType>
    inline bool GetSearchProp(const NRA::TMiscRequestProperties *miscPropPtr, const char *key, TValueType &value) {
        if (const auto* const p = miscPropPtr->GetSearchPropsValues().FindPtr(key)) {
            return TryFromString(*p, value);
        }
        return false;
    }

    inline void ProcessPrsLog(const TString &prsLogBase64, TMultiMap<float, TString> &prsUrls) {
        NPrsLog::TWebData prsInfo;
        try {
            prsInfo = NPrsLog::DeserializeFromBase64(prsLogBase64);
        } catch (yexception& exc) {
            Cerr << "Exception occurred during PRS deserialization from a string of length=" <<
                prsLogBase64.size() << ": " << exc.what() << Endl;
            return;
        }
        for (const auto &doc : prsInfo.Documents) {
            prsUrls.insert({doc.L2Relevance, doc.Url});
        }
    }

    template <class TRequestTypePtr>
    void ProcessRequest(ETableSource ets, TRequestTypePtr requestPtr,
        ERequestSource source, const TMetrikaRobotHits &metrikaRobotHits,
        const TMaybe<NProto::TFraudUIDType> &mbFraudType, TTagedWriter writer)
    {
        const int MAX_QUERY_LEN = 1024;
        NProto::TQuery dstMsg;

        if (mbFraudType.Defined()) {
            dstMsg.SetUIDFraudType(mbFraudType.GetRef().GetUIDFraudType());
            dstMsg.SetUIDFraudRank(mbFraudType.GetRef().GetUIDFraudRank());
        }

        TMultiMap<float, TString> prsLogWebUrls;
        TMultiMap<float, TString> prsLogWebMispellUrls;
        TCostPlusSearchProp::TDetectedDomains costPlusDomains;
        const NRA::TMiscRequestProperties *miscPropPtr = dynamic_cast<const NRA::TMiscRequestProperties *>(requestPtr);
        if (miscPropPtr) {
            for (const auto &obj : miscPropPtr->GetSearchPropsValues()) {
                TCostPlusSearchProp::CInstance().Parse(obj.first, costPlusDomains);
            }

            const NRA::TNameValueMap relev = miscPropPtr->GetRelevValues();

            float cm2 = 0.0f;
            if (relev.contains("cm2")) {
                if (!TryFromString<float>(relev.at("cm2"), cm2)) {
                    Cerr << "Unable to parse cm2: " << relev.at("cm2") << Endl;
                }
            }
            dstMsg.SetCm2(cm2);

            //https://st.yandex-team.ru/SEARCHSPAM-15444
            float isCPQuery = 0.0f;
            if (relev.contains("class_query_is_cp")) {
                if (!TryFromString<float>(relev.at("class_query_is_cp"), isCPQuery)) {
                    Cerr << "Unable to parse isCPQuery: " << relev.at("class_query_is_cp") << Endl;
                }
            }
            dstMsg.SetQueryIsCPPred(isCPQuery);

            const bool isNav = relev.contains("is_nav") ? FromString<bool>(relev.at("is_nav")) : false;
            dstMsg.SetIsNav(isNav);

            //https://a.yandex-team.ru/arc/trunk/arcadia/yweb/antiporno/yt_tools/demon_cat/lib/process_request.cpp?rev=3884165#L442
            i32 upperPornoUpperPl = 0;
            if (GetSearchProp(miscPropPtr, "UPPER.PornoUpper.pl", upperPornoUpperPl)) {
                dstMsg.SetUpperPornoUpperPl(upperPornoUpperPl);
            }

            dstMsg.SetIsPornoQuery(isCPQuery >= 0.8921 || upperPornoUpperPl == 100);

            //https://st.yandex-team.ru/WMC-7451
            float upperQueryNavPred = 0;
            if (GetSearchProp(miscPropPtr, "UPPER.HighlightDocuments.query_nav_pred", upperQueryNavPred)) {
                dstMsg.SetUpperQueryNavPred(upperQueryNavPred);
            }

            //https://a.yandex-team.ru/arc/trunk/arcadia/search/web/rearrange/highlight_documents/highlight_documents.cpp?rev=6874555#L254
            float upperDocNavPred0 = 0;
            if (GetSearchProp(miscPropPtr, "UPPER.HighlightDocuments.doc_nav_pred_0", upperDocNavPred0)) {
                dstMsg.SetUpperDocNavPred0(upperDocNavPred0);
            }

            if (EnablePrsLog) {
                //https://st.yandex-team.ru/WMC-9070
                TString prsLogBase64;
                if (GetSearchProp(miscPropPtr, "WEB.PrsLog.Web", prsLogBase64)) {
                    ProcessPrsLog(prsLogBase64, prsLogWebUrls);
                }

                if (GetSearchProp(miscPropPtr, "WEB_MISSPELL.PrsLog.Web", prsLogBase64)) {
                    ProcessPrsLog(prsLogBase64, prsLogWebMispellUrls);
                }
            }

            const NRA::TNameValueMap &rearr = miscPropPtr->GetRearrValues();
            if (rearr.contains("qfc_id_1")) {
                //https://wiki.yandex-team.ru/searchshare/
                //https://mklimushkin.at.yandex-team.ru/85
                int searchshareClusterId = 0;
                if (TryFromString<int>(rearr.at("qfc_id_1"), searchshareClusterId)) {
                    dstMsg.SetSearchshareClusterId(searchshareClusterId);
                } else {
                    Cerr << "Unable to parse qfc_id_1: " << rearr.at("qfc_id_1") << Endl;
                }
            }
        }

        const NRA::TPageProperties *pagePropPtr = dynamic_cast<const NRA::TPageProperties *>(requestPtr);
        if (pagePropPtr) {
            dstMsg.SetPageNo(pagePropPtr->GetPageNo());
        }

        const NRA::TGeoInfoRequestProperties *geoPropPtr = dynamic_cast<const NRA::TGeoInfoRequestProperties*>(requestPtr);
        if (pagePropPtr) {
            dstMsg.SetRegionId(geoPropPtr->GetUserRegion());
        }

        const NRA::TMisspellRequestProperties *mspPropPtr = dynamic_cast<const NRA::TMisspellRequestProperties*>(requestPtr);
        if (mspPropPtr) {
            TString normQuery;
            if (!CheckAndFixQueryStringUTF8(mspPropPtr->GetCorrectedQuery(), normQuery)) {
                return;
            }
            if (normQuery.size() > MAX_QUERY_LEN) {
                normQuery.resize(MAX_QUERY_LEN);
            }
            dstMsg.SetCorrectedQuery(normQuery);
            dstMsg.SetIsReask(mspPropPtr->IsReask());
            dstMsg.SetIsMSP(mspPropPtr->IsMSP());
        }

        THashMap<TString, THashMap<EResultSource, TClicks>> miscUrls;
        for (auto &click : requestPtr->GetMiscClicks()) {
            if (!click->GetUrl().Defined() || click->GetUrl().GetRef().empty()) {
                continue;
            }

            auto &clicks = miscUrls[click->GetUrl().GetRef()][E_MISC_RESULT];
            if (click->GetBaobabBlock().Defined()) {
                if (NBaobab::GetAttrValueUnsafe<bool>(click->GetBaobabBlock().GetRef(), "turbo").GetOrElse(false) == true) {
                    clicks.TurboClicks++;
                }
            }
            clicks.Clicks++;
        }

        TString normQuery;
        if (!CheckAndFixQueryStringUTF8(requestPtr->GetQuery(), normQuery)) {
            return;
        }
        if (normQuery.size() > MAX_QUERY_LEN) {
            normQuery.resize(MAX_QUERY_LEN);
        }

        TString userAgent;
        if (requestPtr->GetUserAgent().Defined()) {
            userAgent = requestPtr->GetUserAgent().GetRef();
            dstMsg.SetUserAgent(userAgent);
        }
        dstMsg.SetDwellTimeOnService(requestPtr->GetDwellTimeOnService());
        dstMsg.SetTableSource(ets);
        dstMsg.SetRequestSource(source);
        dstMsg.SetQuery(normQuery);
        dstMsg.SetIsMobile(IsMobileRequestSource(source));

        uatraits::detector::result_type uatraitsResult = UADetectorPtr->detect(userAgent.c_str());
        dstMsg.SetIsPad(IsMobileRequestSource(source) && (uatraitsResult["isTablet"] == "true"));
        //dstMsg.SetIsPad(IsPadRequestSource(source));
        //dstMsg.SetIsTurbo(IsTurboRequestSource(source));
        dstMsg.SetShows(1);
        dstMsg.SetReqID(requestPtr->GetReqID());
        //dstMsg.SetReferer(requestPtr->GetReferer());
        //dstMsg.SetReferer(requestPtr->GetFullRequest());
        dstMsg.SetUID(requestPtr->GetUID());
        dstMsg.SetTimestamp(requestPtr->GetTimestamp());
        //if (requestPtr->GetWebParentReqID().Defined()) {
        //    dstMsg.SetWebParentReqID(requestPtr->GetWebParentReqID().GetRef());
        //}

        ui32 isRobot = 0;
        if (metrikaRobotHits.FindIsRobot(requestPtr->GetUID(), isRobot)) {
            dstMsg.SetIsMetrikaRobot(isRobot);
        }

        ui32 uidTimestamp = 0;
        if (GetUIDTimestamp(requestPtr->GetUID(), uidTimestamp)) {
            if (requestPtr->GetTimestamp() < uidTimestamp) {
                dstMsg.SetUIDAge(0);
            } else {
                dstMsg.SetUIDAge(requestPtr->GetTimestamp() - uidTimestamp);
            }
        }

        const NRA::TBlocks &mainBlocks = requestPtr->GetMainBlocks();
        //const NRA::TBlocks &mainBlocks = requestPtr->GetParallelBlocks();
        for (NRA::TBlocks::const_iterator blk = mainBlocks.begin(), blkEnd = mainBlocks.end(); blk != blkEnd; ++blk) {
            const NRA::TResult *mainResult = (*blk)->GetMainResult();
            if (mainResult == nullptr) {
                continue;
            }

            TString resultUrl;
            EResultSource resSrc;
            if (const NRA::TWebResult *result = dynamic_cast<const NRA::TWebResult *>(mainResult)) {
                resSrc = E_WEB_RESULT;
                resultUrl = result->GetUrl();
            } else if (const NRA::TWizardResult *result = dynamic_cast<const NRA::TWizardResult *>(mainResult)) {
                resSrc = E_WIZARD_RESULT;
            } else if (const NRA::TBlenderWizardResult *result = dynamic_cast<const NRA::TBlenderWizardResult *>(mainResult)) {
                resSrc = E_BLENDER_WIZARD_RESULT;
                resultUrl = result->GetUrl();
            } else if (const NRA::TDirectResult *result = dynamic_cast<const NRA::TDirectResult *>(mainResult)) {
                resSrc = E_DIRECT_RESULT;
                resultUrl = result->GetUrl().Defined() ? result->GetUrl().GetRef() : "";
            } else {
                continue;
            }

            TString resultHost, resultPath;
            if (!SplitAndFixUrl(resultUrl, resultHost, resultPath)) {
                continue;
            }

            ClearHostFactors(dstMsg);

            THostFactors hostFactors;
            if (const auto* organicProperties = dynamic_cast<const NRA::TOrganicResultProperties*>(mainResult)) {
                if (const TString *marker = organicProperties->GetMarkers().FindPtr("CostPlus")) {
                    hostFactors[resultHost].DocMarkersCostPlus = *marker;

                    if (mainResult->GetBaobabResultBlock().Defined()) {
                        TCostPlusBlockStatistics cpbs;
                        cpbs.ParseResult(mainResult->GetBaobabResultBlock());
                        for (auto &click : mainResult->GetClicks()) {
                            if (click->GetBaobabBlock().Defined()) {
                                cpbs.ProcessClick(click->GetBaobabBlock());
                            }
                        }
                        cpbs.Serialize(hostFactors[resultHost].CostPlusBlockStatistics);
                    }
                }

                if (const TString *marker = organicProperties->GetMarkers().FindPtr("Slices")) {
                    hostFactors[resultHost].IsFactAnswer =
                            (*marker).Contains("DICT_FACT")     ||
                            (*marker).Contains("FACT_SNIPPET")  ||
                            (*marker).Contains("INDIRECT_FACT") ||
                            (*marker).Contains("STATIC_FACT")
                    ;
                    hostFactors[resultHost].DocMarkersSlices = *marker;
                }

                hostFactors[resultHost].DataAvailableCostPlus2 = GetDataAvailableCostPlus2(organicProperties);
            }

            THashMap<TString, THashMap<EResultSource, TClicks>> urls;
            if (!resultUrl.empty()) {
                urls[resultUrl][resSrc];
            }

            for (auto &click : (*blk)->GetClicks()) {
                TString url = resultUrl;
                if (click->GetUrl().Defined() && !click->GetUrl().GetRef().empty()) {
                    url = click->GetUrl().GetRef();
                }

                auto &clicks = urls[url][resSrc];
                if (click->GetBaobabBlock().Defined()) {
                    if (NBaobab::GetAttrValueUnsafe<bool>(click->GetBaobabBlock().GetRef(), "turbo").GetOrElse(false) == true) {
                        clicks.TurboClicks++;
                    }
                }
                clicks.Clicks++;
            }

            //const size_t clicks = (*blk)->GetClicks().size();
            //if (const NRA::TWebResult *result = dynamic_cast<const NRA::TWebResult *>(mainResult)) {
            //    urls[result->GetUrl()][E_WEB_RESULT] += clicks;
            //} else if (const NRA::TWizardResult *result = dynamic_cast<const NRA::TWizardResult *>(mainResult)) {
            //    urls[result->GetUrl()][E_WIZARD_RESULT] += clicks;
            //} else if (const NRA::TBlenderWizardResult *result = dynamic_cast<const NRA::TBlenderWizardResult *>(mainResult)) {
            //    urls[result->GetUrl()][E_BLENDER_WIZARD_RESULT] += clicks;
            //}

            dstMsg.SetPosition(Min(mainResult->GetPosition(), MAX_POSITION));
            WriteUrls(dstMsg, urls, costPlusDomains, hostFactors, writer);
        }
        ClearHostFactors(dstMsg);

        dstMsg.SetPosition(0);
        WriteUrls(dstMsg, miscUrls, costPlusDomains, {}, writer);
        WritePrsLog(dstMsg, prsLogWebUrls, costPlusDomains, E_PRSLOG_WEB_RESULT, writer);
        WritePrsLog(dstMsg, prsLogWebMispellUrls, costPlusDomains, E_PRSLOG_WEB_MISPELL_RESULT, writer);
    }

    bool ParseRequests(TTagedReader reader, THashMap<ETableSource, NRA::TRequestsContainer> &rConts) {
        NRA::TLogsParserParams lpParams(BlockStatInfo);
        auto filter = lpParams.GetFilterMainServices();
        filter.set(MSS_WEB);
        filter.set(MSS_PORTAL);
        filter.set(MSS_TURBO);
        lpParams.SetFilterMainServices(filter);
        lpParams.SetErrorHandler(new NRA::TCerrLogsParserErrorHandler(true, false));
        THashMap<ETableSource, TAtomicSharedPtr<NRA::TLogsParser>> parsers;

        {
            const static TDeque<std::pair<const TInputTag<NYT::TYamr> &, ETableSource>> SOURCES = {
                { CleanInputTag, E_CLEAN_SOURCE },
                { RobotsInputTag, E_ROBOTS_SOURCE },
                { FraudsInputTag, E_FRAUDS_SOURCE },
            };

            for (const auto &source : SOURCES) {
                for (const auto &row : reader.GetRows(source.first)) {
                    auto &lp = parsers[source.second];
                    if (lp.Get() == nullptr) {
                        lp.Reset(new NRA::TLogsParser(lpParams));
                    }

                    try {
                        lp->AddRec(row.GetKey(), row.GetSubkey(), row.GetValue());
                        if (lp->IsFatUser()) {
                            return false;
                        }
                    } catch (...) {
                    }
                }
            }
        }
        {
            const static TDeque<std::pair<const TInputTag<::NUserSessions::NProto::TTotalEvent> &, ETableSource>> COLUMNAR_SOURCES = {
                { ColumnarCleanInputTag, E_CLEAN_SOURCE },
                { ColumnarRobotsInputTag, E_ROBOTS_SOURCE },
                { ColumnarFraudsInputTag, E_FRAUDS_SOURCE },
                { ColumnarEventMoneyInputTag, E_EVENT_MONEY_SOURCE },
            };

            for (const auto &source : COLUMNAR_SOURCES) {
                for (auto row : reader.GetRows(source.first)) {
                    auto &lp = parsers[source.second];
                    if (lp.Get() == nullptr) {
                        lp.Reset(new NRA::TLogsParser(lpParams));
                    }
                    auto protoSession = ::NUserSessions::MoveToProtoSession(std::move(row));
                    try {
                        lp->AddRec(protoSession);
                        if (lp->IsFatUser()) {
                            return false;
                        }
                    } catch (...) {
                    }
                }
            }
        }

        for (const auto &obj : parsers) {
            const auto &ets = obj.first;
            const auto &lp = obj.second;
            lp->Join();
            rConts[ets] = lp->GetRequestsContainer();
        }

        return true;
    }

    void StartTagged(TTagedWriter) override {
        BrowserConfig = NResource::Find("browser.xml");
        ProfilesConfig = NResource::Find("profiles.xml");
        ExtraConfig = NResource::Find("extra.xml");

        UADetectorPtr.Reset(new uatraits::detector(
            BrowserConfig.c_str(), static_cast<int>(BrowserConfig.size()),
            ProfilesConfig.c_str(), static_cast<int>(ProfilesConfig.size()),
            ExtraConfig.c_str(), static_cast<int>(ExtraConfig.size())
        ));

        TCostPlusSearchProp::CInstance();
    }

    void DoTagged(TTagedReader reader, TTagedWriter writer) override {
        const TMaybe<NProto::TFraudUIDType> mbFraudType = reader.GetRowMaybe(UIDFraudTypesInputTag);
        reader.SkipRows(UIDFraudTypesInputTag);

        THashMap<ETableSource, NRA::TRequestsContainer> rConts;
        if (!ParseRequests(reader, rConts)) {
            return;
        }

        for (const auto &obj : rConts) {
            const ETableSource ets = obj.first;
            const NRA::TRequests &requests = obj.second.GetRequests();
            //for (NRA::TRequests::const_iterator iter = requests.begin(), end = requests.end(); iter != end; ++iter) {
            for (const auto &request : requests) {
                if (const NRA::TYandexWebRequest *requestPtr = dynamic_cast<const NRA::TYandexWebRequest*>(request.Get())) {
                    ProcessRequest(ets, requestPtr, E_YANDEX_WEB_REQUEST, MetrikaRobotHits, mbFraudType, writer);
                } else if (const NRA::TYandexSiteSearchWebRequest *requestPtr = dynamic_cast<const NRA::TYandexSiteSearchWebRequest*>(request.Get())) {
                    ProcessRequest(ets, requestPtr, E_YANDEX_SITESEARCH_WEB_REQUEST, MetrikaRobotHits, mbFraudType, writer);
                } else if (const NRA::TMobileAppYandexWebRequest *requestPtr = dynamic_cast<const NRA::TMobileAppYandexWebRequest*>(request.Get())) {
                    ProcessRequest(ets, requestPtr, E_MOBILE_APP_YANDEX_WEB_REQUEST, MetrikaRobotHits, mbFraudType, writer);
                } else if (const NRA::TMobileYandexWebRequest *requestPtr = dynamic_cast<const NRA::TMobileYandexWebRequest*>(request.Get())) {
                    ProcessRequest(ets, requestPtr, E_MOBILE_YANDEX_WEB_REQUEST, MetrikaRobotHits, mbFraudType, writer);
                } else if (const NRA::TTouchYandexWebRequest *requestPtr = dynamic_cast<const NRA::TTouchYandexWebRequest*>(request.Get())) {
                    ProcessRequest(ets, requestPtr, E_TOUCH_YANDEX_WEB_REQUEST, MetrikaRobotHits, mbFraudType, writer);
                } else if (const NRA::TPadYandexWebRequest *requestPtr = dynamic_cast<const NRA::TPadYandexWebRequest*>(request.Get())) {
                    ProcessRequest(ets, requestPtr, E_PAD_YANDEX_WEB_REQUEST, MetrikaRobotHits, mbFraudType, writer);
                } /*
                 else if (const NRA::TDesktopYandexTurboRequest *requestPtr = dynamic_cast<const NRA::TDesktopYandexTurboRequest*>(request.Get())) {
                    ProcessRequest(ets, requestPtr, E_DESKTOP_YANDEX_TURBO_REQUEST, MetrikaRobotHits, mbFraudType, writer);
                } else if (const NRA::TMobileYandexTurboRequest *requestPtr = dynamic_cast<const NRA::TMobileYandexTurboRequest*>(request.Get())) {
                    ProcessRequest(ets, requestPtr, E_MOBILE_YANDEX_TURBO_REQUEST, MetrikaRobotHits, mbFraudType, writer);
                } else if (const NRA::TPadYandexTurboRequest *requestPtr = dynamic_cast<const NRA::TPadYandexTurboRequest*>(request.Get())) {
                    ProcessRequest(ets, requestPtr, E_PAD_YANDEX_TURBO_REQUEST, MetrikaRobotHits, mbFraudType, writer);
                } else if (const NRA::TTouchYandexTurboRequest *requestPtr = dynamic_cast<const NRA::TTouchYandexTurboRequest*>(request.Get())) {
                    ProcessRequest(ets, requestPtr, E_TOUCH_YANDEX_TURBO_REQUEST, MetrikaRobotHits, mbFraudType, writer);
                } else if (const NRA::TMobileAppYandexTurboRequest *requestPtr = dynamic_cast<const NRA::TMobileAppYandexTurboRequest*>(request.Get())) {
                    ProcessRequest(ets, requestPtr, E_MOBILE_APP_YANDEX_TURBO_REQUEST, MetrikaRobotHits, mbFraudType, writer);
                }
                */
            }
        }
    }

public:
    TBlockStatInfo BlockStatInfo;
    TMetrikaRobotHits MetrikaRobotHits;
    bool EnablePrsLog = true;
    TString BrowserConfig;
    TString ProfilesConfig;
    TString ExtraConfig;
    TSimpleSharedPtr<uatraits::detector> UADetectorPtr;
};

REGISTER_REDUCER(TParseSessionReducer)

//SortBy Host
struct TStatisticsMapper : public NYT::IMapper<NYT::TTableReader<NProto::TQuery>, NYT::TTableWriter<NProto::TStatistics>> {
    void Do(TReader *reader, TWriter *writer) override {
        THashMap<TString, TCounters<NProto::TQuery>> counters;
        for (; reader->IsValid(); reader->Next()) {
            const NProto::TQuery &row = reader->GetRow();
            counters[row.GetHost()].Add(row);
        }

        for (const auto &hostObj : counters) {
            const TString &host = hostObj.first;
            hostObj.second.Write(host, writer);
        }
    }
};

REGISTER_MAPPER(TStatisticsMapper)

//ReduceBy Host
struct TStatisticsReducer : public NYT::IReducer<NYT::TTableReader<NProto::TStatistics>, NYT::TTableWriter<NProto::TStatistics>> {

public:
    void Do(TReader *reader, TWriter *writer) override {
        const TString host = reader->GetRow().GetHost();
        TCounters<NProto::TStatistics> counters;
        for (; reader->IsValid(); reader->Next()) {
            counters.Add(reader->GetRow());
        }
        counters.Write(host, writer);
    }
};

REGISTER_REDUCER(TStatisticsReducer)

void ParseUserSessions(NYT::IClientBasePtr client, const TBlockStatInfo &bsi, const TTableConfig &ttcfg,
    const TMetrikaRobotHits &robotHits, int fraudTypeMask, bool force
) {
    if (!force) {
        if (client->Exists(ttcfg.OutputTableQueries) && client->Exists(ttcfg.OutputStatisticsTable)) {
            return;
        }

        if (ttcfg.EnabledPRS && client->Exists(ttcfg.OutputTablePrsLog)) {
            return;
        }
    }

    NYT::ITransactionPtr tx = client->StartTransaction();

    TReduceCmd<TParseSessionReducer> parseCmd(tx, new TParseSessionReducer(bsi, robotHits, ttcfg.EnabledPRS));
    if (ttcfg.EnabledFraud) {
        parseCmd.Input(TTable<NProto::TFraudUIDType>(tx, NYT::TRichYPath(ttcfg.FraudUidTypesTable).RenameColumns({{"uid", "key"}})), UIDFraudTypesInputTag);
    }

    if (ttcfg.EnabledDirect) {
        if (CheckUserFraudTypeInMask(fraudTypeMask, EUserFraudType::Clean)) {
            parseCmd
                .Input(TTable<NYT::TYamr>(tx, DebugPath(ttcfg.InputDirectUrlsClean)), CleanInputTag);
        }
        if (CheckUserFraudTypeInMask(fraudTypeMask, EUserFraudType::Robots)) {
            parseCmd
                .Input(TTable<NYT::TYamr>(tx, DebugPath(ttcfg.InputDirectUrlsRobots)), RobotsInputTag);
        }
        if (CheckUserFraudTypeInMask(fraudTypeMask, EUserFraudType::Frauds)) {
            parseCmd
                .Input(TTable<NYT::TYamr>(tx, DebugPath(ttcfg.InputDirectUrlsFrauds)), FraudsInputTag);
        }
    }

    if (ttcfg.ColumnarUserSessions) {
        if (CheckUserFraudTypeInMask(fraudTypeMask, EUserFraudType::Clean)) {
            parseCmd
                .Input(TTable<::NUserSessions::NProto::TTotalEvent>(tx, DebugPath(ttcfg.InputUserSessionsClean)), ColumnarCleanInputTag);
        }
        if (CheckUserFraudTypeInMask(fraudTypeMask, EUserFraudType::Robots)) {
            parseCmd
                .Input(TTable<::NUserSessions::NProto::TTotalEvent>(tx, DebugPath(ttcfg.InputUserSessionsRobots)), ColumnarRobotsInputTag);
        }
        if (CheckUserFraudTypeInMask(fraudTypeMask, EUserFraudType::Frauds)) {
            parseCmd
                .Input(TTable<::NUserSessions::NProto::TTotalEvent>(tx, DebugPath(ttcfg.InputUserSessionsFrauds)), ColumnarFraudsInputTag);
        }
    } else {
        if (CheckUserFraudTypeInMask(fraudTypeMask, EUserFraudType::Clean)) {
            parseCmd
                .Input(TTable<NYT::TYamr>(tx, DebugPath(ttcfg.InputUserSessionsClean)), CleanInputTag);
        }
        if (CheckUserFraudTypeInMask(fraudTypeMask, EUserFraudType::Robots)) {
            parseCmd
                .Input(TTable<NYT::TYamr>(tx, DebugPath(ttcfg.InputUserSessionsRobots)), RobotsInputTag);
        }
        if (CheckUserFraudTypeInMask(fraudTypeMask, EUserFraudType::Frauds)) {
            parseCmd
                .Input(TTable<NYT::TYamr>(tx, DebugPath(ttcfg.InputUserSessionsFrauds)), FraudsInputTag);
        }
    }

    if (ttcfg.EnabledEventMoney) {
        parseCmd.Input(TTable<::NUserSessions::NProto::TTotalEvent>(tx, DebugPath(ttcfg.InputEventMoney)), ColumnarEventMoneyInputTag);
    }

    parseCmd
        .MemoryLimit(5_GBs)
        .OperationWeight(TConfig::CInstance().OPERATION_WEIGHT)
        .Output(TTable<NProto::TQuery>(tx, ttcfg.OutputTableQueries), QueryOutputTag)
        .ReduceBy({"key"})
        //.SortBy({"key", "subkey"})
    ;

    if (ttcfg.EnabledPRS) {
        parseCmd.Output(TTable<NProto::TQuery>(tx, ttcfg.OutputTablePrsLog), PrsLogOutputTag);
    }

    parseCmd.Do();

    const auto KEYS = {"Host", "CorrectedQuery", "Path", "RegionId", "IsMobile", "IsPad", "Position", "RequestSource", "ResultSource"};

    TVector<NThreading::TFuture<void>> asyncOps;
    asyncOps.push_back(
        TSortCmd<NProto::TQuery>(tx)
            .OperationWeight(TConfig::CInstance().OPERATION_WEIGHT)
            .Input(TTable<NProto::TQuery>(tx, ttcfg.OutputTableQueries))
            .Output(TTable<NProto::TQuery>(tx, ttcfg.OutputTableQueries)
                .SetCompressionCodec(ECompressionCodec::BROTLI_6)
                .SetErasureCodec(EErasureCodec::LRC_12_2_2)
            )
            .By(KEYS)
            .DoAsync()->Watch()
    );

    if (ttcfg.EnabledPRS) {
        asyncOps.push_back(
            TSortCmd<NProto::TQuery>(tx)
                .OperationWeight(TConfig::CInstance().OPERATION_WEIGHT)
                .Input(TTable<NProto::TQuery>(tx, ttcfg.OutputTablePrsLog))
                .Output(TTable<NProto::TQuery>(tx, ttcfg.OutputTablePrsLog)
                    .SetCompressionCodec(ECompressionCodec::BROTLI_6)
                    .SetErasureCodec(EErasureCodec::LRC_12_2_2)
                )
                .By(KEYS)
                .DoAsync()->Watch()
        );
    }

    NThreading::WaitExceptionOrAll(asyncOps).GetValueSync();

    TMapCombineReduceCmd<TStatisticsMapper, TStatisticsReducer, TStatisticsReducer>(tx)
        .OperationWeight(TConfig::CInstance().OPERATION_WEIGHT)
        .MapperMemoryLimit(6_GBs)
        .ReducerMemoryLimit(1_GBs)
        .Input(TTable<NProto::TQuery>(tx, ttcfg.OutputTableQueries)
            .SelectFields({"Host", "RegionId", "IsMobile", "IsPad", "Position", "RequestSource", "ResultSource", "Clicks", "TurboClicks", "Shows"})
        )
        .Output(TTable<NProto::TStatistics>(tx, ttcfg.OutputStatisticsTable))
        .ReduceBy({"Host"})
        .Do()
    ;

    TSortCmd<NProto::TQuery>(tx)
        .OperationWeight(TConfig::CInstance().OPERATION_WEIGHT)
        .Input(TTable<NProto::TQuery>(tx, ttcfg.OutputStatisticsTable))
        .Output(TTable<NProto::TQuery>(tx, ttcfg.OutputStatisticsTable)
            .SetCompressionCodec(ECompressionCodec::BROTLI_6)
            .SetErasureCodec(EErasureCodec::LRC_12_2_2)
        )
        .By({"Host", "RegionId", "IsMobile", "IsPad", "Position", "RequestSource", "ResultSource"})
        .Do()
    ;

    NYTUtils::SetAttr(tx, ttcfg.OutputTableQueries, "_yql_proto_field_CostPlusBlockStatistics", GenerateProtobufTypeConfig<NProto::TCostPlusBlockStatistics>());
    NYTUtils::SetAttr(tx, ttcfg.OutputTablePrsLog, "_yql_proto_field_CostPlusBlockStatistics", GenerateProtobufTypeConfig<NProto::TCostPlusBlockStatistics>());

    tx->Commit();
}

} //namespace NUserSessions
} //namespace NWebmaster
