#include "push_stats.h"

#include <wmconsole/version3/processors/user_sessions/conf/config.h>
#include <wmconsole/version3/processors/user_sessions/library/common_parser_opts.h>
#include <wmconsole/version3/processors/user_sessions/protos/monitoring.pb.h>

#include <wmconsole/version3/wmcutil/args.h>
#include <wmconsole/version3/wmcutil/log.h>

#include <robot/jupiter/library/monitoring/monitoring.h>

#include <util/datetime/base.h>
#include <util/draft/date.h>

namespace NWebmaster {
namespace NUserSessions {

namespace {

TInstant DailyStateToInstant(const TString& state) {
    return TInstant::ParseIso8601(state + TString("T00:00:00+03:00"));
}

void AddSensor(const TString& label, double value, const TInstant& instant, NJupiter::TMonitoringClient& solomon) {
    NJupiter::NMonitoring::TSensor sensor;
    sensor
        .SetLabel(NJupiter::NMonitoring::TKnownLabels::Sensor, label)
        .SetValue(value)
        .SetTimestamp(instant);
    solomon.AddSensor(std::move(sensor));
}

void AddSensors(const TString& prefix, const NMonitoring::NProto::TStatistics& row, const TInstant& instant, NJupiter::TMonitoringClient& solomon) {
    AddSensor(prefix + "Median", row.GetMetrikaMedian(), instant, solomon);
    AddSensor(prefix + "P90", row.GetMetrikaP90(), instant, solomon);
    AddSensor(prefix + "P98", row.GetMetrikaP98(), instant, solomon);
}

void CollectStats(NYT::IClientPtr client, const TString& tablePath, const TString& targetDateStr, NJupiter::TMonitoringClient& solomon) {
    auto reader = client->CreateTableReader<NMonitoring::NProto::TStatistics>(tablePath);
    for (; reader->IsValid(); reader->Next()) {
        const auto& row = reader->GetRow();
        if (row.GetPeriod() != targetDateStr) {
            continue;
        }
        const auto instant = DailyStateToInstant(row.GetPeriod());
        if (row.GetIsMobile()) {
            if (row.GetIsPad()) {
                AddSensors("Queries_RefMetrika_Pad", row, instant, solomon);
            } else {
                AddSensors("Queries_RefMetrika_Mobile", row, instant, solomon);
            }
        } else if (!row.GetIsPad()) {
            AddSensors("Queries_RefMetrika_Desktop", row, instant, solomon);
        }
    }
}

TString GetFullServerName(const TString& serverName) {
    if (serverName.find('.') == TString::npos) {
        return serverName + ".yt.yandex.net";
    }
    return serverName;
}

} // anonymous namespace

int PushStats(int argc, const char **argv) {
    NJupiter::TCmdParams params;

    TCommonParserOpts commonParserOpts;
    TCommonParserOptsParser(params, commonParserOpts)
        .AddEnvRoot()
        .AddLogPath()
        .AddYtProxy()
        .AddYtPrefix();

    NJupiter::TSolomonOpts solomonOpts;
    NJupiter::TSolomonOptsParser solomonParser(params, solomonOpts);
    solomonParser
        .AddSolomonHost()
        .AddSolomonPort()
        .AddSolomonProject()
        .AddSolomonCluster()
        .AddSolomonService()
        .AddSolomonEnableAndProvideTokenFile()
        .AddSolomonTest();

    TString date;
    params.AddRequired("date", "Date to push", "<YYYY-MM-DD>", &date);

    params.Parse(argc, argv);
    TArgs::Instance().LogPath = commonParserOpts.LogPath;
    TCustomYTEnvironment::Instance().Init(commonParserOpts.EnvRoot);
    TLogger::Instance();
    NYTUtils::DisableLogger();

    Y_ENSURE_BT(!commonParserOpts.YtProxy.empty());
    Y_ENSURE_BT(!commonParserOpts.YtPrefix.empty());
    const TString ytProxy = GetFullServerName(commonParserOpts.YtProxy);
    auto& config = TConfig::Instance();
    config.MR_SERVER_HOST_USER_SESSIONS = ytProxy;
    config.MR_SERVER_HOST_HITLOG = ytProxy;
    TCommonYTConfigSQ::Instance() = TCommonYTConfigSQ(
        ytProxy,
        commonParserOpts.YtPrefix,
        NJupiter::JoinYtPath(commonParserOpts.YtPrefix, "webmaster", commonParserOpts.EnvRoot)
    );

    NYT::IClientPtr client = NYT::CreateClient(commonParserOpts.YtProxy);
    NJupiter::TMonitoringClient solomon(solomonOpts);
    const TString statsTable = TCommonYTConfigSQ::CInstance().TABLE_PARSED_USER_SESSIONS_REF_STATS;
    CollectStats(client, statsTable, date, solomon);
    solomon.Push();
    return 0;
}

} //namespace NUserSessions
} //namespace NWebmaster
