#pragma once

#include <util/draft/date.h>

#include <wmconsole/version3/processors/user_sessions/conf/config.h>
#include <wmconsole/version3/wmcutil/log.h>

#include "utils.h"

namespace NWebmaster {
namespace NUserSessions {

const THashSet<TString> INCOMPLETE_USER_SESSIONS_DATES = {
    "2022-01-27",  // MINILSR-952
};

struct TTableConfig {
    enum EInputType {
        E_INPUT_DAILY,
        E_INPUT_FAST,
    };

    TTableConfig(const TString &timestampStr, EInputType eInputType, bool columnarUserSessions, bool enableIncomplete)
        : TimestampStr(timestampStr),
          ColumnarUserSessions(columnarUserSessions),
          EnableIncomplete(enableIncomplete)
    {
        const auto &cfg = TCommonYTConfigSQ::CInstance();

        if (eInputType == E_INPUT_DAILY) {
            if (EnableIncomplete && INCOMPLETE_USER_SESSIONS_DATES.contains(TimestampStr)) {
                InputUserSessionsRoot   = cfg.TABLE_SOURCE_USER_SESSIONS_INCOMPLETE_DAILY_ROOT;
            } else {
                InputUserSessionsRoot   = cfg.TABLE_SOURCE_USER_SESSIONS_DAILY_ROOT;
            }
            InputBsCheventLogRoot   = cfg.TABLE_SOURCE_USER_SESSIONS_BS_CHEVENT_LOG_DAILY_ROOT;
            InputDirectUrlsRoot     = cfg.TABLE_SOURCE_USER_SESSIONS_DIRECT_URLS_DAILY_ROOT;
            InputEventMoneyRoot     = cfg.TABLE_SOURCE_USER_SESSIONS_EVENT_MONEY_DAILY_ROOT;

            InputBsCheventLogClean  = NYTUtils::JoinPath(InputBsCheventLogRoot, TimestampStr, "clean");
            InputBsCheventLogRobots = NYTUtils::JoinPath(InputBsCheventLogRoot, TimestampStr, "robots");
            InputBsCheventLogFrauds = NYTUtils::JoinPath(InputBsCheventLogRoot, TimestampStr, "frauds");

            InputDirectUrlsClean    = NYTUtils::JoinPath(InputDirectUrlsRoot, TimestampStr, "clean");
            InputDirectUrlsRobots   = NYTUtils::JoinPath(InputDirectUrlsRoot, TimestampStr, "robots");
            InputDirectUrlsFrauds   = NYTUtils::JoinPath(InputDirectUrlsRoot, TimestampStr, "frauds");

            InputEventMoney         = NYTUtils::JoinPath(InputEventMoneyRoot, TimestampStr);

            OutputTableQueries      = NYTUtils::JoinPath(cfg.TABLE_PARSED_USER_SESSIONS_DAILY_ROOT, TimestampStr);
            OutputTablePrsLog       = NYTUtils::JoinPath(cfg.TABLE_PARSED_USER_SESSIONS_PRSLOG_DAILY_ROOT, TimestampStr);

            Timestamp               = TDate(timestampStr, TConfig::FORMAT).GetStart();
            TimestampDisplayStr     = TimestampStr;
            OutputStatisticsTable   = NYTUtils::JoinPath(cfg.TABLE_PARSED_USER_SESSIONS_STATS_DAILY_ROOT, TimestampStr);
            OutputConvertedTable    = NYTUtils::JoinPath(cfg.TABLE_CONVERTED_QUERIES_V4_ROOT, GetClicksShowsTableName(Timestamp));
            FraudUidTypesTable      = NYTUtils::JoinPath(cfg.TABLE_SOURCE_FRAUD_UID_TYPES_ROOT, TimestampStr, "uid_types");
            MetrikaRobotsTable      = NYTUtils::JoinPath(cfg.TABLE_PARSED_USER_SESSIONS_ROBOTS_METRIKA_ROOT, TimestampStr);

            LOG_INFO("table_config, daily, timestamp %s", TimestampStr.c_str());

        } else if (eInputType == E_INPUT_FAST) {
            InputUserSessionsRoot   = cfg.TABLE_SOURCE_USER_SESSIONS_FAST_ROOT;
            //InputBsCheventLogRoot   = cfg.TABLE_SOURCE_USER_SESSIONS_DAILY_ROOT;
            //InputDirectUrlsRoot     = cfg.TABLE_SOURCE_USER_SESSIONS_DAILY_ROOT;
            Timestamp               = FromString<time_t>(TimestampStr);
            TimestampDisplayStr     = TInstant::Seconds(Timestamp).ToStringLocalUpToSeconds();

            OutputTableQueries      = NYTUtils::JoinPath(cfg.TABLE_PARSED_USER_SESSIONS_FAST_ROOT, TimestampDisplayStr);
            OutputStatisticsTable   = NYTUtils::JoinPath(cfg.TABLE_PARSED_USER_SESSIONS_STATS_FAST_ROOT, TimestampDisplayStr);

            EnabledConverted    = false;
            EnabledDirect       = false;
            EnabledEventMoney   = false;
            EnabledFraud        = false;
            EnabledPRS          = false;

            LOG_INFO("table_config, fast, timestamp %s", TimestampStr.c_str());
        } else {
            ythrow yexception() << "Unknown input type for TTableConfig";
        }

        if (ColumnarUserSessions) {
            InputUserSessionsClean  = NYTUtils::JoinPath(InputUserSessionsRoot, TimestampStr, "columns/clean");
            InputUserSessionsRobots = NYTUtils::JoinPath(InputUserSessionsRoot, TimestampStr, "columns/robots");
            InputUserSessionsFrauds = NYTUtils::JoinPath(InputUserSessionsRoot, TimestampStr, "columns/frauds");
        } else {
            InputUserSessionsClean  = NYTUtils::JoinPath(InputUserSessionsRoot, TimestampStr, "clean");
            InputUserSessionsRobots = NYTUtils::JoinPath(InputUserSessionsRoot, TimestampStr, "robots");
            InputUserSessionsFrauds = NYTUtils::JoinPath(InputUserSessionsRoot, TimestampStr, "frauds");
        }

        LOG_INFO("table_config, ColumnarUserSessions = %s",     ColumnarUserSessions ? "true" : "false");
        LOG_INFO("table_config, EnableIncomplete = %s",         EnableIncomplete    ? "true" : "false");
        LOG_INFO("table_config, EnabledConverted = %s",         EnabledConverted    ? "true" : "false");
        LOG_INFO("table_config, EnabledFraud = %s",             EnabledFraud        ? "true" : "false");
        LOG_INFO("table_config, EnabledPRS = %s",               EnabledPRS          ? "true" : "false");
        LOG_INFO("table_config, EnabledDirect = %s",            EnabledDirect       ? "true" : "false");
        LOG_INFO("table_config, FraudUidTypesTable = %s",       FraudUidTypesTable.c_str());
        LOG_INFO("table_config, MetrikaRobotsTable = %s",       MetrikaRobotsTable.c_str());
        LOG_INFO("table_config, InputBsCheventLogRoot = %s",    InputBsCheventLogRoot.c_str());
        LOG_INFO("table_config, InputBsCheventLogClean = %s",   InputBsCheventLogClean.c_str());
        LOG_INFO("table_config, InputBsCheventLogFrauds = %s",  InputBsCheventLogFrauds.c_str());
        LOG_INFO("table_config, InputBsCheventLogRobots = %s",  InputBsCheventLogRobots.c_str());
        LOG_INFO("table_config, InputDirectUrlsRoot = %s",      InputDirectUrlsRoot.c_str());
        LOG_INFO("table_config, InputDirectUrlsClean = %s",     InputDirectUrlsClean.c_str());
        LOG_INFO("table_config, InputDirectUrlsFrauds = %s",    InputDirectUrlsFrauds.c_str());
        LOG_INFO("table_config, InputDirectUrlsRobots = %s",    InputDirectUrlsRobots.c_str());
        LOG_INFO("table_config, InputUserSessionsRoot = %s",    InputUserSessionsRoot.c_str());
        LOG_INFO("table_config, InputUserSessionsClean = %s",   InputUserSessionsClean.c_str());
        LOG_INFO("table_config, InputUserSessionsFrauds = %s",  InputUserSessionsFrauds.c_str());
        LOG_INFO("table_config, InputUserSessionsRobots = %s",  InputUserSessionsRobots.c_str());
        LOG_INFO("table_config, InputEventMoneyRoot = %s",      InputEventMoneyRoot.c_str());
        LOG_INFO("table_config, InputEventMoney = %s",          InputEventMoney.c_str());
        LOG_INFO("table_config, OutputConvertedTable = %s",     OutputConvertedTable.c_str());
        LOG_INFO("table_config, OutputStatisticsTable = %s",    OutputStatisticsTable.c_str());
        LOG_INFO("table_config, OutputTablePrsLog = %s",        OutputTablePrsLog.c_str());
        LOG_INFO("table_config, OutputTableQueries = %s",       OutputTableQueries.c_str());
        LOG_INFO("table_config, Timestamp = %ld",               Timestamp);
        LOG_INFO("table_config, TimestampStr = %s",             TimestampStr.c_str());
        LOG_INFO("table_config, TimestampDisplayStr = %s",      TimestampDisplayStr.c_str());
    }

public:
    TString InputUserSessionsRoot;
    TString InputUserSessionsClean;
    TString InputUserSessionsRobots;
    TString InputUserSessionsFrauds;

    TString InputBsCheventLogRoot;
    TString InputBsCheventLogClean;
    TString InputBsCheventLogRobots;
    TString InputBsCheventLogFrauds;

    TString InputDirectUrlsRoot;
    TString InputDirectUrlsClean;
    TString InputDirectUrlsRobots;
    TString InputDirectUrlsFrauds;

    TString InputEventMoneyRoot;
    TString InputEventMoney;

    TString OutputTableQueries;
    TString OutputTablePrsLog;

    TString OutputStatisticsTable;

    TString TimestampStr;
    bool ColumnarUserSessions;
    bool EnableIncomplete;
    TString TimestampDisplayStr;
    time_t Timestamp = 0;

    TString OutputConvertedTable;
    TString FraudUidTypesTable;
    TString MetrikaRobotsTable;

    bool EnabledConverted   = true;
    bool EnabledDirect      = true;
    bool EnabledEventMoney  = true;
    bool EnabledFraud       = true;
    bool EnabledPRS         = true;
};

} //namespace NUserSessions
} //namespace NWebmaster
