#include <robot/library/yt/static/command.h>

#include <wmconsole/version3/processors/user_sessions/conf/config.h>
#include <wmconsole/version3/processors/user_sessions/library/common_parser_opts.h>
#include <wmconsole/version3/processors/user_sessions/protos/user_sessions.pb.h>
#include <wmconsole/version3/wmcutil/args.h>
#include <wmconsole/version3/wmcutil/log.h>

#include "parse.h"
#include "robots.h"
#include "table_config.h"
#include "task_parse_fast.h"
#include "utils.h"

namespace NWebmaster {
namespace NUserSessions {

using namespace NJupiter;

void ParseFastUserSessions(NYT::IClientBasePtr client, const TBlockStatInfo &bsi,
    const TTableConfig &ttcfg, bool force = true
) {
    LOG_INFO("user_sessions fast, parsing %s", ttcfg.TimestampDisplayStr.c_str());
    TMetrikaRobotHits robotHits;
    ParseUserSessions(client, bsi, ttcfg, robotHits, force);
    LOG_INFO("user_sessions fast, parsing %s - done", ttcfg.TimestampDisplayStr.c_str());
}

void UpdateFastUserSessions(IThreadPool *queue, NYT::IClientPtr client, const TBlockStatInfo &bsi) {
    const time_t now = Now().TimeT();
    TSet<TString> processed;
    for (const auto &node : client->List(TCommonYTConfigSQ::CInstance().TABLE_PARSED_USER_SESSIONS_FAST_ROOT)) {
        processed.insert(node.AsString());
    }

    time_t lastFastTimestamp = 0;
    while(true) {
        try {
            lastFastTimestamp = GetLatestUserSessionsTimestamp(true);
            break;
        } catch (yexception &e) {
            LOG_WARN("user_sessions fast, unable to get latest timestamp: %s (waiting 1m)", e.what());
            Sleep(TDuration::Minutes(1));
        }
    }

    TMap<TString, TString> available;
    for (const auto &node : client->List(TCommonYTConfigSQ::CInstance().TABLE_SOURCE_USER_SESSIONS_FAST_ROOT)) {
        const TString timestampStr = node.AsString();
        const time_t timestamp = FromString<time_t>(timestampStr);
        const TString periodName = TInstant::Seconds(timestamp).ToStringLocalUpToSeconds();
        const bool ready = timestamp <= lastFastTimestamp && (now - timestamp) <= 86400;
        if (ready) {
            available[periodName] = timestampStr;
        }
    }

    for (const auto &periodName : processed) {
        const TString path = NYTUtils::JoinPath(TCommonYTConfigSQ::CInstance().TABLE_PARSED_USER_SESSIONS_FAST_ROOT, periodName);
        const TString pathStats = NYTUtils::JoinPath(TCommonYTConfigSQ::CInstance().TABLE_PARSED_USER_SESSIONS_STATS_FAST_ROOT, periodName);
        if (!available.contains(periodName)) {
            client->Remove(path);
            client->Remove(pathStats);
            LOG_INFO("user_sessions fast, removed %s", periodName.c_str());
        }
    }

    for (const auto &obj : available) {
        if (!processed.contains(obj.first)) {
            //const TString &periodName = obj.first;
            const TString &timestampStr = obj.second;
            queue->SafeAddFunc([=, &bsi, &client]() {
                const TTableConfig ttcfg(timestampStr, TTableConfig::E_INPUT_FAST, /* columnarUserSessions */ false, /* enableIncomplete */ false);
                try {
                    ParseFastUserSessions(client, bsi, ttcfg, false);
                } catch(yexception &e) {
                    LOG_ERROR("user_sessions fast, unable to parse %s: %s", ttcfg.TimestampDisplayStr.c_str(), e.what());
                }
            });
        }
    }
}

int TaskParseFast(int argc, const char **argv) {
    auto commonParserOpts = ParseCommonOptions(argc, argv);
    InitCommonSingletones(commonParserOpts);

    NYT::IClientPtr client = NYT::CreateClient(TConfig::CInstance().MR_SERVER_HOST_USER_SESSIONS);

    NYTUtils::CreatePath(client, TCommonYTConfigSQ::CInstance().TABLE_PARSED_USER_SESSIONS_FAST_ROOT);
    NYTUtils::CreatePath(client, TCommonYTConfigSQ::CInstance().TABLE_PARSED_USER_SESSIONS_STATS_FAST_ROOT);

    const TBlockStatInfo bsi = GetBlockstatInfo(client);
    THolder<IThreadPool> queueFast(CreateThreadPool(4));
    UpdateFastUserSessions(queueFast.Get(), client, bsi);
    queueFast->Stop();
    return 0;
}

} //namespace NUserSessions
} //namespace NWebmaster
