#include "composite_filter.h"

namespace NWebmaster {

template<typename TOperator, typename TIndicator>
TConstraint::Ptr BuildConstraint00(const NWebmaster::proto::querygroups::QueryFilter &queryFilter) {
    if (queryFilter.has_indicator_from_position() || queryFilter.has_indicator_until_position()) {
        return new TConstraintBoundedImpl<TOperator, TIndicator>(queryFilter);
    } else {
        return new TConstraintImpl<TOperator, TIndicator>(queryFilter);
    }
}

template<typename TOperator>
TConstraint::Ptr BuildConstraint0(const NWebmaster::proto::querygroups::QueryFilter &queryFilter) {
    using namespace NWebmaster::proto::querygroups;

    switch (queryFilter.indicator()) {
    case QUERY:
        return BuildConstraint00<TOperator, TIndicatorQuery>(queryFilter);
    case URL:
        return BuildConstraint00<TOperator, TIndicatorUrl>(queryFilter);
    case SHOWS_COUNT:
        return BuildConstraint00<TOperator, TIndicatorShows>(queryFilter);
    case CLICKS_COUNT:
        return BuildConstraint00<TOperator, TIndicatorClicks>(queryFilter);
    case CTR:
        return BuildConstraint00<TOperator, TIndicatorCTR>(queryFilter);
    case AVERAGE_SHOW_POSITION:
        return BuildConstraint00<TOperator, TIndicatorAvgShowPos>(queryFilter);
    case AVERAGE_CLICK_POSITION:
        return BuildConstraint00<TOperator, TIndicatorAvgClickPos>(queryFilter);
    }
}

TConstraint::Ptr BuildConstraint(NWebmaster::proto::querygroups::QueryFilter queryFilter) {
    using namespace NWebmaster::proto::querygroups;

    switch (queryFilter.type()) {
    case LESS_THAN:
        return BuildConstraint0<TOperatorLT>(queryFilter);
    case LESS_OR_EQUAL:
        return BuildConstraint0<TOperatorLE>(queryFilter);
    case GREATER_THAN:
        return BuildConstraint0<TOperatorGT>(queryFilter);
    case GREATER_OR_EQUAL:
        return BuildConstraint0<TOperatorGE>(queryFilter);
    case MATCHES_REGEXP:
        return BuildConstraint0<TOperatorRE>(queryFilter);
    }
}

enum EFilterType {
    TEXT,
    FLOAT,
    BOTH,
};

EFilterType GetFilterType(const NWebmaster::proto::querygroups::QueryIndicator &indicator) {
    using namespace NWebmaster::proto::querygroups;

    switch (indicator) {
    case QUERY:
    case URL:
        return TEXT;
    default:
        return FLOAT;
    }
}

TFilter::Ptr BuildFilter(const NWebmaster::proto::querygroups::FilterExpression &expression, EFilterType requestedType, ui32 &parentFlags) {
    using namespace NWebmaster::proto::querygroups;

    TConstraint::Ptr constraint; //NULL by default, filter will skip it
    ui32 flags = 0;

    if (expression.has_filter()) {
        EFilterType filterType = GetFilterType(expression.filter().indicator());

        if (filterType == requestedType || requestedType == BOTH) {
            constraint = BuildConstraint(expression.filter());
        }

        switch (filterType) {
        case TEXT:
            flags = HAS_TEXT_CONSTRAINT;
            break;
        case FLOAT:
            flags = HAS_FLOAT_CONSTRAINT;
            break;
        default:
            ;
        }
    }

    TFilter::Ptr root;
    switch (expression.type()) {
    case FILTER:
    case AND:
        root.Reset(new TFilterImpl<TExpressionAND>(constraint));
        break;
    case NOT:
        root.Reset(new TFilterImpl<TExpressionNOT>(constraint));
        break;
    case OR:
        root.Reset(new TFilterImpl<TExpressionOR>(constraint));
        break;
    }

    ui32 childrenFlags = 0;
    for (int i = 0; i < expression.filter_expressions_size(); i++) {
        root->Child(BuildFilter(expression.filter_expressions(i), requestedType, childrenFlags));
    }

    root->Flags = flags | childrenFlags;
    parentFlags |= root->Flags;

    return root;
}

TFilter::Ptr BuildFilter(const NWebmaster::proto::querygroups::FilterExpression &expression, EFilterType requestedType) {
    ui32 flags = 0;
    return BuildFilter(expression, requestedType, flags);
}

TFilter::Ptr BuildTextFilter(const NWebmaster::proto::querygroups::FilterExpression &expression) {
    return BuildFilter(expression, TEXT);
}

TFilter::Ptr BuildFloatFilter(const NWebmaster::proto::querygroups::FilterExpression &expression) {
    return BuildFilter(expression, FLOAT);
}

TFilter::Ptr BuildFullFilter(const NWebmaster::proto::querygroups::FilterExpression &expression) {
    return BuildFilter(expression, BOTH);
}

} //namespace NWebmaster
