#include <util/generic/size_literals.h>

#include <mapreduce/yt/interface/protos/yamr.pb.h>

#include <wmconsole/version3/wmcutil/args.h>
#include <wmconsole/version3/wmcutil/periodic.h>
#include <wmconsole/version3/wmcutil/yt/misc.h>
#include <wmconsole/version3/wmcutil/yt/yt_utils.h>

#include <robot/library/yt/static/command.h>

#include "config.h"
#include "monitor.h"
#include "query_utils.h"
#include "source_tables.h"
#include "task_autocampaign.h"
#include "task_catalogia2.h"
#include "task_landings.h"
#include "task_recommended.h"
#include "task_recommended_uc2.h"

namespace NWebmaster {

namespace {
}

typedef void(*TTaskHandler)(const TConfig &config);

//ReduceBy Host
struct TMapper : public NYT::IMapper<NYT::TTableReader<NYT::TYamr>, NYT::TTableWriter<NYT::TYamr>> {

public:
    void Do(TReader *input, TWriter *output) override {
        NWebmaster::proto::queries2::QueryMessage msg;
        TString stream;
        for (; input->IsValid(); input->Next()) {
            NYT::TYamr row = input->GetRow();
            Y_PROTOBUF_SUPPRESS_NODISCARD msg.ParseFromString(row.GetValue());
            msg.set_query(msg.corrected_query());
            msg.clear_corrected_query();
            Y_PROTOBUF_SUPPRESS_NODISCARD msg.SerializeToString(&stream);
            row.SetValue(stream);
            output->AddRow(row);
        }
    }
};

REGISTER_MAPPER(TMapper)

void TaskDev(const TConfig &config) try {
    Y_UNUSED(config);

    //NRecommended::TaskForecasterDev(config);
    //SyncQueriesToMain(config);

    //NCatalogia::TaskBuildCatalogiaQueries(config);
    //NRecommendedUC2::TaskRecomendedUCQueries(config);
    //NLandings::TaskGenerateLandings(config);
    //NUserSessions::SortQueriesDev();
    //NMonitoring::TaskMonitoring(config);

    /*
    NYT::IClientPtr client = NYT::CreateClient("hahn.yt.yandex.net");

    NHost2Vec::TTableConfig host2VecTables(
        config.TABLE_RECOMMENDED_SOURCE_HOSTS,
        config.TABLE_RECOMMENDED_SOURCE_HOST2VEC,
        config.TABLE_RECOMMENDED_SOURCE_HOST2VEC_HOST_GROUPS,
        config.TABLE_RECOMMENDED_SOURCE_HOST2VEC_HOSTS,
        config.TABLE_RECOMMENDED_SOURCE_HOST2VEC_GROUPS
    );

    NHost2Vec::TaskUpdateHost2vecGroups(client, "key", host2VecTables);
    */
} catch (yexception &e) {
    LOG_WARN("unable to complete dev task: %s", e.what());
}

static void LogInfo(const TString &msg) {
    LOG_INFO("%s", msg.data());
}

int RunTask(const TTaskHandler &taskHandler, const TConfig &config) {
    if (!config.IsGlobalOk()) {
        throw yexception() << "during loading configuration files errors occurred";
    }

    TMonitor::Instance();
    TOpRunner::LogInfo = LogInfo;

    try {
        NYT::IClientPtr client = NYT::CreateClient(config.MR_SERVER_HOST_QUERIES);
        NYTUtils::CreatePath(client, config.TABLE_RECOMMENDED_ROOT);
        NYTUtils::CreatePath(client, config.TABLE_TEMP_ROOT);
        NYTUtils::CreatePath(client, config.TABLE_REPORTS_ROOT);
    } catch (yexception &e) {
        LOG_WARN("run task: %s\n", e.what());
    }

    try {
        taskHandler(config);
    } catch (const std::exception &e) {
        LOG_ERROR("something went wrong: %s", e.what());
    }

    TMonitor::Instance()->Shutdown();

    return 0;
}

TString GetAvailableModes(const TMap<TString, TTaskHandler> &taskHandlers) {
    TVector<TString> availableModes;

    for (const auto &obj : taskHandlers) {
        availableModes.push_back(obj.first);
    }

    return JoinSeq(" | ", availableModes);
}

} //namespace NWebmaster

int main(int argc, const char **argv) {
    NYT::Initialize(argc, argv);
    using namespace NWebmaster;

    NYTUtils::DisableLogger();

    int res = 1;

    TMap<TString, TTaskHandler> taskHandlers;
    taskHandlers["dev"]                         = TaskDev;
    taskHandlers["recommended"]                 = NRecommended::TaskForecaster;
    taskHandlers["recommended_uc"]              = NRecommendedUC2::TaskRecomendedUCQueries;
    taskHandlers["autocampaign_build_queries"]  = NAutoCampaign::TaskBuildAutoCampaignQueries;
    taskHandlers["catalogia2"]                  = NCatalogia2::TaskBuildCatalogiaQueries;

    TArgs::Init(argc, argv);
    TArgs::Opts().SetFreeArgsMax(1);
    TArgs::Opts().SetFreeArgTitle(0, GetAvailableModes(taskHandlers), " ");
    TArgs::Opts().AddLongOption("ignore-checks", "Run application in requested mode without checks (not recommended)").NoArgument();

    TVector<TString> freeArgs = TArgs::ParseOpts()->GetFreeArgs();

    TConfig::IgnoreChecks = TArgs::ParseOpts()->Has("ignore-checks");

    TString RequestedMode = "convert";
    if (!freeArgs.empty()) {
        RequestedMode = freeArgs.front();
    }

    LOG_INFO("Started");

    try {
        Cerr << "------ " << Now() << " ------" << Endl;

        TConfig::Instance().Load();

        TPeriodicLog::SetDefaultHandler(TConfig::CInstance().PERIODIC_LOG_HANDLER);

        const auto modeIt = taskHandlers.find(RequestedMode);
        if (modeIt == taskHandlers.end()) {
            LOG_WARN("unknown mode requested: %s", RequestedMode.data());
        } else {
            LOG_INFO("mode requested: %s", RequestedMode.data());
            res = RunTask(modeIt->second, TConfig::CInstance());
        }
    } catch (const std::exception &e) {
        LOG_CRIT("%s", e.what());
    }

    TMonitor::Instance()->Shutdown();
    LOG_INFO("Finished");

    return res;
}
