#include "monitor.h"

namespace NWebmaster {

TMonitor::TMonitor()
    : HistoryReportSuccess("Queries_HReportSuccess")
    , HistoryReportFailed("Queries_HReportFailed")
{
}

void TMonitor::Upload() {
    TMonitoringAgent<> agent;

    agent
        .Next(HistoryReportSuccess)
        .Next(HistoryReportFailed)
        .GlobalDerived()
        .Commit();
}

void MonitorPushHistory(const TString &suffix, const TMap<time_t, TMap<TString, double>> &history, const TString &prefix) {
    TSolomonPushBackend pushBackend("webmaster", "webmaster_performance_" + suffix);
    TMonitoringAgent<TSolomonPushBackend> agent(pushBackend);

    for (const auto &day : history) {
        const time_t timestamp = day.first;

        for (const auto& record : day.second) {
            const TString &source = record.first;
            const double &value = record.second;

            agent
                .Next(prefix + source, value) //daily anomalies
                .Timestamp(TInstant::Seconds(timestamp));
        }
    }

    agent
        .GlobalService("webmaster_quality")
        .Commit();
}

void MonitorPushLabel(const TString &suffix, const TString &label, const TString &service, double value) {
    TSolomonPushBackend pushBackend("webmaster", "webmaster_performance_" + suffix);
    TMonitoringAgent<TSolomonPushBackend> agent(pushBackend);

    agent
        .Next(label, value)
        .GlobalService(service)
        .Commit();
}

void MonitorPushQualityLabel(const TString &suffix, const TString &label, double value) {
    MonitorPushLabel(suffix, label, "webmaster_quality", value);
}

void MonitorPushCountersLabel(const TString &suffix, const TString &label, double value) {
    MonitorPushLabel(suffix, label, "webmaster_counters", value);
}

void MonitorSourceTables(const TString &suffix, int mostRecentSourceLag, int holes) {
    MonitorPushQualityLabel(suffix, "Queries_SourceLag", static_cast<double>(mostRecentSourceLag));
    MonitorPushQualityLabel(suffix, "Queries_SourceGaps", static_cast<double>(holes));
}

void MonitorAutocampaignDomains(const TString &suffix, size_t domains) {
    MonitorPushCountersLabel(suffix, "Autocampaign_Domains", static_cast<double>(domains));
}

void MonitorAutocampaignQueriesPTime(const TString &suffix, size_t minutes) {
    MonitorPushQualityLabel(suffix, "Autocampaign_QueriesProcessTime", static_cast<double>(minutes));
}

void MonitorCatalogiaLinksPTime(const TString &suffix, size_t minutes) {
    MonitorPushQualityLabel(suffix, "Catalogia_LinksProcessTime", static_cast<double>(minutes));
}

void MonitorCatalogiaQueriesPTime(const TString &suffix, size_t minutes) {
    MonitorPushQualityLabel(suffix, "Catalogia_QueriesProcessTime", static_cast<double>(minutes));
}

void MonitorCatalogiaHosts(const TString &suffix, size_t hosts) {
    MonitorPushCountersLabel(suffix, "Catalogia_Hosts", static_cast<double>(hosts));
}

void MonitorToForecasterQueries(const TString &suffix, size_t queries) {
    MonitorPushCountersLabel(suffix, "Recommended_ToForecaster_Queries", static_cast<double>(queries));
}

void MonitorToForecasterUCQueries(const TString &suffix, size_t queries) {
    MonitorPushCountersLabel(suffix, "RecommendedUC_ToForecaster_Queries", static_cast<double>(queries));
}

void UpdateHistoryMonitor(NYT::IClientBasePtr client, const TString &suffix, const TString &historyTable, const TString &prefix) {
    TMap<time_t, TMap<TString, double>> history;

    auto reader = client->CreateTableReader<NYT::TYaMRRow>(historyTable);
    for (; reader->IsValid(); reader->Next()) {
        const NYT::TYaMRRow &row = reader->GetRow();
        time_t timestamp = FromString<time_t>(TString{row.Key});
        TString source = TString{row.SubKey};
        double value = FromString<double>(TString{row.Value});
        history[timestamp][source] = value;
    }

    MonitorPushHistory(suffix, history, prefix);
}

} //namespace NWebmaster
