#include <util/draft/date.h>
#include <util/draft/datetime.h>

#include <wmconsole/version3/searchqueries-mr/conf/yt.h>
#include <wmconsole/version3/wmcutil/log.h>
#include <wmconsole/version3/wmcutil/regex.h>
#include <wmconsole/version3/wmcutil/yt/yt_runner.h>
#include <wmconsole/version3/wmcutil/yt/yt_utils.h>

#include "source_tables.h"

namespace NWebmaster {

TSourceTable::TSourceTable(const TString &name, ETableFormat format)
    : Name(name)
{
    const static char *FORMAT = "%Y-%m-%d";

    if (Name.empty()) {
        return;
    }

    if (format == E_FMT_CONVERTED_V4) {
        TVector<TString> period;
        TRegularExpression regex("(\\d+)_(\\d+)");
        if (regex.GetMatches(Name, period) != 2) {
            ythrow yexception() << "unable to parse source table name " << Name;
        }

        NamePeriodBegin = period[0];
        NamePeriodEnd = period[1];
        PeriodBegin = str2date(NamePeriodBegin);
        PeriodEnd = str2date(NamePeriodEnd);
    } else if (format == E_FMT_USER_SESSIONS) {
        PeriodBegin = TDate(NYTUtils::GetTableName(name), FORMAT).GetStart();
        PeriodEnd = PeriodBegin;
    } else {
        ythrow yexception () << "unknown table format";
    }
}

int TSourceTable::AgeDays() const {
    return AgeHours() / 24;
}

int TSourceTable::AgeHours() const {
    return (Now() - TInstant::Seconds(PeriodBegin)).Hours();
}

bool TSourceTable::IsDaily() const {
    return PeriodBegin == PeriodEnd;
}

bool TSourceTable::operator<(const TSourceTable &rhs) const {
    return PeriodBegin < rhs.PeriodBegin;
}

TSourceTablesForTarget::TSourceTablesForTarget(NYT::IClientBasePtr client, size_t period, const TString &targetTable)
    : TargetTable(targetTable)
{
    if (!LoadUserSessions(client, TCommonYTConfigSQ::CInstance().TABLE_PARSED_USER_SESSIONS_DAILY_ROOT, Tables, period)) {
        ythrow yexception() << "TSourceTablesForTarget, there are no source tables, target: " << targetTable;
    }
    Validate(client);
}

TSourceTablesForTarget::TSourceTablesForTarget(NYT::IClientBasePtr client, const TSourceTablesForTarget &rhs, size_t period, const TString &targetTable)
    : TargetTable(targetTable)
{
    if (period > rhs.Tables.size()) {
        ythrow yexception() << "not enough source tables";
    }
    Tables.insert(Tables.begin(), rhs.Tables.end() - period, rhs.Tables.end());
    Validate(client);
}

TSourceTablesForTarget::TSourceTablesForTarget(NYT::IClientBasePtr client, const TSourceTablesForTarget &rhs, const TString &targetTable)
    : Tables(rhs.Tables)
    , TargetTable(targetTable)
{
    Validate(client);
}

void TSourceTablesForTarget::Validate(NYT::IClientBasePtr client) {
    LeastRecentSource = Tables.begin()->Name;
    MostRecentSource = Tables.rbegin()->Name;
    IsUpdated = !NeedUpdate(client);
    IsComplete = IsSourceTablesSetComplete(Tables, Lag, Holes);
}

bool TSourceTablesForTarget::NeedUpdate(NYT::IClientBasePtr client) {
    TString tag;

    try {
        tag = NYTUtils::GetAttr(client, TargetTable, ATTR_MOST_RECENT_SOURCE_NAME).AsString();
        LOG_INFO("TSourceTablesForTarget, table tag %s/@%s=%s", TargetTable.data(), ATTR_MOST_RECENT_SOURCE_NAME, tag.data());
    } catch (yexception &e) {
        LOG_WARN("TSourceTablesForTarget, unable to get table tag %s", e.what());
    }

    return tag != MostRecentSource;
}

void TSourceTablesForTarget::UpdateTarget(NYT::IClientBasePtr client) {
    NYTUtils::SetAttr(client, TargetTable, ATTR_LEAST_RECENT_SOURCE_NAME, LeastRecentSource);
    NYTUtils::SetAttr(client, TargetTable, ATTR_MOST_RECENT_SOURCE_NAME, MostRecentSource);
}

bool IsSourceTablesSetComplete(const TDeque<TSourceTable> &sourceTables, int &mostRecentSourceLag, int &holes) {
    if (sourceTables.empty()) {
        return false;
    }

    bool complete = true;

    TSourceTable prev = sourceTables[0];
    for (size_t i = 1; i < sourceTables.size(); i++) {
        const TSourceTable &table = sourceTables[i];
        int diff = prev.AgeDays() - table.AgeDays();
        int gap = diff - 1;
        holes += gap;
        if (diff != 1) {
            complete = false;
            LOG_ERROR("gap in source tables %s .. %s (%d days)", prev.Name.data(), table.Name.data(), gap);
        }
        prev = table;
    }

    mostRecentSourceLag = sourceTables.rbegin()->AgeHours();
    return complete;
}

bool LoadSourceTables(NYT::IClientBasePtr client, const TString &prefix, TDeque<TSourceTable> &sourceTables, int days, TSourceTable::ETableFormat format) {
    TDeque<NYTUtils::TTableInfo> tables;
    NYTUtils::GetTableList(client, prefix, tables, Max<size_t>());

    sourceTables.clear();
    for (const NYTUtils::TTableInfo &table : tables) {
        TSourceTable source(table.Name, format);
        if (source.AgeDays() < days) {
            sourceTables.push_back(source);
        }
    }

    std::sort(sourceTables.begin(), sourceTables.end());
    return !sourceTables.empty();
}

bool LoadConvertedTables(NYT::IClientBasePtr client, const TString &prefix, TDeque<TSourceTable> &sourceTables, int days) {
    return LoadSourceTables(client, prefix, sourceTables, days, TSourceTable::E_FMT_CONVERTED_V4);
}

bool LoadUserSessions(NYT::IClientBasePtr client, const TString &prefix, TDeque<TSourceTable> &sourceTables, int days) {
    return LoadSourceTables(client, prefix, sourceTables, days, TSourceTable::E_FMT_USER_SESSIONS);
}

} //namespace NWebmaster
