#include <util/charset/wide.h>
#include <util/digest/fnv.h>
#include <util/draft/datetime.h>
#include <util/random/random.h>
#include <util/string/join.h>
#include <util/string/strip.h>
#include <util/string/type.h>
#include <library/cpp/string_utils/url/url.h>
#include <util/thread/pool.h>

#include <library/cpp/uri/common.h>
#include <library/cpp/containers/comptrie/comptrie.h>
#include <library/cpp/containers/comptrie/prefix_iterator.h>
#include <library/cpp/tld/tld.h>

#include <mapreduce/yt/interface/client.h>

#include <yweb/protos/links.pb.h>
#include <yweb/robot/preparat/io/io.h>

#include <wmconsole/version3/wmcutil/args.h>
#include <wmconsole/version3/wmcutil/yt/yt_runner.h>
#include <wmconsole/version3/wmcutil/yt/yt_utils.h>
#include <wmconsole/version3/wmcutil/yt/transfer_manager.h>
#include <wmconsole/version3/wmcutil/url.h>

#include "batch_matcher.h"
#include "config.h"
#include "forecaster.h"
#include "host2vec.h"
#include "monitor.h"
#include "rivals.h"
#include "task_recommended.h"

namespace NWebmaster {
namespace NRecommended {

namespace {
//const char *F_JUPITER_HOST = "Host";
//const char *F_JUPITER_METADECSR = "MetaDescription";
//const char *F_JUPITER_PATH = "Path";
//const char *F_JUPITER_TITLE = "TitleRawUTF8";
//const char *F_METADESCR_HASHES = "metaDescription_hashes";
//const char *F_METADESCR_ID = "metaDescription_id";
//const char *F_ORIGIN = "Origin";
//const char *F_PATH = "Path";
//const char *F_TITLE_ID = "title_id";
//const char *F_TITLES_HASHES = "titles_hashes";
const char *F_YAMR_KEY      = "key";
const char *F_QUERIES_COUNT = "QueriesCount";
}

/*
//reduce by Host, Path
void TReduceExtractTextsFromJupiter::Do(TReader *input, TWriter *output) {
    const NYT::TNode &row = input->GetRow();
    const TString url = row[F_JUPITER_HOST].AsString() + row[F_JUPITER_PATH].AsString();
    const TString title = NYTUtils::FromNodeOrDefault<TString>(row[F_JUPITER_TITLE], "");
    const TString metaDescription = NYTUtils::FromNodeOrDefault<TString>(row[F_JUPITER_METADECSR], "");
    if (!title.empty() || !metaDescription.empty()) {
        output->AddRow(NYT::TNode()
            (F_URL, url)
            (F_TITLE, title)
            (F_METADESCR, metaDescription)
        );
    }
}

REGISTER_REDUCER(TReduceExtractTextsFromJupiter)

//ReduceBy url
void TReduceEnrichQueries::Do(TReader *input, TWriter *output) {
    const int TABLENO_TEXTS = 0;

    TString title, metaDescription;
    for (; input->IsValid(); input->Next()) {
        NYT::TNode row = input->GetRow();
        if (input->GetTableIndex() == TABLENO_TEXTS) {
            title = row[F_TITLE].AsString();
            metaDescription = row[F_METADESCR].AsString();
        } else {
            row
                (F_TITLE, title)
                (F_METADESCR, metaDescription)
            ;
            output->AddRow(row);
        }
    }
}
REGISTER_REDUCER(TReduceEnrichQueries)

void TMapFilterNonGroups::Do(TReader *input, TWriter *output) {
    for (; input->IsValid(); input->Next()) {
        const NYT::TNode &row = input->GetRow();
        TString domain;
        if (!GetHostWithoutScheme(row[NHost2Vec::F_HOST].AsString(), domain)) {
            continue;
        }
        if (domain != row[NHost2Vec::F_GROUP].AsString()) {
            continue;
        }
        output->AddRow(row);
    }
}

REGISTER_MAPPER(TMapFilterNonGroups)
*/

//ReduceBy F_QUERY, F_HOST, F_REGION_ID
void TJoinSourceQueriesReducer::Do(TReader *input, TWriter *output) {
    const int TABLENO_SOURCE_QUERIES = 0;
    const int TABLENO_FORECASTED_GROUPS = 1;

    size_t totalShows = 0;
    size_t totalClicks = 0;
    size_t weightedPosition = 0;
    bool thereIsSource = false;
    for (; input->IsValid(); input->Next()) {
        NYT::TNode row = input->GetRow();
        switch(input->GetTableIndex()) {
        case TABLENO_SOURCE_QUERIES: {
            size_t shows = row[F_SHOWS].AsUint64();
            totalShows += shows;
            totalClicks += row[F_CLICKS].AsUint64();
            weightedPosition += row[F_POSITION].AsUint64() * shows;
            thereIsSource = true;
            break;
        }
        case TABLENO_FORECASTED_GROUPS:
            if (thereIsSource) {
                //double rel1 = ((row[F_FORECASTED_CLICKS].AsDouble() - static_cast<double>(clicks)) / static_cast<double>(row[F_FORECASTED_BID].AsInt64() / 1000000.0)) / static_cast<double>(row[F_RIVALS_SHOWS].AsUint64());

                //https://st.yandex-team.ru/WMC-3948#1496650699000
                //double rel1 = LOG(UniqRivalsByShows) * (����(ForecastedClicks < 1; 0; LOG10(ForecastedClicks ** 2) + 0,1)) / (ForecastedBid ** 2)
                double fc = row[F_FORECASTED_CLICKS].AsDouble();
                double fb = static_cast<double>(row[F_FORECASTED_BID].AsInt64());
                double rubs = static_cast<double>(row[F_RIVALS_UNIQ_BY_SHOWS].AsUint64());
                double rel1 = std::log10(rubs);
                if (fc < 1.0) {
                    rel1 = 0;
                } else {
                    rel1 *= (std::log10(fc * fc) + 0.1);
                }
                rel1 /= (fb * fb);

                if (totalShows > 0) {
                    size_t position = weightedPosition / totalShows;
                    output->AddRow(row
                        (F_SHOWS, totalShows)
                        (F_CLICKS, totalClicks)
                        (F_POSITION, position)
                        (F_REL1, rel1)
                    );
                }
            }
            break;
        default:
            ythrow yexception() << "unknown input table";
        }
    }
}

REGISTER_REDUCER(TJoinSourceQueriesReducer)

//ReduceBy F_GROUP
struct TEnrichedGroupsCountReducer : public NYT::IReducer<NYT::TTableReader<NYT::TNode>, NYT::TTableWriter<NYT::TNode>> {
    void Do(TReader *input, TWriter *output) override {
        const TString group = input->GetRow()[NHost2Vec::F_GROUP].AsString();
        size_t queriesCount = 0;
        for (; input->IsValid(); input->Next()) {
            const NYT::TNode &row = input->GetRow();
            queriesCount += NYTUtils::FromNodeOrDefault<ui64>(row[F_QUERIES_COUNT], 1);
        }
        output->AddRow(NYT::TNode()
            (NHost2Vec::F_GROUP, group)
            (F_QUERIES_COUNT, queriesCount)
        );
    }
};

REGISTER_REDUCER(TEnrichedGroupsCountReducer)

struct TEnrichedGroupsThinOutMapper : public NYT::IMapper<NYT::TTableReader<NYT::TNode>, NYT::TTableWriter<NYT::TNode>> {
    Y_SAVELOAD_JOB(GroupCounters)

    TEnrichedGroupsThinOutMapper() = default;
    TEnrichedGroupsThinOutMapper(const THashMap<TString, double> &groupCounters)
        : GroupCounters(groupCounters)
    {
    }

public:
    void Do(TReader *input, TWriter *output) override {
        const double QUERIES_LIMIT = 3000;
        for (; input->IsValid(); input->Next()) {
            const NYT::TNode &row = input->GetRow();
            const TString group = row[NHost2Vec::F_GROUP].AsString();
            auto counterIt = GroupCounters.find(group);
            if (counterIt != GroupCounters.end()) {
                const double groupSize = counterIt->second;
                const double p = QUERIES_LIMIT / groupSize;
                if (groupSize < QUERIES_LIMIT || RandomNumber<double>() < p) {
                    output->AddRow(row);
                }
            }
        }
    }

public:
    THashMap<TString, double> GroupCounters;
};

REGISTER_MAPPER(TEnrichedGroupsThinOutMapper)

struct TEnrichedGroupsFilterMapper : public NYT::IMapper<NYT::TTableReader<NYT::TNode>, NYT::TTableWriter<NYT::TNode>> {
    void Do(TReader *input, TWriter *output) override {
        for (; input->IsValid(); input->Next()) {
            const NYT::TNode row = input->GetRow();
            const TString group = row[NHost2Vec::F_GROUP].AsString();
            const TString host = row[NHost2Vec::F_HOST].AsString();
            const TString url = row[F_URL].AsString();
            TString urlHost, urlPath;
            if (!NUtils::SplitUrl(url, urlHost, urlPath)) {
                continue;
            }

            const TString fixedHost = TString{NUtils::GetHost2vecDomain(host)};
            const TString fixedUrlHost = TString{NUtils::GetHost2vecDomain(urlHost)};
            const size_t position = row[F_POSITION].AsUint64(); //if position contains large value, there was no shows, row can be skipped

            if (group == fixedHost && group == fixedUrlHost && position < 100) {
                output->AddRow(row);
            }
        }
    }
};

REGISTER_MAPPER(TEnrichedGroupsFilterMapper)
/*
int TaskBuildTexts(const TConfig &config) {
    if (!config.CATALOGIA_ENABLED_BUILD_TITLES) {
        LOG_WARN("building texts is disabled");
        return 0;
    }

    NYT::IClientPtr client = NYT::CreateClient(config.MR_SERVER_HOST_QUERIES);
    NYT::ITransactionPtr tx = client->StartTransaction();

    TOpRunner(tx)
        .InputNode(config.TABLE_SOURCE_JUPITER_TEXTS)
        .OutputNode(config.TABLE_SOURCE_TEXTS)
        .ReduceBy(F_JUPITER_HOST, F_JUPITER_PATH)
        .Reduce(new TReduceExtractTextsFromJupiter)
        .SortBy(F_URL)
        .Sort(config.TABLE_SOURCE_TEXTS)
    ;

    tx->Commit();

    TOpRunner(client).Archive(config.TABLE_SOURCE_TEXTS);

    return 0;
}
*/
void MonitorForecasterQueries(NYT::IClientBasePtr client, const TConfig &config, const TString &table) try {
    NYTUtils::TTableInfo info;
    NYTUtils::GetTableInfo(client, table, info);
    MonitorToForecasterQueries(config.MONITOR_PERFORMANCE_SUFFIX, info.RecordCount);
} catch (yexception &e) {
    LOG_WARN("recommended: unable to monitor table %s: %s", table.data(), e.what());
}

void ReadGroupCounters(NYT::IClientBasePtr client, const TString &sourceTable, THashMap<TString, double> &groupCounters) {
    THashMap<TString, double> tmpCounters;
    auto reader = client->CreateTableReader<NYT::TNode>(sourceTable);
    for (; reader->IsValid(); reader->Next()) {
        const NYT::TNode &row = reader->GetRow();
        const TString group = row[NHost2Vec::F_GROUP].AsString();
        tmpCounters[group] = static_cast<double>(row[F_QUERIES_COUNT].AsUint64());
    }
    groupCounters.swap(tmpCounters);
}

void TaskForecaster(const TConfig &config) try {
    if (!config.RECOMMENDED_ENABLED_BUILD_DATA) {
        LOG_INFO("recommended, queries building is disabled");
        return;
    }

    NYT::IClientPtr client = NYT::CreateClient(config.MR_SERVER_HOST_QUERIES);

    NForecaster::TForecastTable forecaster(client, config.TABLE_RECOMMENDED_QUERIES_TO_FORECASTER, config.TABLE_RECOMMENDED_QUERIES_FROM_FORECASTER);

    if (forecaster.IsForecastPending()) {
        LOG_INFO("direct, there is pending forecaster request %ld", forecaster.Timestamp);
        MonitorForecasterQueries(client, config, forecaster.GetSourceTable());
    } else {
        LOG_INFO("direct, there is no pending forecaster request");

        NHost2Vec::TTableConfig host2VecTables(
            config.TABLE_RECOMMENDED_SOURCE_HOSTS,
            config.TABLE_RECOMMENDED_SOURCE_HOST2VEC,
            config.TABLE_RECOMMENDED_SOURCE_HOST2VEC_HOST_GROUPS,
            config.TABLE_RECOMMENDED_SOURCE_HOST2VEC_HOSTS,
            config.TABLE_RECOMMENDED_SOURCE_HOST2VEC_GROUPS
        );

        NHost2Vec::TaskUpdateHost2vecGroups(client, F_YAMR_KEY, host2VecTables);

        TRivalsTables rivalsTables(
            config.TABLE_RECOMMENDED_EXTRACTED_QUERIES,
            config.TABLE_RECOMMENDED_EXTRACTED_QUERIES_FILTER,
            config.TABLE_RECOMMENDED_ENRICHED_QUERIES,
            config.TABLE_RECOMMENDED_IDS_QUERIES,
            config.TABLE_RECOMMENDED_IDS_URLS,
            config.TABLE_RECOMMENDED_INTM_RAW_GROUPS,
            config.TABLE_RECOMMENDED_INTM_ENRICHED_GROUPS,
            config.TABLE_RECOMMENDED_ENRICHED_GROUPS
        );

        //Positions in java backend starts from 1.0
        TaskBuildQueries(client, config, host2VecTables, rivalsTables, 1, true);
        TaskEnrichQueries(client, config, rivalsTables);
        TaskGroupQueries(client, config, host2VecTables, rivalsTables, true);

        const TString enrichedGroups = config.TABLE_RECOMMENDED_ENRICHED_GROUPS;
        const TString enrichedGroupsCounters = enrichedGroups + ".counters";
        const TString enrichedGroupsUnchanged = enrichedGroups + ".unchanged";

        NYT::ITransactionPtr tx = client->StartTransaction();

        TOpRunner(tx)
            .Copy(enrichedGroups, enrichedGroupsUnchanged)
            .InputNode(enrichedGroups)
            .OutputNode(NYT::TRichYPath(enrichedGroups).Schema(NYTUtils::GetTableSchema(tx, enrichedGroups)))
            .EnableOrderedMap()
            .Map(new TEnrichedGroupsFilterMapper)

            .InputNode(enrichedGroups)
            .OutputNode(enrichedGroupsCounters)
            .ReduceBy(NHost2Vec::F_GROUP)
            .MapReduce(nullptr, new TEnrichedGroupsCountReducer, new TEnrichedGroupsCountReducer)
        ;

        THashMap<TString, double> groupCounters;
        ReadGroupCounters(tx, enrichedGroupsCounters, groupCounters);

        TOpRunner(tx)
            .InputNode(enrichedGroups)
            .OutputNode(NYT::TRichYPath(enrichedGroups).Schema(NYTUtils::GetTableSchema(tx, enrichedGroups)))
            .EnableOrderedMap()
            .Map(new TEnrichedGroupsThinOutMapper(groupCounters))
        ;

        const TString output = forecaster.PostTableToForecast(tx);

        TOpRunner(tx)
            .SortBy(F_QUERY, F_REGION_ID)
            .Sort(enrichedGroups)
            .InputNode(config.TABLE_RECOMMENDED_ENRICHED_GROUPS)
            .OutputNode(NYT::TRichYPath(output).Schema(NForecaster::GetToForecasterSchema()))
            .ReduceBy(F_QUERY, F_REGION_ID)
            .Reduce(new NForecaster::TReduceQueriesToForecaster)
        ;

        tx->Commit();

        MonitorForecasterQueries(client, config, forecaster.GetSourceTable());
        LOG_INFO("direct, posted request to forecaster %s", output.data());
    }

    LOG_INFO("direct, waiting for result from forecaster");
    while(!forecaster.IsThereForecast()) {
        Sleep(TDuration::Minutes(5));
    }

    const TString rawForecasted = forecaster.GetForecastTable();
    const TString convertedForecasted = NYTUtils::JoinPath(config.TABLE_TEMP_ROOT, "direct", "forecasted_" + ToString(forecaster.Timestamp));

    LOG_INFO("direct, there is forecasted queries %s", rawForecasted.data());

    NYT::TTableSchema convertedSchema;
    convertedSchema.Strict(true);
    convertedSchema.AddColumn(NYT::TColumnSchema().Name(F_QUERY).Type(NYT::VT_STRING));
    convertedSchema.AddColumn(NYT::TColumnSchema().Name(F_REGION_ID).Type(NYT::VT_UINT64));
    convertedSchema.AddColumn(NYT::TColumnSchema().Name(F_FORECASTED_BID).Type(NYT::VT_INT64));
    convertedSchema.AddColumn(NYT::TColumnSchema().Name(F_FORECASTED_BUDGET).Type(NYT::VT_DOUBLE));
    convertedSchema.AddColumn(NYT::TColumnSchema().Name(F_FORECASTED_SHOWS).Type(NYT::VT_INT64));
    convertedSchema.AddColumn(NYT::TColumnSchema().Name(F_FORECASTED_CLICKS).Type(NYT::VT_DOUBLE));

    NYT::TTableSchema forecastedSchema = NYTUtils::GetTableSchema(client, config.TABLE_RECOMMENDED_ENRICHED_GROUPS);
    forecastedSchema.AddColumn(NYT::TColumnSchema().Name(F_FORECASTED_BID).Type(NYT::VT_INT64));
    forecastedSchema.AddColumn(NYT::TColumnSchema().Name(F_FORECASTED_BUDGET).Type(NYT::VT_DOUBLE));
    forecastedSchema.AddColumn(NYT::TColumnSchema().Name(F_FORECASTED_SHOWS).Type(NYT::VT_INT64));
    forecastedSchema.AddColumn(NYT::TColumnSchema().Name(F_FORECASTED_CLICKS).Type(NYT::VT_DOUBLE));

    NYT::ITransactionPtr tx = client->StartTransaction();
    TOpRunner(tx)
        .InputNode(rawForecasted)
        .OutputNode(NYT::TRichYPath(convertedForecasted).Schema(convertedSchema))
        .Map(new NForecaster::TMapQueriesFromForecaster)
        .SortBy(F_QUERY, F_REGION_ID)
        .Sort(convertedForecasted)

        .InputNode(convertedForecasted)
        .InputNode(config.TABLE_RECOMMENDED_ENRICHED_GROUPS)
        .OutputNode(NYT::TRichYPath(config.TABLE_RECOMMENDED_FORECASTED_GROUPS).Schema(forecastedSchema))
        .ReduceBy(F_QUERY, F_REGION_ID)
        .Reduce(new NForecaster::TReduceJoinForecatedData)
        .Drop(convertedForecasted)
    ;

    forecastedSchema = NYTUtils::DropSortOrder(forecastedSchema);
    forecastedSchema.AddColumn(NYT::TColumnSchema().Name(F_SHOWS).Type(NYT::VT_UINT64));
    forecastedSchema.AddColumn(NYT::TColumnSchema().Name(F_CLICKS).Type(NYT::VT_UINT64));
    forecastedSchema.AddColumn(NYT::TColumnSchema().Name(F_REL1).Type(NYT::VT_DOUBLE));

    TOpRunner(tx)
        .SortBy(F_QUERY, NHost2Vec::F_HOST, F_REGION_ID)
        .Sort(config.TABLE_RECOMMENDED_ENRICHED_QUERIES, ASYNC_CTX0)
        .SortBy(F_QUERY, NHost2Vec::F_HOST, F_REGION_ID)
        .Sort(config.TABLE_RECOMMENDED_FORECASTED_GROUPS, ASYNC_CTX0)
        .Wait(ASYNC_CTX0)
        .InputNode(config.TABLE_RECOMMENDED_ENRICHED_QUERIES)
        .InputNode(config.TABLE_RECOMMENDED_FORECASTED_GROUPS)
        .OutputNode(NYT::TRichYPath(config.TABLE_RECOMMENDED_FORECASTED_GROUPS).Schema(forecastedSchema))
        .ReduceBy(F_QUERY, NHost2Vec::F_HOST, F_REGION_ID)
        .Reduce(new TJoinSourceQueriesReducer)
        .Drop(config.TABLE_RECOMMENDED_ENRICHED_QUERIES)
    ;

    LOG_INFO("direct, forecaster tag %ld will be removed", forecaster.Timestamp);
    forecaster.Reset(tx);

    tx->Commit();

} catch (yexception &e) {
    LOG_ERROR("unable to complete recommended task: %s", e.what());
}

void TaskForecasterDev(const TConfig &config) {
    Y_UNUSED(config);

    NYT::IClientPtr client = NYT::CreateClient(config.MR_SERVER_HOST_QUERIES);
    //NYT::ITransactionPtr tx = client->StartTransaction();

    NHost2Vec::TTableConfig host2VecTables(
        config.TABLE_RECOMMENDED_SOURCE_HOSTS,
        config.TABLE_RECOMMENDED_SOURCE_HOST2VEC,
        config.TABLE_RECOMMENDED_SOURCE_HOST2VEC_HOST_GROUPS,
        config.TABLE_RECOMMENDED_SOURCE_HOST2VEC_HOSTS,
        config.TABLE_RECOMMENDED_SOURCE_HOST2VEC_GROUPS
    );

    //NHost2Vec::TaskUpdateHost2vecGroups(client, F_YAMR_KEY, host2VecTables);

    TRivalsTables rivalsTables(
        config.TABLE_RECOMMENDED_EXTRACTED_QUERIES,
        config.TABLE_RECOMMENDED_EXTRACTED_QUERIES_FILTER,
        config.TABLE_RECOMMENDED_ENRICHED_QUERIES,
        config.TABLE_RECOMMENDED_IDS_QUERIES,
        config.TABLE_RECOMMENDED_IDS_URLS,
        config.TABLE_RECOMMENDED_INTM_RAW_GROUPS,
        config.TABLE_RECOMMENDED_INTM_ENRICHED_GROUPS,
        config.TABLE_RECOMMENDED_ENRICHED_GROUPS
    );

    //Positions in java backend starts from 1.0
    TaskBuildQueries(client, config, host2VecTables, rivalsTables, 1, false);

/*
    TOpRunner(tx)
        .Move(config.TABLE_RECOMMENDED_EXTRACTED_QUERIES, config.TABLE_RECOMMENDED_ENRICHED_QUERIES)

        .SortBy(F_QUERY, NHost2Vec::F_HOST, F_REGION_ID)
        .Sort(config.TABLE_RECOMMENDED_ENRICHED_QUERIES, ASYNC_CTX0)
        .SortBy(F_QUERY, NHost2Vec::F_HOST, F_REGION_ID)
        .Sort(config.TABLE_RECOMMENDED_FORECASTED_GROUPS, ASYNC_CTX0)
        .Wait(ASYNC_CTX0)
        .InputNode(config.TABLE_RECOMMENDED_ENRICHED_QUERIES)
        .InputNode(config.TABLE_RECOMMENDED_FORECASTED_GROUPS)
        .OutputNode(NYT::TRichYPath(config.TABLE_RECOMMENDED_FORECASTED_GROUPS).Schema(NYTUtils::GetTableSchema(tx, config.TABLE_RECOMMENDED_FORECASTED_GROUPS)))
        .ReduceBy(F_QUERY, NHost2Vec::F_HOST, F_REGION_ID)
        .Reduce(new TJoinSourceQueriesReducer)
    ;

*/
    //const TString enrichedGroups = config.TABLE_RECOMMENDED_ENRICHED_GROUPS;
    //const TString enrichedGroupsCounters = enrichedGroups + ".counters";
    //const TString enrichedGroupsUnchanged = enrichedGroups + ".unchanged";

    //TOpRunner(tx)
    //    .InputNode(enrichedGroups)
    //    .OutputNode(enrichedGroupsCounters)
    //    .ReduceBy(NHost2Vec::F_GROUP)
    //    .MapReduce(nullptr, new TEnrichedGroupsCountReducer, new TEnrichedGroupsCountReducer)
    //;

/*
    TOpRunner(tx)
        .InputNode("//home/webmaster/prod/searchqueries/direct/forecasted_groups")
        .OutputNode(NYT::TRichYPath("//home/webmaster/prod/searchqueries/direct/forecasted_groups").Schema(NYTUtils::GetTableSchema(tx, "//home/webmaster/prod/searchqueries/direct/forecasted_groups")))
        .EnableOrderedMap()
        .Map(new TEnrichedGroupsFilterMapper)
    ;

    tx->Commit();
*/

/*
    NYT::TTableSchema forecastedSchema = NYTUtils::GetTableSchema(client, config.TABLE_RECOMMENDED_ENRICHED_GROUPS);
    forecastedSchema.AddColumn(NYT::TColumnSchema().Name(F_FORECASTED_BID).Type(NYT::VT_INT64));
    forecastedSchema.AddColumn(NYT::TColumnSchema().Name(F_FORECASTED_BUDGET).Type(NYT::VT_DOUBLE));
    forecastedSchema.AddColumn(NYT::TColumnSchema().Name(F_FORECASTED_SHOWS).Type(NYT::VT_INT64));
    forecastedSchema.AddColumn(NYT::TColumnSchema().Name(F_FORECASTED_CLICKS).Type(NYT::VT_DOUBLE));

    forecastedSchema = NYTUtils::DropSortOrder(forecastedSchema);
    forecastedSchema.AddColumn(NYT::TColumnSchema().Name(F_SHOWS).Type(NYT::VT_UINT64));
    forecastedSchema.AddColumn(NYT::TColumnSchema().Name(F_CLICKS).Type(NYT::VT_UINT64));
    forecastedSchema.AddColumn(NYT::TColumnSchema().Name(F_REL1).Type(NYT::VT_DOUBLE));

    TOpRunner(tx)
        .SortBy(F_QUERY, NHost2Vec::F_HOST, F_REGION_ID, F_POSITION)
        .Sort(config.TABLE_RECOMMENDED_ENRICHED_QUERIES, ASYNC_CTX0)
        .SortBy(F_QUERY, NHost2Vec::F_HOST, F_REGION_ID, F_POSITION)
        .Sort(config.TABLE_RECOMMENDED_FORECASTED_GROUPS, ASYNC_CTX0)
        .Wait(ASYNC_CTX0)
        .InputNode(config.TABLE_RECOMMENDED_ENRICHED_QUERIES)
        .InputNode(config.TABLE_RECOMMENDED_FORECASTED_GROUPS)
        .OutputNode(NYT::TRichYPath(config.TABLE_RECOMMENDED_FORECASTED_GROUPS).Schema(forecastedSchema))
        .ReduceBy(F_QUERY, NHost2Vec::F_HOST, F_REGION_ID, F_POSITION)
        .Reduce(new TJoinSourceQueriesReducer)
    ;
*/
/*
    NYT::TTableSchema schemaQueriesFilter;
    schemaQueriesFilter.Strict(true);
    schemaQueriesFilter.AddColumn(NYT::TColumnSchema().Name(F_QUERY).Type(NYT::VT_STRING).SortOrder(NYT::SO_ASCENDING));
    schemaQueriesFilter.AddColumn(NYT::TColumnSchema().Name(F_SHOWS).Type(NYT::VT_UINT64));
    schemaQueriesFilter.AddColumn(NYT::TColumnSchema().Name(F_CLICKS).Type(NYT::VT_UINT64));

    TOpRunner(client)

        .InputNode(config.TABLE_RECOMMENDED_EXTRACTED_QUERIES)
        .OutputNode(NYT::TRichYPath(config.TABLE_RECOMMENDED_EXTRACTED_QUERIES_FILTER).Schema(schemaQueriesFilter))
        .ReduceBy(F_QUERY)
        .Reduce(new TPreparePreFilterReducer)

        .InputNode(config.TABLE_RECOMMENDED_EXTRACTED_QUERIES_FILTER)
        .InputNode("//home/webmaster/prod/searchqueries/direct/enriched_groups")
        .OutputNode("//home/webmaster/prod/searchqueries/direct/enriched_groups_filtered_20_c1")
        .ReduceBy(F_QUERY)
        .Reduce(new TPreFilterQueriesReducer)
    ;
*/
}

} //namespace NRecommended
} //namespace NWebmaster
