#include <util/string/vector.h>

#include <library/cpp/getopt/last_getopt.h>

#include <mapreduce/yt/interface/client.h>

#include <mapreduce/yt/common/config.h>
#include <wmconsole/version3/wmcutil/yt/yt_runner.h>
#include <wmconsole/version3/wmcutil/log.h>
#include <wmconsole/version3/wmcutil/args.h>

#include "compressor.h"
#include "partition.h"
#include "run_config.h"
#include "task_achievements.h"
#include "task_digest_messages.h"
#include "task_excluded.h"
#include "task_host_threats.h"
#include "task_iks.h"
#include "task_indexed.h"
#include "task_indexed_url_samples.h"
#include "task_indexing_event_samples.h"
#include "task_indexing_events.h"
#include "task_indexing_events.h"
#include "task_link_stats.h"
#include "task_links.h"
#include "task_memorandum.h"
#include "task_newgone.h"
#include "task_queries_top_urls.h"
#include "task_queries_week.h"
#include "task_query_groups.h"
#include "task_query_stats.h"
#include "task_recommended_queries.h"
#include "task_recommended_urls.h"
#include "task_turbo_urls.h"
#include "task_urls.h"
#include "tasks.h"
#include "util.h"

namespace NWebmaster {

static void LogInfo(const TString &msg) {
    LOG_INFO("%s", msg.data());
}

} //namespace NWebmaster

int main(int argc, const char **argv) {
    using namespace NWebmaster;

    NYT::Initialize(argc, argv);
    NYTUtils::DisableLogger();
    TOpRunner::LogInfo = LogInfo;

    TArgs::Init(argc, argv);

    TString taskTypeString;
    TString stage;
    size_t outTablesCount;
    size_t textsTablesCount;
    size_t outLinesCount;
    TString dateFieldValue;
    TString exportQueryTextS;
    time_t dateFrom = 0;
    time_t dateTo = 0;
    TString tableDatesString;

    TArgs::Opts()
        .AddLongOption("task", "Task type (SEARCH_URL_SAMPLES, SEARCH_URL_EVENT_SAMPLES, EXCLUDED_URL_SAMPLES, etc)")
        .StoreResult(&taskTypeString);
    TArgs::Opts()
        .AddLongOption("stage", "Stage (map or reduce)")
        .StoreResult(&stage);
    TArgs::Opts()
        .AddLongOption("tables", "Number of output tables")
        .StoreResult(&outTablesCount);
    TArgs::Opts()
        .AddLongOption("texts-tables", "Number of query texts tables")
        .StoreResult(&textsTablesCount);
    TArgs::Opts()
        .AddLongOption("lines", "Number of lines in each output table")
        .StoreResult(&outLinesCount)
        .DefaultValue("0");
    TArgs::Opts()
        .AddLongOption("date", "Provided 'date' field value")
        .StoreResult(&dateFieldValue);
    TArgs::Opts()
        .AddLongOption("query-text", "Export query text (for query stats only)")
        .StoreResult(&exportQueryTextS);
    TArgs::Opts()
        .AddLongOption("date-from", "Export from date")
        .StoreResult(&dateFrom);
    TArgs::Opts()
        .AddLongOption("date-to", "Export to date")
        .StoreResult(&dateTo);
    TArgs::Opts()
        .AddLongOption("table-dates", "Dates for query group tables")
        .StoreResult(&tableDatesString);
    TArgs::ParseOpts();

    ETaskType taskType = FromString<ETaskType>(taskTypeString);

    if (stage == "map") {
        if (outLinesCount == 0) {
            ythrow yexception() << "Lines count is required for map stage";
        }
        if (taskType == INDEXED_URLS_COUNT || taskType == SEARCH_QUERIES_WEEK || taskType == RECOMMENDED_URLS) {
            if (dateFieldValue.empty()) {
                ythrow yexception() << "date is required for INDEXED_URLS_COUNT and SEARCH_QUERIES_WEEK tasks";
            }
        }
    }
    TVector<TString> tableDates;
    if (!tableDatesString.empty()) {
        Split(tableDatesString, ",", tableDates);
    }

    TRunConfig::Init(outTablesCount, textsTablesCount, outLinesCount, taskType, dateFieldValue,
        exportQueryTextS == "true", dateFrom, dateTo, tableDates, stage);

    int res;
    TBufferStream jobStateStream(0);
    if (stage == "map" || stage == "map_v2") {
        try {
            switch (taskType) {
                case SEARCH_URL_SAMPLES:
                    res = NYT::RunMapJob<TMapUrls>(1, jobStateStream);
                    break;
                case SEARCH_URL_EVENT_SAMPLES:
                    res = NYT::RunMapJob<TMapEvents>(1, jobStateStream);
                    break;
                case EXCLUDED_URL_SAMPLES:
                    res = NYT::RunMapJob<TMapExcluded>(1, jobStateStream);
                    break;
                case INDEXED_URLS_COUNT:
                    res = NYT::RunMapJob<TMapIndexedCount>(1, jobStateStream);
                    break;
                case INDEXING_EVENTS_COUNT:
                    res = NYT::RunMapJob<TMapIndexingEvents>(1, jobStateStream);
                    break;
                case INDEXED_URL_SAMPLES:
                    res = NYT::RunMapJob<TMapIndexedUrlSamples>(1, jobStateStream);
                    break;
                case INDEXING_EVENT_SAMPLES:
                    res = NYT::RunMapJob<TMapIndexingEventSamples>(1, jobStateStream);
                    break;
                case RECOMMENDED_QUERIES:
                case EXTENDED_RECOMMENDED_QUERIES:
                    res = NYT::RunMapJob<TMapRecommendedQueries>(1, jobStateStream);
                    break;
                case SEARCH_QUERIES_TOP_URLS:
                    res = NYT::RunMapJob<TMapQueriesTopUrls>(1, jobStateStream);
                    break;
                case SEARCH_QUERIES_WEEK:
                    res = NYT::RunMapJob<TMapQueriesWeek>(1, jobStateStream);
                    break;
                case SEARCH_QUERY_STATS:
                    res = NYT::RunMapJob<TMapQueryStats>(1, jobStateStream);
                    break;
                case SEARCH_QUERY_GROUPS:
                    res = NYT::RunMapJob<TMapQueryGroups>(1, jobStateStream);
                    break;
                case LINK_SAMPLES:
                    res = NYT::RunMapJob<TMapLinks>(1, jobStateStream);
                    break;
                case LINK_STATS:
                    res = NYT::RunMapJob<TMapLinkStats>(1, jobStateStream);
                    break;
                case TURBO_URLS:
                    res = NYT::RunMapJob<TMapTurboUrls>(1, jobStateStream);
                    break;
                case HOST_THREATS:
                    res = NYT::RunMapJob<TMapHostThreats>(1, jobStateStream);
                    break;
                case RECOMMENDED_URLS:
                    res = NYT::RunMapJob<TMapRecommendedUrls>(1, jobStateStream);
                    break;
                case DIGEST_MESSAGES:
                    res = NYT::RunMapJob<TMapDigestMessages>(1, jobStateStream);
                    break;
                case MEMORANDUM:
                    res = NYT::RunMapJob<TMapMemorandum>(1, jobStateStream);
                    break;
                case ACHIEVEMENTS:
                    res = NYT::RunMapJob<TMapAchievements>(1, jobStateStream);
                    break;
                case ACHIEVEMENTS_KUUB:
                    res = NYT::RunMapJob<TMapAchievementsKuub>(1, jobStateStream);
                    break;
                case IKS:
                    res = NYT::RunMapJob<TMapIKS>(1, jobStateStream);
                    break;
            }
        } catch (yexception &e) {
            LOG_CRIT("In map %s", e.what());
        }
    } else if (stage == "reduce") {
        try {
            res = NYT::RunReduceJob<TReduceCompressPartition>(outTablesCount, jobStateStream);
        } catch (yexception &e) {
            LOG_CRIT("In reduce %s", e.what());
        }
    } else if (stage == "reduce_v2") {
        try {
            res = NYT::RunReduceJob<TPartitionAndOrderAwareReducer>(outTablesCount, jobStateStream);
        } catch (yexception &e) {
            LOG_CRIT("In reduce %s", e.what());
        }
    } else if (stage == "combine_v2") {
        try {
            res = NYT::RunReduceJob<TPartitionAndOrderAwareReducer>(1, jobStateStream);
        } catch (yexception &e) {
            LOG_CRIT("In combine %s", e.what());
        }
    } else {
        LOG_CRIT("Unknown stage %s", stage.data());
        return 1;
    }

    return res;
}
