#include <util/digest/fnv.h>
#include <util/string/builder.h>

#include <library/cpp/getopt/last_getopt.h>
#include <mapreduce/yt/interface/client.h>

#include "partition.h"
#include "run_config.h"
#include "task_recommended_queries.h"
#include "tasks.h"
#include "util.h"

namespace NWebmaster {

namespace {
const char *FIELD_HOST = "Host";
const char *FIELD_QUERY = "Query";
const char *FIELD_REGION_ID = "RegionId";
const char *FIELD_FORECASTED_BID = "ForecastedBid";
const char *FIELD_FORECASTED_BUDGET = "ForecastedBudget";
const char *FIELD_FORECASTED_CLICKS = "ForecastedClicks";
const char *FIELD_FORECASTED_SHOWS = "ForecastedShows";
const char *FIELD_URL = "Url";
const char *FIELD_PATH = "Path";
const char *FIELD_POSITION = "Position";
const char *FIELD_REL1 = "Rel1";
}

TString firstUrl(const TString &urls) {
    size_t indx = urls.find_first_of('\t');
    if (indx == TString::npos) {
        return urls;
    }
    return urls.substr(0, indx);
}

TString GetUrl(const NYT::TNode &row, bool isExtended) {
    if (isExtended) {
        return encode(row[FIELD_HOST].AsString() + firstUrl(row[FIELD_PATH].AsString()));
    } else {
        return encode(firstUrl(row[FIELD_URL].AsString()));
    }
}

void PackRecommendedQueries(const NYT::TNode &row, TStringBuilder &data, bool isExtended) {
    data << "2017-03-01" << "\t"
         << GetHostId(row[FIELD_HOST].AsString()) << "\t"
         << encode(row[FIELD_QUERY].AsString()) << "\t"
         << PrintNumOrDefault(row, FIELD_REGION_ID, 0) << "\t"
         << PrintNumOrDefault(row, FIELD_FORECASTED_BID, 0) << "\t"
         << PrintNumOrDefault(row, FIELD_FORECASTED_BUDGET, 0) << "\t"
         << PrintNumOrDefault(row, FIELD_FORECASTED_CLICKS, 0) << "\t"
         << PrintNumOrDefault(row, FIELD_FORECASTED_SHOWS, 0) << "\t"
         << GetUrl(row, isExtended) << "\t"
         << PrintNumOrDefault(row, FIELD_POSITION, 0) << "\t"
         << PrintNumOrDefault(row, FIELD_REL1, 0) << "\t"
         << (isExtended ? "1" : "0") << "\n";
}

void TMapRecommendedQueries::Do(TReader *reader, TWriter *writer) {
    const TRunConfig &config = TRunConfig::CInstance();
    const bool isExtended = config.TaskType == NWebmaster::ETaskType::EXTENDED_RECOMMENDED_QUERIES;
    TStringBuilder data;
    for (; reader->IsValid(); reader->Next()) {
        const TString &hostId = GetHostId(reader->GetRow()[FIELD_HOST].AsString());
        size_t partId = (FnvHash<size_t>(hostId)) % config.TablesCount;
        size_t lineId = (FnvHash<size_t>(reader->GetRow()[FIELD_QUERY].AsString())) % config.LinesCount;
        PackRecommendedQueries(reader->GetRow(), data, isExtended);
        if (data.Size() > 0) {
            writer->AddRow(NYT::TNode()
                               (TReduceCompressPartition::F_PARTITION_ID, partId)
                               (TReduceCompressPartition::F_LINE_ID, lineId)
                               (TReduceCompressPartition::F_ROW_ID, reader->GetRowIndex())
                               (TReduceCompressPartition::F_DATA, data)
            );
        }
        data.clear();
    }
}

} //namespace NWebmaster
